/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#include "tbox.h"

/*
** This function searches for a configuration entry and returns
** the first matching entry.
*/

readstat_t examine_cfg_file( const char **, unsigned short, char, const char *, size_t, const char *, size_t, const char * );

readstat_t cfg_lookup( const char **field, unsigned short num_of_fields, char field_delim, const char *fname, const char *env_path_var, const char *key )
{
	char	*cfg_path, *dir_delim;
	readstat_t	readstat;
	size_t	dir_len, fname_len;

	assert( field != (const char **) 0 );
	assert( fname != (const char *) 0 );
	assert( env_path_var != (const char *) 0 );
	assert( key != (const char *) 0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( num_of_fields = %hu, fname = [%s], env_path_var = [%s], key = [%s] )\n", __func__, num_of_fields, fname, env_path_var, key );
	}

	DEBUG_FUNC_START;

	fname_len = strlen( fname );

	cfg_path = getenv( env_path_var );
	if ( cfg_path == (char *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Configuration environment variable [", stderr );
		(void) fputs( env_path_var, stderr );
		(void) fputs( "] was not found.\n", stderr );
		RETURN_INT( Read_err );
	}

	readstat = Read_eof;

	/*
	** Loop through directories listed in environment variable.
	*/
	while ( ( dir_delim = strchr( cfg_path, ':' ) ) != (char *)0 ) {
		assert( dir_delim >= cfg_path );

		dir_len = (size_t)( dir_delim - cfg_path );

		readstat = examine_cfg_file( field, num_of_fields, field_delim, fname, fname_len, cfg_path, dir_len, key );

		if ( readstat == Read_fatal || readstat == Read_ok ) {
			break;
		}

		cfg_path = &dir_delim[ 1 ];
	}

	if ( readstat != Read_fatal && readstat != Read_ok ) {
		/*
		** Check last directory in environment variable path list.
		*/
		readstat = examine_cfg_file( field, num_of_fields, field_delim, fname, fname_len, cfg_path, strlen( cfg_path ), key );
	}


	if ( Debug ) {
		if ( readstat == Read_ok ) {
			(void) fputs( "Found", stderr );
		} else {
			(void) fputs( "Did not find", stderr );
		}
		(void) fputs( " matching configuration record.\n", stderr );

	}

	RETURN_INT( readstat );
}

readstat_t examine_cfg_file( const char **field, unsigned short num_of_fields, char field_delim, const char *fname, size_t fname_len, const char *cfg_dir, size_t cfg_dir_len, const char *key )
{
	char	file_path[PATH_MAX];
	size_t	path_len;
	unsigned long	rec_cnt;
	FILE	*fptr;
	readstat_t	readstat;

	/*
	** +1 is for '/' character between directory and file name.
	*/
	path_len = cfg_dir_len + fname_len + (size_t)1;

	if ( path_len >= sizeof( file_path ) ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "File path [", stderr );
		(void) fwrite( (void *)cfg_dir, sizeof( char ), cfg_dir_len, stderr );
		(void) fputs( "] / [", stderr );
		(void) fwrite( (void *)fname, sizeof( char ), fname_len, stderr );
		(void) fputs( "] exceeds maximum length ", stderr );
		(void) fput_uint( sizeof( file_path ), stderr );
		(void) fputs( ".\n", stderr );

		return Read_err;
	}

	(void) memcpy( (void *)file_path, (void *)cfg_dir, cfg_dir_len );
	file_path[ cfg_dir_len ] = '/';
	(void) memcpy( (void *)&file_path[ cfg_dir_len + 1 ], (void *)fname, fname_len );
	file_path[ path_len ] = (char)0;

	if ( access( file_path, F_OK ) == -1 ) {
		if ( Debug ) {
			(void) fprintf( stderr, "file [%s] does not exist\n", file_path );
		}
		return Read_err;
	}

	fptr = fopen( file_path, "r" );
	if ( fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open configuration file [", stderr );
		(void) fputs( file_path, stderr );
		(void) fputs( "].\n", stderr );
		return Read_err;
	}

	rec_cnt = 0UL;

	for ( ;; ) {
		readstat = cfg_read( field, num_of_fields, &rec_cnt, field_delim, fptr, file_path );
		if ( readstat != Read_ok ) {
			break;
		}

		if ( strcasecmp( field[ 0 ], key ) == 0 ) {
			/*
			** Found matching configuration record.
			*/
			break;
		}
	}

	if ( fclose( fptr ) == EOF ) {
		UNIX_ERROR( "fclose() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not close configuration file [", stderr );
		(void) fputs( file_path, stderr );
		(void) fputs( "].\n", stderr );
		/*
		** Do not fail from this error.
		*/
	}

	return readstat;
}
