/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <regex.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "where.h"
#include "dfile_utility.h"
#include "fixed2dfile.h"

static size_t left_trim( const char *, size_t );
static size_t right_trim( const char *, size_t );

/*
** This function reads the input file and writes the record to an output file.
*/
int process_file( parse_t *parse_tbl, unsigned short parse_cnt, dfile_t *output_dfile, void *filter, size_t input_record_length )
{
	char	*input_record, *field_start;
	size_t	read_cnt, trim_cnt;
	unsigned short	ndx;
	parse_t	*parse;

	assert( parse_tbl != (parse_t *)0 );
	assert( output_dfile != (dfile_t *)0 );
	assert( input_record_length > (size_t)0 );
	/*
	** When filter is not used, pointer will be null.
	*/

	DEBUG_FUNC_START;

	input_record = (char *)malloc( input_record_length );
	if ( input_record == (char *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	parse = parse_tbl;
	for ( ndx = parse_cnt; ndx > (unsigned short)0; --ndx ) {
		parse->field_value = &input_record[ parse->offset ];
		parse->field_length = parse->max_field_length;
		++parse;
	}

	for ( ;; ) {
		read_cnt = fread( (void *)input_record, sizeof( char ), input_record_length, stdin );
		if ( read_cnt != input_record_length ) {
			break;
		}

		parse = parse_tbl;
		for ( ndx = parse_cnt; ndx > (unsigned short)0; --ndx ) {
			switch ( parse->trim_ind ) {
			case 'L':
				field_start = &input_record[ parse->offset ];
				trim_cnt = left_trim( field_start, parse->max_field_length );
				parse->field_value = &field_start[ trim_cnt ];
				parse->field_length = parse->max_field_length - trim_cnt;
				break;
			case 'R':
				field_start = &input_record[ parse->offset ];
				trim_cnt = right_trim( field_start, parse->max_field_length );
				parse->field_length = trim_cnt;
				break;
			}
			++parse;
		}

		if ( output_record( output_dfile, filter ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	if ( !feof( stdin ) ) {
		if ( ferror( stdin ) ) {
			UNIX_ERROR( "fread() failed" );
			RETURN_INT( -1 );
		}
	}

	if ( read_cnt > (size_t)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Found partial record at end of input file.\n", stderr );
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}

static size_t left_trim( const char *ptr, size_t max_len )
{
	size_t	len;

	for ( len = max_len; len > (size_t)0; --len ) {
		if ( *ptr != ' ' ) {
			break;
		}
		++ptr;
	}

	return max_len - len;
}

static size_t right_trim( const char *ptr, size_t len )
{
	const char	*end_ptr;

	assert( len > (size_t)0 );
	end_ptr = &ptr[ len - (size_t)1 ];

	for ( ; len > (size_t)0; --len ) {
		if ( *end_ptr != ' ' ) {
			break;
		}
		--end_ptr;
	}

	return len;
}
