/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <regex.h>
#include <ctype.h>
#include <limits.h>
#include <string.h>
#include "tbox.h"
#include "dfile.h"
#include "fixed2dfile.h"

#define NUM_OF_FIELDS	4

/*
** This function loads the control file.
*/
int load_control_file( parse_t **ret_parse, unsigned short *ret_parse_cnt, const char *filename, size_t input_record_length )
{
	size_t	alloc_size;
	parse_t	*new, *parse;
	unsigned short	parse_cnt;
	FILE	*fptr;
	readstat_t	readstat;
	const char	*field[ NUM_OF_FIELDS ];
	const unsigned short	num_of_fields = NUM_OF_FIELDS;
	unsigned long	line_cnt = 0;
	const char	*trim_ind, field_delim = ':';
	char	*ptr;
	unsigned short	ndx;
	long	offset, field_length;

	assert( ret_parse != (parse_t **)0 );
	assert( ret_parse_cnt != (unsigned short *)0 );
	assert( filename != (const char *)0 );
	assert( input_record_length > (size_t)0 );

	DEBUG_FUNC_START;

	*ret_parse = (parse_t *)0;
	*ret_parse_cnt = (unsigned short)0;

	fptr = fopen( filename, "r" );

	if ( fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open control file [", stderr );
		(void) fputs( filename, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	parse = (parse_t *)0;
	parse_cnt = (unsigned short)0;

	for ( ;; ) {
		readstat = cfg_read( field, num_of_fields, &line_cnt, field_delim, fptr, filename );

		if ( readstat == Read_fatal ) {
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		if ( readstat == Read_eof ) {
			break;
		}

		if ( readstat != Read_ok ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Unexpected return code (", stderr );
			(void) fput_int( readstat, stderr );
			(void) fputs( ") received from read_cfg().\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		for ( ndx = 0; ndx < num_of_fields; ++ndx ) {
			if ( field[ ndx ] == (const char *)0 ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Unexpected null value returned from read_cfg().\n", stderr );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}
		}

		if ( parse_cnt == USHRT_MAX ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Number of entries in control file exceeded maximum ", stderr );
			(void) fput_uint( USHRT_MAX, stderr );
			(void) fputs( ".\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		alloc_size = sizeof( parse_t ) * ( parse_cnt + (unsigned short)1 );

		new = (parse_t *)realloc( (void *)parse, alloc_size );
		if ( new == (parse_t *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		parse = new;
		new = &new[ parse_cnt ];

		(void) memset( (void *)new, 0, sizeof( parse_t ) );

		new->field_name = strdup( field[ 0 ] );
		if ( new->field_name == (char *)0 ) {
			UNIX_ERROR( "strdup() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		offset = strtol( field[ 1 ], &ptr, 10 );
		assert( ptr >= field[ 1 ] );

		if ( ptr == field[ 1 ] || *ptr != (char)0 || offset < 0L ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Start field position offset [", stderr );
			(void) fputs( field[ 1 ], stderr );
			(void) fputs( "] must be integer greater than or equal to zero.\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		new->offset = (size_t)offset;

		field_length = strtol( field[ 2 ], &ptr, 10 );
		assert( ptr >= field[ 2 ] );

		if ( ptr == field[ 2 ] || *ptr != (char)0 || field_length < 1L ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Start field position offset [", stderr );
			(void) fputs( field[ 1 ], stderr );
			(void) fputs( "] must be integer greater than zero.\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		new->max_field_length = (size_t)field_length;

		trim_ind = field[ 3 ];
		if ( strlen( trim_ind ) > (size_t)1 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Trim indicator must be one character (L,R).\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		new->trim_ind = toupper( *trim_ind );

		switch ( new->trim_ind ) {
		case 0:
		case 'L':
		case 'R':
			break;
		default:
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Trim indicator must be (L)eft or (R)ight.\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		if ( ( new->offset + new->max_field_length ) > input_record_length ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Field position (", stderr );
			(void) fput_uint( new->offset, stderr );
			(void) fputs( ") + field length (", stderr );
			(void) fput_uint( new->max_field_length, stderr );
			(void) fputs( ") exceeds input record length (", stderr );
			(void) fput_uint( input_record_length, stderr );
			(void) fputs( ").\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		++parse_cnt;
	}

	(void) fclose( fptr );

	*ret_parse = parse;
	*ret_parse_cnt = parse_cnt;

	RETURN_INT( 0 );
}
