/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <errno.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile_exec.h"


/*
** Record layout:
**	1. GMT Time
**	2. Job Name
**	3. Step Name
**	4. Partition
**	5. Process ID
**	6. Action Code - S)tart, E)nd
**	7. Exit Code
**	8. Signal
*/

int load_completed( completed_t **ret_completed, unsigned long *ret_completed_cnt, const char *log_name, const char *exec_job_name )
{
	FILE	*fptr;
	char	buf[512], msg[512], *rec;
	long	exit_value;
	static char	*job_name, *step_name, *partition;
	static char	*gmt, *pid, *action_code, *exit_code, *end;
	static char	**field_tbl[] = {
		&gmt, &job_name, &step_name, &partition,
		&pid, &action_code, &exit_code
	};
	const unsigned short	field_tbl_cnt = sizeof( field_tbl ) / sizeof( char ** );
	char	***field;
	unsigned short	ndx;
	size_t	len, alloc_size;
	unsigned long	rec_cnt = 0;
	const char	fs = '|';
	completed_t	*completed = 0, *new, *ptr;
	unsigned long	completed_cnt = 0;

	assert( ret_completed != (completed_t **)0 );
	assert( ret_completed_cnt != (unsigned long *)0 );
	assert( exec_job_name != (const char *)0 );

	DEBUG_FUNC_START;

	if ( log_name == (const char *)0 ) {
		*ret_completed = (completed_t *)0;
		*ret_completed_cnt = 0UL;
		RETURN_INT( 0 );
	}

	if ( access( log_name, F_OK ) == -1 && errno == ENOENT ) {
		/*
		** File does not exist.
		*/
		*ret_completed = (completed_t *)0;
		*ret_completed_cnt = 0UL;
		RETURN_INT( 0 );
	}

	fptr = fopen( log_name, "r" );
	if ( fptr == (FILE *)0 ) {
		(void) strcpy( msg, "Failed to open file [" );
		(void) strncat( msg, log_name, sizeof( msg ) - 35 );
		(void) strcat( msg, "]." );
		UNIX_ERROR( msg );
		RETURN_INT( -1 );
	}

	while ( fgets( buf, sizeof( buf ), fptr ) != (char *)0 ) {
		++rec_cnt;

		len = strlen( buf );
		if ( len == (size_t)0 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "File [", stderr );
			(void) fputs( log_name, stderr );
			(void) fputs( "] has zero length record, ", stderr );
			(void) fput_uint( rec_cnt, stderr );
			(void) fputs( ".\n", stderr );
			RETURN_INT( -1 );
		}

		if ( buf[ len - 1 ] != '\n' ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "File [", stderr );
			(void) fputs( log_name, stderr );
			(void) fputs( "] has record, ", stderr );
			(void) fput_uint( rec_cnt, stderr );
			(void) fputs( ", that is too long.\n", stderr );
			RETURN_INT( -1 );
		}

		--len;
		buf[ len ] = (char)0;

		rec = strdup( buf );
		if ( rec == (char *)0 ) {
			UNIX_ERROR( "strdup() failed" );
			RETURN_INT( -1 );
		}

		end = &rec[ -1 ];
		field = field_tbl;

		for ( ndx = field_tbl_cnt; ndx > (unsigned short)0; --ndx ) {
			**field = &end[ 1 ];
			end = strchr( **field, fs );
			if ( end == (char *)0 ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "File [", stderr );
				(void) fputs( log_name, stderr );
				(void) fputs( "] has record, ", stderr );
				(void) fput_uint( rec_cnt, stderr );
				(void) fputs( ", with incorrect number of fields.\n", stderr );
				RETURN_INT( -1 );
			}
			*end = (char)0;
			++field;
		}

		if ( strcmp( action_code, "S" ) == 0 ) {
			/*
			** Skip start records.
			*/
			continue;
		}

		if ( strcmp( action_code, "E" ) != 0 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "File [", stderr );
			(void) fputs( log_name, stderr );
			(void) fputs( "] has record, ", stderr );
			(void) fput_uint( rec_cnt, stderr );
			(void) fputs( ", with action code not S or E.\n", stderr );
			RETURN_INT( -1 );
		}

		if ( strcmp( job_name, exec_job_name ) != 0 ) {
			/*
			** Record not associated with this job.
			*/
			continue;
		}

		exit_value = strtol( exit_code, &end, 10 );

		if ( exit_code == end || *end != (char)0 || exit_value < 0 || exit_value > UCHAR_MAX ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "File [", stderr );
			(void) fputs( log_name, stderr );
			(void) fputs( "] has record, ", stderr );
			(void) fput_uint( rec_cnt, stderr );
			(void) fputs( ", with exit code not between 0 and 255.\n", stderr );
			RETURN_INT( -1 );
		}

		alloc_size = sizeof( completed_t ) * ( completed_cnt + 1UL );
		new = (completed_t *)realloc( completed, alloc_size );
		if ( new == (completed_t *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			RETURN_INT( -1 );
		}

		ptr = &new[ completed_cnt ];

		ptr->step_name = step_name;

		if ( *partition == (char)0 ) {
			ptr->partition = (char *)0;
		} else {
			ptr->partition = partition;
#if 0
fprintf( stderr, "step_name [%s], partition [%s], exit_code [%s]\n", step_name, partition, exit_code );
#endif
		}

		ptr->exit_code = atoi( exit_code );
		completed = new;
		++completed_cnt;
	}

	if ( ferror( fptr ) || !feof( fptr ) ) {
		UNIX_ERROR( "fgets() failed" );
		RETURN_INT( -1 );
	}

	(void) fclose( fptr );

	qsort( (void *)completed, (size_t)completed_cnt, sizeof( completed_t ), completed_cmp );

	*ret_completed = completed;
	*ret_completed_cnt = completed_cnt;

	RETURN_INT( 0 );
}
