/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile_exec.h"

/*
** This function assigns information from control structure.
*/
int assign_job( const char **job_name, step_t **step, unsigned short *step_cnt, char ***env_var, unsigned short *env_var_cnt, const char *control_file )
{
	int	ret;
	unsigned short	tuple_cnt;
	sexpr_t	*level0, *field;
	char	*str;

	assert( job_name != (const char **)0 );
	assert( step != (step_t **)0 );
	assert( step_cnt != (unsigned short *)0 );
	assert( env_var != (char ***)0 );
	assert( env_var_cnt != (unsigned short *)0 );

	DEBUG_FUNC_START;

	*job_name = (const char *)0;
	*step = (step_t *)0;
	*step_cnt = (unsigned short)0;
	*env_var = (char **)0;
	*env_var_cnt = (unsigned short)0;


	if ( control_file == (const char *)0 ) {
		/*
		** no control file
		*/
		RETURN_INT( 0 );
	}

	ret = sexpr_load( &level0, control_file );
	if ( ret != 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "failed to load s-expression from file [", stderr );
		(void) fputs( control_file, stderr );
		(void) fputs( "]: [", stderr );
		(void) fputs( sexpr_error( ret ), stderr );
		(void) fputs( "]\n", stderr );

		RETURN_INT( -1 );
	}

	tuple_cnt = (unsigned short)0;

	/*
	** Loop through each tuple at depth 1.
	*/
	while ( level0 != (sexpr_t *)0 ) {
		++tuple_cnt;
		if ( SEXPR_CAR_TYPE( level0 ) == string_sexpr ) {
			if( *job_name != (const char *)0 ) {
				control_syntax_error( "expected only one JOB", __FILE__, __LINE__, tuple_cnt );
				RETURN_INT( -1 );
			}
			str = SEXPR_CAR_STRING( level0 );

			if ( strcasecmp( str, "job" ) != 0 ) {
				control_syntax_error( "expected string to be JOB", __FILE__, __LINE__, tuple_cnt );
				RETURN_INT( -1 );
			}

			level0 = SEXPR_CDR_LIST( level0 );

			ret = assign_string( (char **)job_name, level0, "job", tuple_cnt );
			if ( ret == -1 ) {
				RETURN_INT( -1 );
			}

			level0 = SEXPR_CDR_LIST( level0 );

			continue;
		}

		field = SEXPR_CAR_LIST( level0 );
		if ( field == (sexpr_t *)0 ) {
			CONTROL_SYNTAX_ERROR( "expected a string", tuple_cnt );
			RETURN_INT( -1 );
		}

		if ( SEXPR_CAR_TYPE( field ) != string_sexpr ) {
			CONTROL_SYNTAX_ERROR( "expected a string", tuple_cnt );
			RETURN_INT( -1 );
		}

		str = SEXPR_CAR_STRING( field );

		assert( SEXPR_CDR_TYPE( field ) == list_sexpr );

		if ( SEXPR_CDR_LIST( field ) == (sexpr_t *)0 ) {
			CONTROL_SYNTAX_ERROR( "null list", tuple_cnt );
			RETURN_INT( -1 );
		}


		if ( strcasecmp( str, "step" ) == 0 ) {

			ret = assign_step( step, step_cnt, SEXPR_CDR_LIST( field ), tuple_cnt );
			if ( ret != 0 ) {
				RETURN_INT( ret );
			}
		} else {
			if ( strcasecmp( str, "setenv" ) == 0 ) {

				ret = assign_setenv( env_var, env_var_cnt, SEXPR_CDR_LIST( field ), tuple_cnt );
				if ( ret != 0 ) {
					RETURN_INT( ret );
				}
			} else {
				CONTROL_SYNTAX_ERROR( "unknown section", tuple_cnt );
				RETURN_INT( -1 );
			}
		}

		assert( SEXPR_CDR_TYPE( level0 ) == list_sexpr );

		level0 = SEXPR_CDR_LIST( level0 );
	}

	RETURN_INT( 0 );
}
