/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "where.h"
#include "_where.h"


/*
** This function is part of the interpreter for executing operators.
*/

int _where_condition( where_result_t *ret_result, condition_t *condition )
{
	const char	*op_desc;
	int	ret, cmp_result;
	where_result_t	result;

	assert( ret_result != (where_result_t *)0 );
	assert( condition != (condition_t *)0 );

	DEBUG_FUNC_START;

	op_desc = (const char *)0;

	/*
	** Perform expected operation.
	*/
	switch ( COND_OP( condition ) ) {
	case And:
		ret = _where_multi_condition( &result, COND_MULTI_COND( condition ), And );
		op_desc = "AND";
		break;

	case Or:
		ret = _where_multi_condition( &result, COND_MULTI_COND( condition ), Or );
		op_desc = "OR";
		break;

	case Equal:
	case Greater:
	case Greater_or_equal:
	case Less:
	case Less_or_equal:
		ret = _where_compare_condition( &cmp_result, &COND_MULTI_DATUM( condition ) );
		break;

	case Not:
		ret = _where_condition( &result, COND_MULTI_COND( condition )->condition );
		break;

	case In:
		ret = _where_in_condition( &result, COND_IN( condition ) );
		break;

	case Like:
		ret = _where_like_condition( &result, COND_LIKE( condition ) );
		break;

	default:
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Programming error--unknown condition operator [", stderr );
		(void) fput_int( (int)COND_OP( condition ), stderr );
		(void) fputs( "] encountered.\n", stderr );
		abort();
	}

	if ( ret != 0 ) {
		/*
		** An error occurred.
		*/
		RETURN_INT( ret );
	}

	/*
	** Interpret results of operation.
	** Set variable result to zero for true and negative one for false.
	*/
	switch ( COND_OP( condition ) ) {
	case In:
		op_desc = "IN";
		break;
	case Like:
		op_desc = "LIKE";
		break;
	case Equal:
		result = ( cmp_result == 0 ) ? Where_result_true : Where_result_false;
		op_desc = "EQUAL";
		break;
	case Greater:
		result = ( cmp_result > 0 ) ? Where_result_true : Where_result_false;
		op_desc = "GREATER";
		break;
	case Greater_or_equal:
		result = ( cmp_result >= 0 ) ? Where_result_true : Where_result_false;
		op_desc = "GREATER OR EQUAL";
		break;
	case Less:
		result = ( cmp_result < 0 ) ? Where_result_true : Where_result_false;
		op_desc = "LESS";
		break;
	case Less_or_equal:
		result = ( cmp_result <= 0 ) ? Where_result_true : Where_result_false;
		op_desc = "LESS OR EQUAL";
		break;

	case Not:
		result = ( result == Where_result_true ) ? Where_result_false : Where_result_true;
		op_desc = "NOT";
		break;
	}

	assert( result == Where_result_true || result == Where_result_false );
	*ret_result = result;

	if ( Debug ) {
		(void) fputs( "Performed ", stderr );
		(void) fputs( op_desc, stderr );
		(void) fputs( " operation.\n", stderr );
		(void) fputs( "Condition was ", stderr );
		if ( *ret_result == Where_result_true ) {
			(void) fputs( "true.\n", stderr );
		} else {
			(void) fputs( "false.\n", stderr );
		}
	}

	RETURN_INT( WHERE_NOERR );
}
