/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "where.h"
#include "dfile_utility.h"
#include "dfile_unique.h"


/*
** This function reads the input file and writes the record to an output file.
*/
int process_file( dfile_t *output_dfile, dfile_t *input_dfile, dfile_bind_t **key_bind_tbl, unsigned short key_bind_tbl_cnt, unsigned short *field_ndx_tbl, void *input_filter, void *output_filter )
{
	static int	initial_record = 1;
	unsigned long	record_cnt;
	unsigned short	input_bind_cnt, output_bind_cnt;
	where_result_t	where_result;
	char	err_msg[ 256 ];
	field_t	*rec, **rec_key;
	dfile_bind_t	*input_bind, *output_bind;
	size_t	rec_size;

	assert( output_dfile != (dfile_t *)0 );
	assert( input_dfile != (dfile_t *)0 );
	assert( key_bind_tbl != (dfile_bind_t **)0 );
	assert( field_ndx_tbl != (unsigned short *)0 );
	/*
	** When filters are not used, pointers will be null.
	*/

	DEBUG_FUNC_START;

	record_cnt = 0UL;
	input_bind = input_dfile->bind;
	input_bind_cnt = input_dfile->bind_cnt;
	output_bind = output_dfile->bind;
	output_bind_cnt = output_dfile->bind_cnt;
	rec = (field_t *)0;
	rec_size = (size_t)0;

	while ( dfile_read( input_dfile ) == 0 ) {
		if ( input_filter != (void *)0 ) {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), input_filter ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "input where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				RETURN_INT( -1 );
			}
			if ( where_result == Where_result_false ) {
				/*
				** Skip record.
				*/
				continue;
			}
		}

		if ( initial_record ) {
			if ( copy_record( &rec, &rec_size, (const dfile_bind_t *)input_bind, input_bind_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}

			if ( alloc_record_key( &rec_key, key_bind_tbl_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}

			assign_field_key( rec_key, rec, key_bind_tbl, key_bind_tbl_cnt, input_bind );

			/*
			** Set initial control break to false.
			*/
			initial_record = 0;

			++record_cnt;
			continue;
		}

		if ( compare_field_bind( (const field_t **)rec_key, (const dfile_bind_t **)key_bind_tbl, key_bind_tbl_cnt ) != 0 ) {
			copy_record_to_bind( output_bind, rec, field_ndx_tbl, output_bind_cnt );

			if ( output_record( output_dfile, output_filter ) == -1 ) {
				RETURN_INT( -1 );
			}
		}

#if 0
		/*
		** Freeing first value frees all values.
		** copy_record() has been changed to not allocate
		** each time. These free() calls should not be
		** needed.
		*/
		free( (void *)rec->value );
		free( (void *)rec );
#endif

		if ( copy_record( &rec, &rec_size, (const dfile_bind_t *)input_bind, input_bind_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		++record_cnt;
	}

	if ( record_cnt > 0UL ) {
		copy_record_to_bind( output_bind, rec, field_ndx_tbl, output_bind_cnt );

		if ( output_record( output_dfile, output_filter ) == -1 ) {
			RETURN_INT( -1 );
		}

		assert( rec != (field_t *)0 );
		assert( rec->value != (char *)0 );

		free( (void *)rec->value );
		free( (void *)rec );
	}

	if ( input_dfile->error != Dfile_all_data_processed ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to read all data.\n", stderr );
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
