/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <search.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "rbtree.h"
#include "sexpr.h"
#include "dfile_exec.h"


void rm_proc_from_list( proc_t **, proc_t * );

/*
** This function waits for a proc to complete, interprets proc's exit code
** and performs post proc cleanup.
*/
int wait_proc( unsigned short *procs_remaining_cnt, void **proc_tree, int *exit_code, proc_t **proc_list, int log_fd, const char *job_name, const char *step_name )
{
	pid_t	pid;
	int	status, *sig_ptr;
	char	msg[300], pid_str[24], int_str[24];
	size_t	len;
	proc_t	key, **ptr, *proc;

	assert( procs_remaining_cnt != (unsigned short *)0 );
	assert( proc_tree != (void **)0 );
	assert( proc_list != (proc_t **)0 );
	assert( exit_code != (int *)0 );

	DEBUG_FUNC_START;

	/*
	** Wait for an application slice (proc) to complete.
	*/
	pid = wait( &status );

	if ( pid < (pid_t) 0 ) {
		UNIX_ERROR( "wait() failed" );
		RETURN_INT( -1 );
	}

	if ( snprintf( pid_str, sizeof( pid_str ), "%d", pid ) <= 0 ) {
		UNIX_ERROR( "snprintf() failed to convert PID to ASCII" );
		RETURN_INT( -1 );
	}

	/*
	** Find which proc completed using Unix process ID.
	*/
	key.pid = pid;
	ptr = (proc_t **)rbtfind( (void *)&key, proc_tree, proc_cmp );
	if ( ptr == (proc_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to find PID ", stderr );
		(void) fputs( pid_str, stderr );
		(void) fputs( " in proc list.\n", stderr );
		RETURN_INT( 0 );
	}

	proc = *ptr;

	proc->signal_value = 0;

	if ( WIFSIGNALED( status ) != 0 ) {
		/*
		** Child process death caused by signal.
		*/
		proc->signal_value = WTERMSIG( status );
		(void) strcpy( msg, get_ctime() );
		(void) strcat( msg, " PROCESS " );
		(void) strcat( msg, pid_str );
		(void) strcat( msg, " TERMINATED FROM RECEIVING SIGNAL " );
		if ( snprintf( int_str, sizeof( int_str ), "%d", WTERMSIG( status ) ) <= 0 ) {
			UNIX_ERROR( "snprintf() failed to convert signal value to ASCII" );
			RETURN_INT( -1 );
		}
		(void) strcat( msg, int_str );
		(void) strcat( msg, ".\n" );
		len = strlen( msg );
		if ( write( proc->stderr_fd, (void *)msg, len ) != len ) {
			UNIX_ERROR( "write() failed on stderr file" );
		}
	}

	/*
	** Default status to failure.
	*/
	proc->status = 1;

	if ( WIFEXITED( status ) != 0 ) {
		/*
		** Process terminated normally.
		*/
		proc->status = WEXITSTATUS( status );

		(void) strcpy( msg, get_ctime() );
		(void) strcat( msg, " PROCESS " );
		(void) strcat( msg, pid_str );
		if ( WEXITSTATUS( status ) == 0 ) {
			(void) strcat( msg, " COMPLETED" );
		} else {
			(void) strcat( msg, " FAILED WITH RETURN CODE " );
			if ( snprintf( int_str, sizeof( int_str ), "%d", WEXITSTATUS( status ) ) <= 0 ) {
				UNIX_ERROR( "snprintf() failed to convert return value to ASCII" );
				RETURN_INT( -1 );
			}
			(void) strcat( msg, int_str );
		}
		(void) strcat( msg, ".\n" );
		len = strlen( msg );

		if ( proc->stderr_fd != 2 ) {
			/*
			** Completed proc did not inherit parent's stderr.
			*/
			if ( write( proc->stderr_fd, (void *)msg, len ) != len ) {
				UNIX_ERROR( "write() failed on stderr file" );
			}
		}
	}

	/*
	** Close parent's copy of stdout and stderr file descriptors used
	** by completed proc.
	*/
	if ( proc->stdout_fd != 1 ) {
		/*
		** Completed proc did not inherit parent's stdout.
		*/
		if ( close( proc->stdout_fd ) == -1 ) {
			UNIX_ERROR( "close() failed on stdout file" );
		}
	}

	if ( proc->stderr_fd != 2 ) {
		/*
		** Completed proc did not inherit parent's stderr.
		*/
		if ( close( proc->stderr_fd ) == -1 ) {
			UNIX_ERROR( "close() failed on stderr file" );
		}
	}

	if ( proc->status == 0 ) {
		/*
		** Job completed successfully.
		*/
		--*procs_remaining_cnt;

		/*
		** Remove proc from run list.
		*/
		rm_proc_from_list( proc_list, proc );
	} else {
		/*
		** Job did not complete successfully.
		*/
		--*procs_remaining_cnt;

		/*
		** At least one process did not complete successfully.
		** Eventually exit with a failure return code.
		*/
		*exit_code = proc->status;

		/*
		** Remove proc from run list.
		*/
		rm_proc_from_list( proc_list, proc );
	}

	if ( proc->signal_value == 0 ) {
		sig_ptr = (int *)0;
	} else {
		sig_ptr = &proc->signal_value;
	}

	if ( write_log_record( log_fd, job_name, step_name, proc->partition, pid, 'E', &proc->status, sig_ptr ) == -1 ) {
		RETURN_INT( -1 );
	}

	/*
	** Remove process ID from tree.
	*/
	ptr = (proc_t **)rbtdelete( (void *)&key, proc_tree, proc_cmp );
	if ( ptr == (proc_t **)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to delete PID ", stderr );
		(void) fputs( pid_str, stderr );
		(void) fputs( " from proc tree.\n", stderr );
	}

#if 0
	(void) fputs( get_ctime(), stderr );
	(void) fputs( " Process ", stderr );
	(void) fputs( pid_str, stderr );
	(void) fputs( " completed (", stderr );
	(void) fprintf( stderr, "%d", proc->status );
	(void) fputs( ").\n", stderr );
#endif
	

	RETURN_INT( 0 );
}

void rm_proc_from_list( proc_t **proc_list, proc_t *proc )
{
	if ( *proc_list == proc ) {
		*proc_list = proc->next;
		if ( *proc_list != (proc_t *)0 ) {
			( *proc_list )->prev = (proc_t *)0;
		}
	} else {
		if ( proc->next != (proc_t *)0 ) {
			proc->next->prev = proc->prev;
		}
		proc->prev->next = proc->next;
	}
}
