/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "where.h"
#include "_where.h"


typedef struct {
	char	function_name[8];
	operation_type_t	operation;
} func_op_t;

/*
** This function contributes to building syntax tree for binary expressions.
*/

int _where_compare_syntax( condition_t *condition, const char *function_name, sexpr_t *sexpr, dfile_t *dfile )
{
	int	ret;
	unsigned short	ndx;

	static const func_op_t	func_op[] = {
		{ "=", Equal },
		{ ">", Greater },
		{ ">=", Greater_or_equal },
		{ "<", Less },
		{ "<=", Less_or_equal }
	};

	const unsigned short	func_op_cnt = sizeof( func_op ) / sizeof( func_op_t );

	const func_op_t	*func_op_ptr;

	assert( condition != (condition_t *)0 );
	assert( function_name != (const char *)0 );
	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	if ( sexpr == (sexpr_t *)0 || SEXPR_CAR_TYPE( sexpr ) != string_sexpr ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected constant or variable value.\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	if ( strcasecmp( function_name, "in" ) == 0 ) {
		COND_OP( condition ) = In;

		ret = _where_in_syntax( COND_IN( condition ), sexpr, dfile );

		RETURN_INT( ret );
	} else {
		if ( strcasecmp( function_name, "like" ) == 0 ) {
			COND_OP( condition ) = Like;

			ret = _where_like_syntax( COND_LIKE( condition ), sexpr, dfile );

			RETURN_INT( ret );
		}
	}

	func_op_ptr = func_op;

	for ( ndx = (unsigned short)0; ndx < func_op_cnt; ++ndx ) {
		if ( strcmp( function_name, func_op_ptr->function_name ) == 0 ) {
			COND_OP( condition ) = func_op_ptr->operation;
			break;
		}
		++func_op_ptr;
	}

	if ( ndx >= func_op_cnt ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Expected constant or variable value.\n", stderr );
		RETURN_INT( WHERE_SYNTAX );
	}

	ret = _where_multi_datum_syntax( &COND_MULTI_DATUM( condition ), sexpr, dfile );

	if ( ret != WHERE_NOERR ) {
		RETURN_INT( ret );
	}

	switch ( COND_OP( condition ) ) {
	case Equal:
	case Greater:
	case Greater_or_equal:
	case Less:
	case Less_or_equal:
		if ( COND_MULTI_DATUM( condition ).datum_cnt != 2UL ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Binary operator requires two arguments.\n", stderr );
			RETURN_INT( WHERE_SYNTAX );
		}
		break;
	}

	RETURN_INT( WHERE_NOERR );
}
