/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"


int substitute_str( char *buf, size_t max_buf_size, const char *search, const char *replace )
{
	char	*buf_ptr, *tmp_str;
	size_t	search_len, replace_len;

	assert( buf != (char *) 0 );
	assert( search != (char *) 0 );
	assert( replace != (char *) 0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( buf [%s], max_buf_size [%u], search [%s], replace [%s] )\n", __func__, buf, max_buf_size, search, replace );
	}

	DEBUG_FUNC_START;

	buf_ptr = buf;
	search_len = strlen( search );
	replace_len = strlen( replace );

	if ( search_len == (size_t) 0 ) {
		/*
		** Searching for nothing.
		*/
		if ( Debug ) {
			(void) fputs( "Searching for nothing; buf remains unchanged.\n", stderr );
		}

		RETURN_INT( 0 );
	}

	if ( ( tmp_str = malloc( max_buf_size ) ) == (char *) 0 ) {
		(void) fputs( __FILE__, stderr );
		(void) fprintf( stderr, "(%d)", __LINE__ );
		(void) fputs( ": Could not allocate needed memory.\n", stderr );

		RETURN_INT( -1 );
	}

	while ( ( buf_ptr = strstr( buf_ptr, search ) ) != (char *) 0 ) {

		if ( ( strlen( buf ) - search_len + replace_len ) > ( max_buf_size - 1 ) ) {
			(void) fputs( __FILE__, stderr );
			(void) fprintf( stderr, "(%d)", __LINE__ );
			(void) fputs( ": Substitution of [", stderr );
			(void) fputs( search, stderr );
			(void) fputs( "] with [", stderr );
			(void) fputs( replace, stderr );
			(void) fputs( "] would exceed maximum string size ", stderr );
			(void) fprintf( stderr, "%u", max_buf_size );
			(void) fputs( ".\n", stderr );

			free( tmp_str );

			RETURN_INT( -1 );
		}

		/*
		** Save portion of string that is after the matched fragment.
		*/
		(void) strcpy( tmp_str, &buf_ptr[ search_len ] );
		/*
		** Replace matched string with replacement value.
		*/
		(void) strcpy( buf_ptr, replace );
		/*
		** Append fragement that was originally after matched string
		** to the replaced string.
		*/
		buf_ptr = &buf_ptr[ replace_len ];
		(void) strcpy( buf_ptr, tmp_str );
	}

	free( tmp_str );

	if ( Debug ) {
		(void) fputs( "After substitution buf contains [", stderr );
		(void) fputs( buf, stderr );
		(void) fputs( "].\n", stderr );
	}

	RETURN_INT( 0 );
}

#ifdef MT_substitute_str
/*
** This function is used to regression test substitute_str().
** The following command is used to compile:
** x=substitute_str; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/

void main( void )
{
	static const char	blank_line[] = ">>>\n";
	static const char	test_func[] = "substitute_str";
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	incorrectly_successful[] = ">>>\n>>> %s() was incorrectly successful.\n";
	static const char	correctly_unsuccessful[] = ">>>\n>>> %s() was correctly unsuccessful.\n";
	static const char	correctly_successful[] = ">>>\n>>> %s() was correctly successful.\n";
	static const char	incorrectly_unsuccessful[] = ">>>\n>>> %s() was incorrectly unsuccessful.\n";

	char	buf[10];

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );


	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Test failure for buffer too small.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( buf, "ABCDE" );
	if ( substitute_str( buf, sizeof( buf ), "CD", "12345678" ) == -1 ) {	
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Test substituting first character.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( buf, "ABCDE" );
	if ( substitute_str( buf, sizeof( buf ), "A", "123" ) == 0 && strcmp( buf, "123BCDE" ) == 0 ) {	
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Test substituting last character.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( buf, "ABCDE" );
	if ( substitute_str( buf, sizeof( buf ), "E", "123" ) == 0 && strcmp( buf, "ABCD123" ) == 0 ) {	
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Test substituting multiple times.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( buf, "ABBBE" );
	if ( substitute_str( buf, sizeof( buf ), "B", "XX" ) == 0 && strcmp( buf, "AXXXXXXE" ) == 0 ) {	
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #5\n", stderr );
	(void) fputs( ">>> Test replace string null.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( buf, "ABBBE" );
	if ( substitute_str( buf, sizeof( buf ), "B", "" ) == 0 && strcmp( buf, "AE" ) == 0 ) {	
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
