/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"


static int	validate_record( char *, size_t );
static int	copy_record( char ***, unsigned long, char * );

/*
** This function loads a text file into memory.
*/
int load_text_file( char ***text_tbl, unsigned long *text_tbl_cnt, const char *fname )
{
	FILE	*fptr;
	char	buf[BUFSIZ];

	assert( text_tbl != (char ***)0 );
	assert( text_tbl_cnt != (unsigned long *)0 );
	assert( fname != (const char *)0 );

	DEBUG_FUNC_START;

	fptr = fopen( fname, "r" );
	if ( fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open text file [", stderr );
		(void) fputs( fname, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	while ( fgets( buf, sizeof( buf ), fptr ) != (char *)0 ) {
		if ( validate_record( buf, sizeof( buf ) ) == -1 ) {
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		if ( copy_record( text_tbl, *text_tbl_cnt, buf ) == -1 ) {
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		++*text_tbl_cnt;
	}

	if ( ferror( fptr ) ) {
		UNIX_ERROR( "fgets() failed" );
		(void) fclose( fptr );
		RETURN_INT( -1 );
	}

	(void) fclose( fptr );

	RETURN_INT( 0 );
}

static int validate_record( char *buf, size_t buf_size )
{
	size_t	len;

	len = strlen( buf );

	if ( len == (size_t)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "fgets() returned a zero length string.\n", stderr );
		return -1;
	}

	/*
	** About to truncate newline character.
	*/
	--len;

	if ( buf[ len ] != '\n' ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "text record was greater than ", stderr );
		(void) fput_uint( buf_size, stderr );
		(void) fputs( " characters.\n", stderr );
		return -1;
	}

	/*
	** Truncate newline character.
	*/
	buf[ len ] = (char)0;

	return 0;
}

static int copy_record( char ***text, unsigned long text_cnt, char *buf )
{
	char	**new;
	size_t	alloc_size;

	alloc_size = ( (size_t)text_cnt + (size_t)1 ) * sizeof( char * );
	new = (char **)realloc( (void *)*text, alloc_size );

	if ( new == (char **)0 ) {
		UNIX_ERROR( "realloc() failed" );
		return -1;
	}

	new[ text_cnt ] = (char *)strdup( buf );
	if ( new[ text_cnt ] == (char *)0 ) {
		UNIX_ERROR( "strdup() failed" );
		return -1;
	}

	*text = new;

	return 0;
}
