/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "tbox.h"

int file_basename( const char **base, const char ***suffix, unsigned short *suffix_cnt, const char *path )
{
	const char	*fname;
	char	*ptr;
	size_t	alloc_size;

	assert( base != (const char **)0 );
	assert( path != (const char *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%s] )\n", __func__, path );
	}

	DEBUG_FUNC_START;

	*base = (const char *)0;

	fname = (const char *)strrchr( path, '/' );
	if ( fname == (const char *)0 ) {
		/*
		** No directory path.
		*/
		fname = path;
	} else {
		/*
		** Advance past last '/'.
		*/
		++fname;
	}

	if ( suffix == (const char ***)0 || suffix_cnt == (unsigned short *)0 ) {
		/*
		** Don't process suffix.
		*/
		*base = fname;

		RETURN_INT( 0 );
	}

	*suffix = (const char **)0;
	*suffix_cnt = (unsigned short)0;

	ptr = strdup( fname );
	if ( ptr == (char *)0 ) {
		UNIX_ERROR( "strdup() failed" );
		RETURN_INT( -1 );
	}

	*base = (const char *)ptr;

	while ( ( ptr = strchr( ptr, '.' ) ) != (char *)0 ) {
		alloc_size = ( (size_t)*suffix_cnt + (size_t)1 ) * sizeof( const char * );
		*suffix = (const char **)realloc( *suffix, alloc_size );
		if ( *suffix == (const char **)0 ) {
			UNIX_ERROR( "realloc() failed" );
			RETURN_INT( -1 );
		}

		/*
		** Replace dot (.) with null character.
		*/
		*ptr = (char)0;

		/*
		** Advance pointer to character after dot (.).
		*/
		++ptr;

		( *suffix )[ *suffix_cnt ] = (const char *)ptr;
		++*suffix_cnt;
	}

	if ( Debug ) {
		unsigned short	ndx;
		(void) fprintf( stderr, "base [%s]\n", *base );
		for ( ndx = (unsigned short)0; ndx < *suffix_cnt; ++ndx ) {
			(void) fprintf( stderr, "suffix %hu. [%s]\n", ndx+1, ( *suffix )[ ndx ] );
		}
	}

	RETURN_INT( 0 );
}

#ifdef MT_file_basename

/*
** This function is used to regression test file_basename().
** The following command is used to compile:
**   x=file_basename; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "file_basename";
	static const char	correctly_successful[] = ">>>\n>>> %s() was correctly successful.\n";
	static const char	incorrectly_unsuccessful[] = ">>>\n>>> %s() was incorrectly unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";

	const char	*base;
	const char	**suffix;
	unsigned short	suffix_cnt;

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Check successful call with directory supplied as argument.\n", stderr );
	(void) fputs( blank_line, stderr );

	file_basename( &base, (const char ***)0, (unsigned short *)0, "abc" );
	if ( strcmp( base, "abc" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}
	(void) fputs( blank_line, stderr );

	file_basename( &base, (const char ***)0, (unsigned short *)0, "abc.txt" );
	if ( strcmp( base, "abc.txt" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	file_basename( &base, &suffix, &suffix_cnt, "abc" );
	if ( strcmp( base, "abc" ) == 0 && suffix == (const char **)0 && suffix_cnt == (unsigned short)0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	file_basename( &base, &suffix, &suffix_cnt, "/abc" );
	if ( strcmp( base, "abc" ) == 0 && suffix == (const char **)0 && suffix_cnt == (unsigned short)0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	file_basename( &base, &suffix, &suffix_cnt, "/usr/local/abc" );
	if ( strcmp( base, "abc" ) == 0 && suffix == (const char **)0 && suffix_cnt == (unsigned short)0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	file_basename( &base, &suffix, &suffix_cnt, "/usr/local/abc.txt" );
	if ( strcmp( base, "abc" ) == 0 && suffix_cnt == (unsigned short)1 && strcmp( *suffix, "txt" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	file_basename( &base, &suffix, &suffix_cnt, "/usr/local/abc.txt.gz" );
	if ( strcmp( base, "abc" ) == 0 && suffix_cnt == (unsigned short)2 && strcmp( *suffix, "txt" ) == 0 && strcmp( suffix[ 1 ], "gz" ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
