/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"

/*
** The hashing algorithm used in this module is an implementation of
** Knuth's 6.4 Algorithm D from 'The Art of Computer Programming,
** Vol. III, Second Edition'. Algorithm D uses a double hashing
** technique for open addressing. It also sports Brent's Variation
** described by Knuth. This enhancement reduces the number of collisions
** during searches.
*/

#define NEVER_BEEN_USED_BUCKET( x )	( x == empty_bucket )
#define UNOCCUPIED_BUCKET( x )	( x == empty_bucket || x == deleted_bucket )
#define OCCUPIED_BUCKET( x )	( !UNOCCUPIED_BUCKET( x ) )
#define	CALC_DELTA_NDX( x, y )	( 1UL + ( x % ( y - 2 ) ) );

static int optimize_collisions( unsigned long *, const void **, unsigned long *, unsigned long, unsigned long, unsigned long ( * )( const void * ), unsigned long );
static unsigned long apply_delta_ndx( unsigned long, unsigned long, unsigned long );
static int greater_collision( const void **, unsigned long, unsigned long (*)( const void * ), unsigned long *, unsigned long, unsigned long );
static void *expand_hash( void *, unsigned long ( * )( const void * ), int ( * )( const void *, const void * ) );
static int copy_hash( void *, const void *, unsigned long ( * )( const void * ), int ( * )( const void *, const void * ) );

static void * const empty_bucket = 0;
static void * const deleted_bucket = (void *)~0;
static dherrno_t	dherrno = Dh_ok;
static const unsigned short	expansion_factor = 3;

typedef struct {
	unsigned long	used_buckets;
	unsigned long	total_buckets;
	unsigned long	allowed_collisions;
} htable_header_t;

/*
** This function creates hash table.
*/
void *dhcreate( unsigned long buckets, unsigned long allowed_collisions )
{
	htable_header_t	*header;
	size_t	alloc_size;
	void	*ptr;
	size_t	bucket_arena_size;
	int	prev_prime;

	DEBUG_FUNC_START;

	/*
	** Initial number of hash buckets, variable buckets, is treated as a
	** starting point for finding a prime number that is numerically
	** friendly for double hashing.
	**
	** Make sure buckets is an odd number. With the exception of 2,
	** only odd numbers are prime.
	*/
	buckets |= 1;

	/*
	** Set prev_prime to false.
	*/
	prev_prime = 0;

	/*
	** Find twin prime numbers, as suggested by Knuth.
	*/
	for ( ;; ) {
		if ( isprime( buckets, (unsigned short)5 ) ) {
			if ( prev_prime ) {
				break;
			}
			/*
			** Set prev_prime to true.
			*/
			prev_prime = 1;
		} else {
			/*
			** Set prev_prime to false.
			*/
			prev_prime = 0;
		}
		/*
		** Skip even numbers.
		*/
		buckets += 2UL;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "buckets is prime number %lu\n", buckets );
	}

	bucket_arena_size = buckets * sizeof(void *);
	alloc_size = sizeof( htable_header_t ) + bucket_arena_size;

	ptr = malloc( alloc_size );

	if ( ptr == (void *)0 ) {
		dherrno = Dh_alloc;
		RETURN_POINTER( (void *)0 );
	}

	header = (htable_header_t *)ptr;

	header->total_buckets = buckets;
	header->allowed_collisions = ( allowed_collisions > 0UL ) ? allowed_collisions : buckets;
	(void) memset( (void *)&header[ 1 ], 0, bucket_arena_size );

	RETURN_POINTER( ptr );
}

/*
** This function searches for existence of data key in hash table. If it
** is not found, it is inserted. Either way, the address of the hash bucket
** is returned.
**
** Arguments:
**	1. data is the address of data that can be used to generate a hash key.
**	2. arena is the address returned by dhcreate()
**	3. calc_key is the address to a function that generates a hash key
**	   from data variable.
**	4. key_cmp is the address to a function that can compare passed in
**	   data with existing data in hash table to decide if it already exists.
*/
void *dhsearch( const void *data, void *arena, unsigned long ( *calc_key )( const void * ), int ( *key_cmp )( const void *, const void * ) )
{
	htable_header_t	*header;
	unsigned long	key, ndx, delta_ndx, total_buckets;
	unsigned long	collision_cnt, *ndx_tbl = 0;
	unsigned long	first_del_col_cnt_ndx, allowed_collisions;
	const void	**bucket;

	assert( arena != (void *)0 );
	assert( data != (const void *)0 );
	assert( calc_key != (unsigned long (*)( const void * ))0 );
	assert( key_cmp != (int (*)( const void *, const void * ))0 );

	DEBUG_FUNC_START;

	header = (htable_header_t *)arena;
	bucket = (const void **)&header[ 1 ];
	total_buckets = header->total_buckets;
	allowed_collisions = header->allowed_collisions;

	key = ( *calc_key )( data );

	if ( Debug ) {
		(void) fprintf( stderr, "Key value is %lu.\n", key );
	}

	/*
	** Knuth's D1 step -- first hash.
	*/
	ndx = key % total_buckets;

	if ( Debug ) {
		(void) fprintf( stderr, "Index into hash table with %lu.\n", ndx );
	}

	collision_cnt = 0UL;
	first_del_col_cnt_ndx = allowed_collisions;

	while ( collision_cnt < total_buckets ) {
		/*
		** Step D2 and D5 -- first and subsequent probes.
		*/
		if ( OCCUPIED_BUCKET( bucket[ ndx ] ) ) {
			if ( ( *key_cmp )( data, bucket[ ndx ] ) == 0 ) {
				/*
				** Key already exists.
				*/
				if ( ndx_tbl != (unsigned long *)0 ) {
					free( (void *)ndx_tbl );
				}
				if ( Debug ) {
					(void) fprintf( stderr, "Already exists at index %lu.\n", ndx );
				}
				RETURN_POINTER( &bucket[ ndx ] );
			}
		} else {
			/*
			** Unoccupied bucket.
			*/
			if ( bucket[ ndx ] == empty_bucket ) {
				/*
				** Bucket has never been populated.
				*/
				break;
			}

			/*
			** Encountered a deleted bucket.
			*/
			if ( first_del_col_cnt_ndx == allowed_collisions ) {
				/*
				** Keep track of the first deleted bucket
				** encountered.
				*/
				first_del_col_cnt_ndx = collision_cnt;
			}
		}

		if ( Debug ) {
			(void) fputs( "Collision\n", stderr );
		}

		if ( ++collision_cnt < 2UL ) {
			/*
			** First collision. Calculate second and subsequent
			** hash value. Knuth's step D3.
			*/
			delta_ndx = CALC_DELTA_NDX( key, total_buckets );

			if ( Debug ) {
				(void) fprintf( stderr, "Delta index into hash table with %lu.\n", delta_ndx );
			}

			ndx_tbl = (unsigned long *)malloc( (size_t) ( sizeof( unsigned long ) * allowed_collisions ) );
			if ( ndx_tbl == (unsigned long *)0 ) {
				dherrno = Dh_alloc;
				RETURN_POINTER( (void *)0 );
			}
		}

		if ( collision_cnt <= allowed_collisions ) {
			/*
			** Save index for later optimizations.
			*/
			ndx_tbl[ collision_cnt - 1 ] = ndx;
		}

		/*
		** Step D4, advance to next, is in apply_delta_ndx().
		*/
		ndx = apply_delta_ndx( ndx, delta_ndx, total_buckets );
	}

	if ( first_del_col_cnt_ndx < allowed_collisions ) {
		/*
		** Have passed a deleted bucket. Step back to it.
		*/
		ndx = ndx_tbl[ first_del_col_cnt_ndx ];
		collision_cnt = first_del_col_cnt_ndx + 1UL;
	}

	if ( collision_cnt >= total_buckets ) {
		/*
		** Collision count exceeds maximum
		** number of allowed collisions.
		*/
		if ( ndx_tbl != (unsigned long *)0 ) {
			free( (void *)ndx_tbl );
		}
		if ( Debug ) {
			(void) fprintf( stderr, "Maximum number of collisions (%lu) were exceeded.\n", header->allowed_collisions );
		}
		dherrno = Dh_collision;
		RETURN_POINTER( (void *)0 );
	}

	if ( collision_cnt > 1UL ) {
		/*
		** Perform Brent's Variation.
		*/
		if ( optimize_collisions( &ndx, bucket, ndx_tbl, collision_cnt, allowed_collisions, calc_key, total_buckets ) == -1 ) {
			/*
			** Collision count exceeds maximum
			** number of allowed collisions.
			*/
			if ( ndx_tbl != (unsigned long *)0 ) {
				free( (void *)ndx_tbl );
			}
			if ( Debug ) {
				(void) fprintf( stderr, "Maximum number of collisions (%lu) were exceeded.\n", header->allowed_collisions );
			}
			dherrno = Dh_collision;
			RETURN_POINTER( (void *)0 );
		}
	}

	/*
	** Step D6 -- insert.
	*/
	bucket[ ndx ] = data;
	++header->used_buckets;
	if ( ndx_tbl != (unsigned long *)0 ) {
		free( (void *)ndx_tbl );
	}
	if ( Debug ) {
		(void) fprintf( stderr, "Inserted at index %lu.\n", ndx );
	}

	RETURN_POINTER( &bucket[ ndx ] );
}

/*
** This function does Brent's Variation as described by Knuth.
** It checks the path of collisions looking for existing entries that 
** could be displaced using their second hash values. Existing entries
** are relocated if they require fewer collisions than the new entry.
** The new entry would then take newly vacated spot in the hash table.
*/
static int optimize_collisions( unsigned long *ret_ndx, const void **bucket, unsigned long *ndx_tbl, unsigned long collision_cnt, unsigned long allowed_collisions, unsigned long ( *calc_key )( const void * ), unsigned long total_buckets )
{
	unsigned long	ndx, prev_swap_ndx, i, max_ndx, max_collision;

	if ( collision_cnt > allowed_collisions ) {
		max_collision = allowed_collisions;
	} else {
		max_collision = collision_cnt;
	}

	max_ndx = max_collision - 1;

	for ( i = 0UL; i < max_ndx; ++i ) {
		ndx = ndx_tbl[ i ];

		if ( greater_collision( bucket, ndx, calc_key, &prev_swap_ndx, max_collision, total_buckets ) == 0 ) {
			/*
			** Swap buckets.
			*/
			bucket[ prev_swap_ndx ] = bucket[ ndx ];
			*ret_ndx = ndx;
			if ( Debug ) {
				(void) fprintf( stderr, "SWAP: Move existing bucket %lu to %lu and insert at %lu.\n", *ret_ndx, prev_swap_ndx, *ret_ndx );
			}
			break;
		}
		--max_collision;
	}

	/*
	** Could not optimize any collisions.
	** If collisions exceeded maximum allowable collisions, return -1.
	*/

	return ( i >= max_ndx && collision_cnt > allowed_collisions ) ? -1 : 0;
}

/*
** This function returns a new hash table index based on the current index,
** second hash value and hash table size.
*/
static unsigned long apply_delta_ndx( unsigned long ndx, unsigned long delta_ndx, unsigned long total_buckets )
{
	unsigned long	new_ndx;

	assert( ULONG_MAX - delta_ndx > ndx );

	if ( Debug ) {
		(void) fputs( "Start apply_delta_ndx().\n", stderr );
	}

	new_ndx = ndx + delta_ndx;

	if ( new_ndx >= total_buckets ) {
		new_ndx -= total_buckets;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "New index is %lu.\n", new_ndx );
		(void) fputs( "End apply_delta_ndx().\n", stderr );
	}

	return new_ndx;
}

/*
** This function check to see if an existing hash table entry would have
** fewer collisions than a entry about to be inserted. This is part of
** Brent's Variation.
*/
static int greater_collision( const void **bucket, unsigned long ndx, unsigned long ( *calc_key )( const void * ), unsigned long *avail_ndx, unsigned long collision_cnt, unsigned long total_buckets )
{
	unsigned long	key, delta_ndx, i;

	if ( Debug ) {
		(void) fputs( "Start get_collision_cnt().\n", stderr );
	}

	key = ( *calc_key )( bucket[ ndx ] );
	delta_ndx = CALC_DELTA_NDX( key, total_buckets );

	if ( Debug ) {
		(void) fprintf( stderr, "Delta index into hash table with %lu.\n", delta_ndx );
	}

	for ( i = 1UL; i < collision_cnt; ++i ) {
		ndx = apply_delta_ndx( ndx, delta_ndx, total_buckets );
		if ( UNOCCUPIED_BUCKET( bucket[ ndx ] ) ) {
			break;
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "%lu collisions occurred while trying to improve on %lu.\n", i, collision_cnt );
		(void) fputs( "End get_collision_cnt().\n", stderr );
	}

	if ( i < collision_cnt ) {
		/*
		** Better index found.
		*/
		*avail_ndx = ndx;
		return 0;
	}

	/*
	** Could not improve on index.
	*/
	return -1;
}

/*
** This function searches for an existing hash table entry.
**
** Arguments:
**	1. data is the address of data that can be used for comparing.
**	2. arena is the address returned by dhcreate()
**	3. key is the hash key generated from data.
**	4. key_cmp is the address to a function that can compare passed in
**	   data with existing data in hash table to decide matches.
*/
void *dhfind( const void *data, const void *arena, unsigned long key, int ( *key_cmp )( const void *, const void * ) )
{
	htable_header_t	*header;
	unsigned long	ndx, delta_ndx, total_buckets;
	unsigned long	collision_cnt;
	const void	**bucket;

	assert( arena != (void *)0 );
	assert( data != (const void *)0 );
	assert( key_cmp != (int (*)( const void *, const void * ))0 );

	DEBUG_FUNC_START;

	header = (htable_header_t *)arena;
	bucket = (const void **)&header[ 1 ];
	total_buckets = header->total_buckets;

	/*
	** First hash.
	*/
	ndx = key % total_buckets;

	if ( Debug ) {
		(void) fprintf( stderr, "Index into hash table with %lu.\n", ndx );
	}

	collision_cnt = 0UL;

	while ( collision_cnt < header->total_buckets ) {
		if ( OCCUPIED_BUCKET( bucket[ ndx ] ) ) {
			if ( ( *key_cmp )( data, bucket[ ndx ] ) == 0 ) {
				/*
				** Match found.
				*/
				if ( Debug ) {
					(void) fprintf( stderr, "Match found at index %lu.\n", ndx );
				}
				RETURN_POINTER( &bucket[ ndx ] );
			}
		} else {
			/*
			** Unoccupied bucket.
			*/
			if ( NEVER_BEEN_USED_BUCKET( bucket[ ndx ] ) ) {
				break;
			}
		}

		if ( ++collision_cnt < 2UL ) {
			/*
			** First collision. Calculate second and subsequent
			** hash value.
			*/
			delta_ndx = CALC_DELTA_NDX( key, total_buckets );

			if ( Debug ) {
				(void) fprintf( stderr, "Delta index into hash table with %lu.\n", delta_ndx );
			}
		}

		/*
		** Advance to next bucket.
		*/
		ndx = apply_delta_ndx( ndx, delta_ndx, header->total_buckets );
	}

	/*
	** Did not find match.
	*/
	if ( Debug ) {
		(void) fputs( "Did not find match.\n", stderr );
	}

	RETURN_POINTER( (void *)0 );
}

/*
** This function deletes an existing hash table entry.
**
** Arguments:
**	1. data is the address of data that can be used for comparing.
**	2. arena is the address returned by dhcreate()
**	3. key is the hash key generated from data.
**	4. key_cmp is the address to a function that can compare passed in
**	   data with existing data in hash table to decide matches.
*/
void *dhdelete( const void *data, const void *arena, unsigned long key, int ( *key_cmp )( const void *, const void * ) )
{
	void	**bucket, *existing_data;

	assert( arena != (void *)0 );
	assert( data != (const void *)0 );
	assert( key_cmp != (int (*)( const void *, const void * ))0 );

	DEBUG_FUNC_START;

	bucket = (void **)dhfind( data, arena, key, key_cmp );

	if ( bucket == (void **) 0 ) {
		RETURN_POINTER( (void *)0 );
	}

	existing_data = *bucket;
	*bucket = deleted_bucket;

	RETURN_POINTER( existing_data );
}

/*
** This function is a wrapper for dhsearch(). If dhsearch() fails because
** maximum collisions were encountered, it expands the hash area and retries
** dhsearch().
*/
void *dhesearch( const void *data, void **arena, unsigned long ( *calc_key )( const void * ), int ( *key_cmp )( const void *, const void * ) )
{
	htable_header_t	*old_header, *new_header;
	void 	*ptr, *new_hash;
	unsigned short	expansion_cnt;

	assert( arena != (void **)0 );
	assert( data != (const void *)0 );
	assert( calc_key != (unsigned long (*)( const void * ))0 );
	assert( key_cmp != (int (*)( const void *, const void * ))0 );

	DEBUG_FUNC_START;

	for ( expansion_cnt = (unsigned short)0; expansion_cnt < expansion_factor; ++expansion_cnt ) {
		ptr = dhsearch( data, *arena, calc_key, key_cmp );
		if ( ptr != (void *)0 ) {
			/*
			** Successful search/insert.
			*/
			break;
		}
		/*
		** Unsuccessful search/insert.
		*/
		if ( dherrno != Dh_collision ) {
			/*
			** Non-collision related error occurred.
			*/
			break;
		}

		new_hash = expand_hash( *arena, calc_key, key_cmp );

		if ( new_hash == (void *)0 ) {
			/*
			** Unsuccessful expansion.
			*/
			break;
		}

		if ( Debug ) {
			old_header = *arena;
			new_header = new_hash;
			(void) fprintf( stderr, "Expanded hash table from %lu to %lu buckets.\n", old_header->total_buckets, new_header->total_buckets );
		}

		free( *arena );
		*arena = new_hash;
	}

	RETURN_POINTER( ptr );
}

static void *expand_hash( void *old_hash, unsigned long ( *calc_key )( const void * ), int ( *key_cmp )( const void *, const void * ) )
{
	htable_header_t	*header;
	void 	*new_hash;
	unsigned long	new_bucket_cnt, expansion_increment;
	unsigned short	expand_cnt;
	int	copy_hash_status;

	header = (htable_header_t *)old_hash;

	/*
	** Resolve collision problem by expanding hash table.
	*/
	new_bucket_cnt = header->total_buckets;

	/*
	** Increase the number of buckets by expansion_factor.
	*/
	expansion_increment = header->total_buckets / (unsigned long)expansion_factor;

	/*
	** Allow hash table to grow no more than twice its current size
	** in expansion_factor increments.
	*/ 
	for ( expand_cnt = (unsigned short)0; expand_cnt < expansion_factor; ++expand_cnt ) {
		new_bucket_cnt += expansion_increment;

		new_hash = dhcreate( new_bucket_cnt, header->allowed_collisions );
		if ( new_hash == (void *)0 ) {
			/*
			** Could not create new hash table.
			*/
			break;
		}

		copy_hash_status = copy_hash( new_hash, old_hash, calc_key, key_cmp );
		if ( copy_hash_status == 0 ) {
			/*
			** Successfully copied hash table.
			*/
			break;
		}

		/*
		** Hash table copy failed.
		*/
		free( new_hash );

		if ( dherrno != Dh_collision ) {
			/*
			** Hash table copy failed from a problem
			** other than collision.
			*/
			new_hash = (void *)0;
			break;
		}

		/*
		** Hash table copy failed because maximum number
		** of collisions were reached. Expand and try again.
		*/
	}

	return new_hash;
}

static int copy_hash( void *new_hash, const void *old_hash, unsigned long ( *calc_key )( const void * ), int ( *key_cmp )( const void *, const void * ) )
{
	const htable_header_t	*header;
	const void	**bucket;
	void	**result;
	unsigned long	ndx, total_buckets;

	header = (htable_header_t *)old_hash;
	bucket = (const void **)&header[ 1 ];
	total_buckets = header->total_buckets;

	for ( ndx = 0UL; ndx < total_buckets; ++ndx, ++bucket ) {
		if ( UNOCCUPIED_BUCKET( *bucket ) ) {
			continue;
		}

		result = (void **)dhsearch( *bucket, (htable_header_t *)new_hash, calc_key, key_cmp );
		if ( result == (void **)0 ) {
			/*
			** dhsearch() failed copying existing hash
			** table into larger hash table.
			*/
			return -1;
		}
		/*
		** There should be no duplicates.
		*/
		assert( *result == *bucket );
	}

	return 0;
}

dherrno_t dherror( void )
{
	return dherrno;
}

unsigned long dhused_buckets( void *arena )
{
	assert( arena != (void *)0 );

	return ( (htable_header_t *)arena )->used_buckets;
}

unsigned long dhtotal_buckets( void *arena )
{
	assert( arena != (void *)0 );

	return ( (htable_header_t *)arena )->total_buckets;
}

unsigned long dhallowed_collisions( void *arena )
{
	assert( arena != (void *)0 );

	return ( (htable_header_t *)arena )->allowed_collisions;
}

int dhis_occupied_bucket( const void *bucket )
{
	return ( OCCUPIED_BUCKET( bucket ) ) ? 1 : 0;
}

void dhtable( const void *arena, const void **table, unsigned long *nel )
{
	const htable_header_t	*header;

	assert( arena != (void *)0 );
	assert( table != (const void **)0 );
	assert( nel != (unsigned long *)0 );

	header = (htable_header_t *)arena;
	*nel = header->total_buckets;
	*table = (const void *) &header[ 1 ];
}

#ifdef MT_dhsearch
/*
** This main() function is used to regression test routines in this module.
** The following command is used to compile:
** 	x=dhsearch; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/

static void print_hash_table( const void * );
int compare_colors( const void *, const void * );
extern unsigned long strhkey( const char * );

static unsigned long calc_key( const void *x )
{
	return strhkey( (char *)x );
}

int main( void )

{
	static const char	blank_line[] = ">>>\n";
	static char	*keys[] = {
		"yellow", "green", "brown", "orange", "black",
		"red", "blue", "purple", "gray", "white", 0
	};
	char	**key_ptr, **key_find, *key_delete;
	void	*hash_tbl;

	Debug = 1;

	setbuf( stdout, (char *) 0 );

	(void) fprintf( stderr, ">>> Start double hash module test.\n" );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Build hash table with color names as keys.\n", stderr );
	(void) fputs( ">>> Allocate enough space for ten colors.\n", stderr );
	(void) fputs( blank_line, stderr );

	hash_tbl = dhcreate( 10UL, 3UL );

	if ( hash_tbl == (void *)0 ) {
		(void) fputs( ">>> dhcreate() failed.\n", stderr );
		return 5;
	}

	(void) fputs( ">>> dhcreate() was successful.\n", stderr );
	(void) fputs( blank_line, stderr );

	for ( key_ptr = keys; *key_ptr != (char *) 0; ++key_ptr ) {
		(void) fprintf( stderr, ">>> Adding color [%s] to hash table.\n", *key_ptr );
		if ( ( key_find = (char **)dhsearch( (void *) *key_ptr, hash_tbl, calc_key, compare_colors ) ) == (char **) 0 ) {
			(void) fputs( ">>> dhsearch() failed.\n", stderr );
			return 10;
		}
		if ( strcmp( *key_ptr, *key_find ) != 0 ) {
			(void) fprintf( stderr, ">>> dhsearch() incorrectly return [%s] when adding [%s].\n", *key_find, *key_ptr );
			return 12;
		}
		(void) fprintf( stderr, ">>> Color [%s] was correctly added to hash table.\n", *key_ptr );
		(void) fputs( blank_line, stderr );
		(void) fputs( ">>> Print current hash table.\n", stderr );
		print_hash_table( hash_tbl );
		(void) fputs( blank_line, stderr );
	}

	(void) fputs( ">>> Find the color orange using dhfind().\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **) dhfind( (void *) "orange", hash_tbl, calc_key( "orange" ), compare_colors ) ) == (char **) 0 ) {
		(void) fputs( ">>> dhfind() failed to find the color orange.\n", stderr );
		return 15;
	}

	if ( strcmp( *key_find, "orange" ) != 0 ) {
		(void) fprintf( stderr, ">>> dhfind() incorrectly return [%s] when searching for orange.\n", *key_find );
		return 17;
	}
	(void) fputs( ">>> dhfind() correctly found the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Delete orange node from hash table.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_delete = (char *) dhdelete( (void *) "orange", hash_tbl, calc_key( "orange" ), compare_colors ) ) == (char *) 0 ) {
		(void) fputs( ">>> dhdelete() failed to delete the color orange.\n", stderr );
		return 20;
	}

	if ( strcmp( key_delete, "orange" ) != 0 ) {
		(void) fprintf( stderr, ">>> dhdelete() incorrectly deleted [%s] while attempting to delete orange.\n", key_delete );
		return 25;
	}
	(void) fputs( ">>> dhdelete() correctly deleted the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Attempt invalid search for the color orange using dhfind().\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **) dhfind( (void *) "orange", hash_tbl, calc_key( "orange" ), compare_colors ) ) != (char **) 0 ) {
		(void) fputs( ">>> dhfind() incorrectly returned a color from hash table.\n", stderr );
		return 27;
	}
	(void) fputs( ">>> dhfind() correctly failed to find the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Attempt to delete orange bucket from hash table.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_delete = (char *) dhdelete( (void *) "orange", hash_tbl, calc_key( "orange" ), compare_colors ) ) != (char *) 0 ) {
		(void) fputs( ">>> dhdelete() incorrectly deleted bucket from hash table.\n", stderr );
		return 30;
	}
	(void) fputs( ">>> dhdelete() correctly did not delete bucket from hash table.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Print current hash table.\n", stderr );
	(void) fputs( blank_line, stderr );
	print_hash_table( hash_tbl );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Add color orange to hash table.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **)dhsearch( (void *) "orange", hash_tbl, calc_key, compare_colors ) ) == (char **) 0 ) {
		(void) fputs( ">>> dhsearch() failed.\n", stderr );
		return 33;
	}
	if ( strcmp( "orange", *key_find ) != 0 ) {
		(void) fprintf( stderr, ">>> dhsearch() incorrectly returned [%s] when adding orange.\n", *key_find );
		return 35;
	}
	(void) fputs( ">>> Color orange was correctly added to hash table.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Print current hash table.\n", stderr );
	(void) fputs( blank_line, stderr );
	print_hash_table( hash_tbl );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Double hash module testing is complete.\n", stderr );
	return 0;
}

int compare_colors( const void *x, const void *y )
{
	return strcmp( (char *) x, (char *) y );
}

/*
** This function prints all hash table entries.
*/
static void print_hash_table( const void *arena )
{
	const char	**hash_tbl;
	unsigned long	i, cnt = 0, nel;

	dhtable( arena, (const void **)&hash_tbl, &nel );

	for ( i = 0; i < nel; ++i ) {
		if ( dhis_occupied_bucket( (void *)hash_tbl[ i ] ) ) {
			(void) printf( "%d. [%s], location %lu\n", ++cnt, hash_tbl[ i ], i );
		}
	}
}
#endif
