/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"


static const dfile_state_t	write_state_tbl[] = {
	{ Dfile_format_state, &write_state_tbl[ 1 ] },
	{ Dfile_io_state, &write_state_tbl[ 0 ] }
};

/*
** This function is called by user application to allocate and initialize
** dfile_t structure for writing data file. It also opens the output file.
*/

dfile_t *dfile_write_open( const dfile_cfg_t *cfg, const dfile_bind_t *bind, unsigned short bind_cnt, const dfile_tag_t *tag, unsigned short tag_cnt, unsigned short buffer_block_cnt, unsigned short buffer_cnt, dfile_open_mode_t open_mode )
{
	dfile_t	*dfile;

	DEBUG_FUNC_START;

	if ( _dfile_vdate_rwopen_args( cfg, bind, bind_cnt, buffer_block_cnt, buffer_cnt ) == -1 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Invalid function arguments while opening [", stderr );
		if ( cfg != (dfile_cfg_t *)0 && cfg->dfile_name != (char *)0 ) {
			(void) fputs( cfg->dfile_name, stderr );
		} else {
			(void) fputs( "unknown", stderr );
		}
		(void) fputs( "].\n", stderr );

		RETURN_POINTER( (dfile_t *)0 );
	}

	dfile = _dfile_construct( cfg, bind, bind_cnt, tag, tag_cnt, buffer_block_cnt, buffer_cnt, write_state_tbl );

	if ( dfile == (dfile_t *)0 ) {
		RETURN_POINTER( (dfile_t *)0 );
	}

	dfile->record_failure_func = _dfile_var_field_failure;
	dfile->rec_attribute = cfg->rec_attribute;

	if ( dfile->rec_attribute.separator_escape != (char)0 ) {
		dfile->format_str_func = _dfile_format_escaped_str_field;
		dfile->format_func = _dfile_format_escaped_field;
		dfile->record_type = Dfile_delimited;
	} else {
		if ( dfile->rec_attribute.field_separator != (char)0 ) {
			dfile->format_str_func = _dfile_format_str_field;
			dfile->format_func = _dfile_format_field;
			dfile->record_type = Dfile_delimited;
		} else {
			dfile->format_str_func = (int ( * )( char **, const char *, const char **, dfile_rec_t ))0;
			dfile->format_func = (int ( * )( char **, const char *, const char **, size_t *, dfile_rec_t ))0;
			dfile->record_type = Dfile_variable_length;
		}
	}

	dfile->open_mode = open_mode;

	if ( _dfile_write_open_file( dfile ) == -1 ) {
		RETURN_POINTER( (dfile_t *)0 );
	}

	if ( dfile->format == Dfile_zipped ) {
		if ( _dfile_allocate_zipped_stream( dfile, 'w' ) == -1 ) {
			RETURN_POINTER( (dfile_t *)0 );
		}

		dfile->io_buffer_func = _dfile_flush_zipped_buffer;
		_dfile_write_gz_header( dfile );
		dfile->crc = crc32( 0UL, (const Bytef *)0, 0U );
		dfile->file_char_cnt = 0UL;
	} else {
		dfile->io_buffer_func = _dfile_flush_ascii_buffer;
	}

#ifdef DFILE_THREAD
	if ( buffer_cnt > (unsigned short)1 ) {
		if ( _dfile_buffer_wait( dfile->io_buffer, Dfile_format_state, dfile->buffer_cnt ) == -1 ) {
			RETURN_POINTER( (dfile_t *)0 );
		}

		if ( _dfile_create_thread( &dfile->thread, _dfile_buffer_write_thread, dfile ) == -1 ) {
			RETURN_POINTER( (dfile_t *)0 );
		}
	}
#endif

	RETURN_POINTER( dfile );
}
