/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"


static const dfile_state_t	read_state_tbl[] = {
	{ Dfile_io_state, &read_state_tbl[ 1 ] },
	{ Dfile_parse_state, &read_state_tbl[ 0 ] }
};

/*
** This function is called by user application to allocate and initialize
** dfile_t structure for reading data files.
*/

dfile_t *dfile_read_open( const dfile_cfg_t *cfg, dfile_bind_t *bind, unsigned short bind_cnt, const dfile_tag_t *tag, unsigned short tag_cnt, unsigned short buffer_block_cnt, unsigned short buffer_cnt )
{
	dfile_t	*dfile;

	DEBUG_FUNC_START;

	if ( _dfile_vdate_rwopen_args( cfg, bind, bind_cnt, buffer_block_cnt, buffer_cnt ) == -1 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Invalid function arguments while opening [", stderr );
		if ( cfg != (const dfile_cfg_t *)0 && cfg->dfile_name != (const char *)0 ) {
			(void) fputs( cfg->dfile_name, stderr );
		} else {
			(void) fputs( "unknown", stderr );
		}
		(void) fputs( "].\n", stderr );
		RETURN_POINTER( (dfile_t *)0 );
	}

	dfile = _dfile_construct( cfg, bind, bind_cnt, tag, tag_cnt, buffer_block_cnt, buffer_cnt, read_state_tbl );

	if ( dfile == (dfile_t *)0 ) {
		RETURN_POINTER( (dfile_t *)0 );
	}

	dfile->record_failure_func = _dfile_parse_record_failure;

	if ( dfile->rec_attribute.separator_escape != (char)0 ) {
		dfile->parse_func = _dfile_parse_escaped_field;
		dfile->overflow_func = _dfile_escaped_overflow;
		dfile->record_type = Dfile_delimited;
	} else {
		if ( dfile->rec_attribute.field_separator != (char)0 ) {
			dfile->parse_func = _dfile_parse_field;
			dfile->overflow_func = _dfile_overflow;
			dfile->record_type = Dfile_delimited;
		} else {
			dfile->format_str_func = (int ( * )( char **, const char *, const char **, dfile_rec_t ))0;
			dfile->format_func = (int ( * )( char **, const char *, const char **, size_t *, dfile_rec_t ))0;
			dfile->record_type = Dfile_variable_length;
		}
	}

	if ( _dfile_read_open_file( dfile ) == -1 ) {
		RETURN_POINTER( (dfile_t *)0 );
	}

#ifdef DFILE_THREAD
	if ( buffer_cnt > (unsigned short)1 ) {
		if ( _dfile_create_thread( &dfile->thread, _dfile_buffer_read_thread, dfile ) == -1 ) {
			RETURN_POINTER( (dfile_t *)0 );
		}
	}
#endif

	RETURN_POINTER( dfile );
}
