/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "sexpr.h"
#include "dfile_utility.h"
#include "dfile_sort.h"


typedef struct {
	char	*description;
	char	code;
} code_map_t;

/*
** This function assigns information from control structure.
*/
int assign_algorithm( char *sort_algorithm, sexpr_t *sexpr, unsigned short tuple_cnt )
{
	static const char	str_msg[] = "expected a string";
	static code_map_t	code_map[] = {
		{ "insertion-sort", 'I' },
		{ "shell-sort", 'S' },
		{ "heap-sort", 'H' },
		{ "merge-sort", 'M' },
		{ "quick-sort", 'Q' }
	};
	const size_t	code_map_cnt = sizeof( code_map ) / sizeof( code_map_t );
	size_t	ndx;

	assert( sort_algorithm != (char *)0 );
	assert( sexpr != (sexpr_t *)0 );

	DEBUG_FUNC_START;

	if ( SEXPR_CAR_TYPE( sexpr ) != string_sexpr ) {
		CONTROL_SYNTAX_ERROR( str_msg, tuple_cnt );
		RETURN_INT( -1 );
	}

	assert( SEXPR_CDR_TYPE( sexpr ) == list_sexpr );
	if ( SEXPR_CDR_LIST( sexpr ) != (sexpr_t *)0 ) {
		control_syntax_error( "extra argument found for algorithm", __FILE__, __LINE__, tuple_cnt );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "algorithm: [%s]\n", SEXPR_CAR_STRING( sexpr ) );
	}

	for ( ndx = (size_t)0; ndx < code_map_cnt; ++ndx ) {
		if ( strcasecmp( SEXPR_CAR_STRING( sexpr ), code_map[ ndx ].description ) == 0 ) {
			*sort_algorithm = code_map[ ndx ].code;
			break;
		}
	}

	if ( ndx >= code_map_cnt ) {
		control_syntax_error( "unknown algorithm", __FILE__, __LINE__, tuple_cnt );
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
