/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"

static const char       rcsid[] = "$Id: parse_list.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: parse_list.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 18:21:58  keith
** Initial revision
**
*/

static int	copy_element( const char ***, unsigned long, const char *, size_t );

/*
** Parses a simple list and copies results into an array of pointers.
*/
int parse_list( const char ***list_tbl, unsigned long *list_tbl_cnt, const char *str, char delim )
{
	static const char	func[] = "parse_list";
	const char	*delim_ptr, *element;
	size_t	element_length;

	assert( list_tbl != (const char ***)0 );
	assert( list_tbl_cnt != (unsigned long *)0 );
	assert( str != (const char *)0 );

	DEBUG_FUNC_START;

	*list_tbl_cnt = 0UL;
	*list_tbl = (const char **)0;

	if ( *str == (char)0 ) {
		/*
		** Zero length string.
		*/
		RETURN_INT( 0 );
	}

	element = str;

	for ( ;; ) {
		delim_ptr = (const char *)strchr( element, delim );
		if ( delim_ptr == (const char *)0 ) {
			break;
		}

		assert( delim_ptr >= element );
		element_length = (size_t)( delim_ptr - element );

		if ( copy_element( list_tbl, *list_tbl_cnt, element, element_length ) == -1 ) {
			RETURN_INT( -1 );
		}

		element = &delim_ptr[ 1 ];
		++*list_tbl_cnt;
	}

	if ( copy_element( list_tbl, *list_tbl_cnt, element, strlen( element ) ) == -1 ) {
		RETURN_INT( -1 );
	}

	++*list_tbl_cnt;

	RETURN_INT( 0 );
}

static int copy_element( const char ***list, unsigned long list_cnt, const char *element, size_t element_length )
{
	char	**new, *new_str;
	size_t	alloc_size;

	alloc_size = ( (size_t)list_cnt + (size_t)1 ) * sizeof( char * );
	new = (char **)realloc( (void *)*list, alloc_size );

	if ( new == (char **)0 ) {
		unix_error( "realloc() failed", __FILE__, __LINE__ );
		return -1;
	}

	alloc_size = element_length + (size_t)1;

	new_str = (char *)malloc( alloc_size );
	if ( new_str == (char *)0 ) {
		unix_error( "malloc() failed", __FILE__, __LINE__ );
		return -1;
	}

	if ( element_length > (size_t)0 ) {
		(void) memcpy( (void *)new_str, (void *)element, element_length );
	}

	new_str[ element_length ] = (char)0;
	new[ list_cnt ] = (char *)new_str;

	*list = (const char **)new;

	return 0;
}

#ifdef MT_parse_list
/*
** This function is used to regression test parse_list().
** The following command is used to compile:
** x=parse_list; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/

int main( void )
{
	static const char	blank_line[] = ">>>\n";
	static const char	abort_msg[] = ">>> Aborting test.\n";
	static const char	test_func[] = "parse_list";
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";

	const char	**field;
	unsigned long	field_cnt;

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );


	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Test zero length string.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( parse_list( &field, &field_cnt, "", ',' ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	if ( field_cnt != 0UL || field != (const char **)0 ) {
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Test one element.\n", stderr );
	(void) fputs( blank_line, stderr );


	if ( parse_list( &field, &field_cnt, "ABC", ',' ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	if ( field_cnt != 1UL || strcmp( field[ 0 ], "ABC" ) != 0 ) {
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Test four elements.\n", stderr );
	(void) fputs( blank_line, stderr );


	if ( parse_list( &field, &field_cnt, "AAA,,CCC,DDD", ',' ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	if ( field_cnt != 4UL || strcmp( field[ 0 ], "AAA" ) != 0 || strcmp( field[ 1 ], "" ) != 0 || strcmp( field[ 2 ], "CCC" ) != 0 || strcmp( field[ 3 ], "DDD" ) != 0 ) {
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		return 10;
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	return 0;
}
#endif
