/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "tbox.h"

static const char       rcsid[] = "$Id: fput_wrappers.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: fput_wrappers.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 09:15:54  keith
** Initial revision
**
*/

#undef Debug
#define Debug	0

/*
** This file contains wrapper functions for outputting non-ASCII values.
*/

/*
** This function outputs a double in ASCII format.
*/
int fput_double( double x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_double";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%g", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for double failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a short in ASCII format.
*/
int fput_short( short x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_short";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%hd", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for short failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a ushort in ASCII format.
*/
int fput_ushort( unsigned short x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_ushort";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%hu", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for unsigned short failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a long in ASCII format.
*/
int fput_long( long x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_long";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%ld", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for long failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a ulong in ASCII format.
*/
int fput_ulong( unsigned long x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_ulong";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%lu", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for unsigned long failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a int in ASCII format.
*/
int fput_int( int x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_int";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%d", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for int failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs a uint in ASCII format.
*/
int fput_uint( unsigned int x, FILE *fptr )
{
	static const char	end_msg[] = "End %s() returning %d at %s\n";
	static const char	func[] = "fput_uint";

	assert( fptr != (FILE *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	if ( fprintf( fptr, "%u", x ) <= 0 ) {
		UNIX_ERROR( "fprintf() for unsigned int failed" );
		if ( Debug ) {
			(void) fprintf( stderr, end_msg, func, -1, get_ctime() );
		}

		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, end_msg, func, 0, get_ctime() );
	}

	return 0;
}

/*
** This function outputs source code location.
*/
void fput_src_code( const char *fname, int line, FILE *fptr )
{
	assert( fname != (const char *)0 );
	assert( line > 0 );
	assert( fptr != (FILE *)0 );

	(void) fputs( fname, fptr );
	(void) fputc( '(', fptr );
	(void) fput_int( line, fptr );
	(void) fputs( "): ", fptr );
}


#ifdef MT_fput_wrappers
/*
** This function is used to regression test fput wrappers.
** The following command is used to compile:
** x=fput_wrappers; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
int main( void )

{
	Debug = 0;

	(void) fputs( "fput_double( 5.0 ) is [", stdout );
	if ( fput_double( 5.0, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_short( 5 ) is [", stdout );
	if ( fput_short( (short)5, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_ushort( 5 ) is [", stdout );
	if ( fput_ushort( (unsigned short)5, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_long( 5 ) is [", stdout );
	if ( fput_long( 5L, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_ulong( 5 ) is [", stdout );
	if ( fput_ulong( 5UL, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_int( 5 ) is [", stdout );
	if ( fput_int( 5, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "fput_uint( 5 ) is [", stdout );
	if ( fput_uint( 5, stdout ) == -1 ) {
		fputs( "FAILURE\n", stderr );
		return 1;
	}
	(void) fputs( "].\n\n", stdout );

	(void) fputs( "Display source code location:\n", stdout );
	fput_src_code( __FILE__, __LINE__, stdout );
	(void) fputc( '\n', stdout );

	exit( 0 );
}
#endif
