/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"

static const char       rcsid[] = "$Id: cfg_read.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: cfg_read.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 09:08:31  keith
** Initial revision
**
*/

static readstat_t get_rec( char *, size_t, unsigned long, FILE *, const char * );
static int parse_rec( char **, unsigned short, char *, unsigned long, char, char, const char * );
static void strip_escape( char **, unsigned short, char, char );

static const char	col_proc_msg[] = "Columns processed:\n";

/*
** This function reads and parses configuration file records.
** Blank and comment lines are skipped.
** If the delimiter character exists as data, it is expected to be
** preceded by the escape character '\'.
*/
readstat_t cfg_read( const char **field, unsigned short num_of_fields, unsigned long *line_cnt, char field_delim, FILE *fptr, const char *file_path )
{
	static const char	func[] = "cfg_read";

	static char		buf[BUFSIZ*4];

	readstat_t	readstat;
	const char	escape_delim = '\\';

	assert( field != (const char **) 0 );
	assert( fptr != (FILE *)0 );
	assert( file_path != (const char *) 0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( num_of_fields = %hu )\n", func, num_of_fields );
	}

	DEBUG_FUNC_START;

	/*
	** Loop will discard blank lines and comments.
	*/
	for ( ;; ) {
		++*line_cnt;

		readstat = get_rec( buf, sizeof( buf ), *line_cnt, fptr, file_path );
		if ( readstat == Read_err ) {
			/*
			** Skip incorrectly formatted record or comment.
			*/
			continue;
		}

		if ( readstat == Read_fatal || readstat == Read_eof ) {
			--*line_cnt;
			break;
		}

		if ( parse_rec( (char **)field, num_of_fields, buf, *line_cnt, field_delim, escape_delim, file_path ) == 0 ) {
			/*
			** Parsed a good record.
			*/
			break;
		}

		/*
		** Skip record.
		*/
	}

	assert ( readstat != Read_err );

	switch ( readstat ) {
	case Read_fatal:
	case Read_eof:
		RETURN_INT( readstat );
	}

	if ( Debug ) {
		(void) fputs( col_proc_msg, stderr );
	}

	strip_escape( (char **)field, num_of_fields, field_delim, escape_delim );

	RETURN_INT( Read_ok );
}

static readstat_t get_rec( char *buf, size_t buf_size, unsigned long line_cnt, FILE *fptr, const char *file_path )
{
	size_t	length;

	if ( fgets( buf, (int)buf_size, fptr ) == (char *) 0 ) {
		if ( ferror( fptr ) ) {
			/*
			** Error occurred in fgets().
			*/
			UNIX_ERROR( "fgets() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "fgets() failed on line ", stderr );
			(void) fput_ulong( line_cnt, stderr );
			(void) fputs( " of data for file [", stderr );
			(void) fputs( file_path, stderr );
			(void) fputs( "].\n", stderr );
			return Read_fatal;
		}

		/*
		** End of file processing reached.
		*/
		return Read_eof;
	}

	length = strlen( buf );
	assert ( length > (size_t) 0 );
	--length;

	if ( buf[ length ] != '\n' ) {
		/*
		** Did not get entire line.
		*/
		FPUT_SRC_CODE( stderr );
		(void) fputs( "line ", stderr );
		(void) fput_ulong( line_cnt, stderr );
		(void) fputs( " of data was longer than expected maximum ", stderr );
		(void) fput_uint( (unsigned int)buf_size - 1U, stderr );
		(void) fputs( " characters.\n", stderr );

		return Read_fatal;
	}

	if ( length == 0 || *buf == '#' ) {
		return Read_err;
	}

	/*
	** Strip newline character.
	*/
	buf[ length ] = (char) 0;

	return Read_ok;
}

static int parse_rec( char **field, unsigned short num_of_fields, char *buf, unsigned long line_cnt, char field_delim, char escape_delim, const char *file_path )
{
	static const char	parsed_msg[] = "Parsed field %hu value [%s].\n";
	char		*delim_ptr, *field_ptr, *field_start;
	unsigned short	field_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, "Line %lu of configuration [%s].\n", line_cnt, buf );
	}

	if ( field_delim == (char) 0 ) {
		/*
		** Field delimiters are not being used.
		*/
		field[ 0 ] = buf;
		if ( Debug ) {
			(void) fputs( col_proc_msg, stderr );
			(void) fprintf( stderr, "1. [%s]\n", field[ 0 ] );
		}

		return 0;
	}

	/*
	** Replace field delimiters with null values to terminate
	** field strings.
	*/
	field_start = field_ptr = buf;
	field_cnt = (unsigned short)0;

	while ( ( delim_ptr = strchr( field_ptr, (int) field_delim ) ) != (char *) 0 ) {
		if ( delim_ptr > field_ptr && delim_ptr[ -1 ] == escape_delim ) {
			/*
			** A field delimiter value was found to be escaped.
			** Do not use it as a field delimiter.
			** Advance pointer to next field.
			*/
			field_ptr = &delim_ptr[ 1 ];
			continue;
		}

		/*
		** Make field a null terminated string.
		*/
		*delim_ptr = (char) 0;

		/*
		** Save address of field if it has not progressed beyond
		** the end of the expected number of field.
		*/
		if ( field_cnt < num_of_fields ) {
			field[ field_cnt ] = field_start;
		}

		if ( Debug ) {
			(void) fprintf( stderr, parsed_msg, field_cnt + 1, field_start );
		}

		/*
		** Advance pointer to next field.
		*/
		field_start = field_ptr = &delim_ptr[ 1 ];
		++field_cnt;
	}

	/*
	** Save address of last field if it has not progressed beyond
	** the end of the expected number of fields.
	*/
	if ( field_cnt < num_of_fields ) {
		field[ field_cnt ] = field_start;
	}

	++field_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, parsed_msg, field_cnt, field_start );
	}

	if ( field_cnt != num_of_fields ) {
		/*
		** Parsed incorrect number of fields.
		*/
		FPUT_SRC_CODE( stderr );
		(void) fputs( "line ", stderr );
		(void) fput_ulong( line_cnt, stderr );
		(void) fputs( " of file [", stderr );
		(void) fputs( file_path, stderr );
		(void) fputs( "] had ", stderr );
		(void) fput_ushort( field_cnt, stderr );
		(void) fputs( " fields; expected ", stderr );
		(void) fput_ushort( num_of_fields, stderr );
		(void) fputs( " fields.\n", stderr );

		return -1;
	}

	return 0;
}

static void strip_escape( char **field, unsigned short num_of_fields, char field_delim, char escape_delim )
{
	unsigned short	field_cnt;
	char	*field_ptr, *delim_ptr;

	for ( field_cnt = (unsigned short) 0; field_cnt < num_of_fields; ++field_cnt ) {
		/*
		** Strip any escape characters that precedes field delimiter
		** characters.
		*/
		field_ptr = field[ field_cnt ];
		while ( ( delim_ptr = strchr( field_ptr, (int) field_delim ) ) != (char *) 0 ) {
			assert( delim_ptr > field_ptr && delim_ptr[ -1 ] == escape_delim );
			(void) strcpy( &delim_ptr[ -1 ], delim_ptr );
			field_ptr = delim_ptr;
		}
		if ( Debug ) {
			(void) fprintf( stderr, "%hu. [%s]\n", field_cnt + 1, field[ field_cnt ] );
		}
	}
}

#ifdef MT_cfg_read
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
/*
** This function is used to regression test cfg_read().
** The following command is used to compile:
** x=cfg_read; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/

static int call_cfg_read( char **, unsigned short, pid_t *, int * );

void main( void )
{
	static const char	blank_line[] = ">>>\n";
	static const char	abort_msg[] = ">>> Aborting test.\n";
	static const char	test_func[] = "cfg_read";
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";

	char	*field[10];
	pid_t	pid;
	int	fd[2];
	char	big_guy[BUFSIZ*5];
	unsigned long	rec_cnt;

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );


	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Test four records with one field of data.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 1, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) puts( "A" );
	(void) puts( "B" );
	(void) puts( "C" );
	(void) puts( "D" );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Test four records with more than one field of data.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 4, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) puts( "A|BB|CCC|DDDD" );
	(void) puts( "E|FF|GGG|HHHH" );
	(void) puts( "I|JJ|KKK|LLLL" );
	(void) puts( "M|NN|OOO|PPPP" );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Test four records with field delimiter in data.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 3, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) puts( "echo hello world\\|pg|BB|CCC" );
	(void) puts( "A|x == y \\|\\| y == z|BB" );
	(void) puts( "A|B|\\|" );
	(void) puts( "\\||\\|\\||\\|\\|\\|" );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Test incorrect number of fields per record.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 4, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) puts( "A|BB|CCC|DDDD" );
	(void) puts( "E|FF||HHHH" );
	(void) puts( "I|KKK|LLLL" );
	(void) puts( "M|NN|OOO|PPPP" );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 4, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) puts( "A|BB|CCC|DDDD" );
	(void) puts( "E|FF||HHHH" );
	(void) puts( "I|JJ|12345|KKK|LLLL" );
	(void) puts( "M|NN|OOO|PPPP" );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #5\n", stderr );
	(void) fputs( ">>> Test data record too large.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( call_cfg_read( field, (unsigned short) 4, &pid, fd ) == -1 ) {	
		(void) fputs( abort_msg, stderr );
		(void) fprintf( stderr, complete_msg, test_func );
		exit( 10 );
	}

	(void) memset( (void *) big_guy, 'X', sizeof( big_guy ) );
	big_guy[ sizeof( big_guy ) - 1 ] = (char) 0;
	(void) puts( big_guy );
	(void) fclose( stdout );
	(void) waitpid( pid, (int *) 0, 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #6\n", stderr );
	(void) fputs( ">>> Test fgets() error.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) close( 0 );
	rec_cnt = 0UL;
	(void) cfg_read( field, 1, &rec_cnt, ':', stdin );

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}

static int call_cfg_read( char **field, unsigned short num_of_fields, pid_t *pid, int *fd )
{
	FILE	*new_out, *fdopen();
	unsigned long	rec_cnt;

	/*
	** Run module from child process.
	*/
	if ( pipe( fd ) == -1 ) {
		perror( "pipe() failed" );
		return -1;
	}

	if ( ( *pid = fork() ) == (pid_t) -1 ) {
		perror( "fork() failed" );
		return -1;
	}

	if ( *pid > (pid_t) 0 ) {
		/*
		** Parent process.
		*/
		if ( ( new_out = fdopen( fd[ 1 ], "w" ) ) == (FILE *) 0 ) {
			perror( "fdopen() failed" );
		}
		stdout[ 0 ] = *new_out;
		free( new_out );
		return 0;
	}

	/*
	** Child process.
	*/
	(void) close( fd[ 1 ] );
	(void) close( 0 );

	if ( dup( fd[ 0 ] ) == -1 ) {
		perror( "fork() failed" );
		return -1;
	}

	rec_cnt = 0UL;

	while ( cfg_read( field, num_of_fields, &rec_cnt, '|', stdin ) == Read_ok ) {
		;
	}
	exit( 0 );
}
#endif
