/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: dfile_read_close.c,v 1.4 2009/10/16 16:24:23 keith Exp $";

/*
** $Log: dfile_read_close.c,v $
** Revision 1.4  2009/10/16 16:24:23  keith
** Needed to #ifdef thread variables.
**
** Revision 1.3  2009/10/16 07:35:52  keith
** Add GPL to source code.
**
** Revision 1.2  2009/05/21 22:04:18  keith
** Made changes to use DFILE_THREAD as compile time switch for threading.
**
** Revision 1.1  2009/02/28 20:14:45  keith
** Initial revision
**
*/

/*
** This function is used by user application to cleanup after calling
** dfile_read_open() and dfile_read().
*/

int dfile_read_close( dfile_t *dfile )
{
	static const char	func[] = "dfile_read_close";
	int	ret = 0;
#ifdef DFILE_THREAD
	dfile_buffer_t	*buffer;
	void	*thread_return;
#endif

	DEBUG_FUNC_START;

	if ( dfile == (dfile_t *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Function argument to dfile_read_close() cannot be null.\n", stderr );
		RETURN_INT( -1 );
	}

#ifdef DFILE_THREAD
	if ( dfile->buffer_cnt > (unsigned short)1 ) {
		buffer = dfile->application_buffer;
		if ( _dfile_buffer_wait( buffer, Dfile_parse_state, dfile->buffer_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		buffer->processing_complete_flag = 'Y';

		if ( _dfile_release_buffer( buffer ) == -1 ) {
			RETURN_INT( -1 );
		}

		ret = pthread_join( dfile->thread, &thread_return );
		if ( ret != 0 ) {
			thread_error( "pthread_join() failed", __FILE__, __LINE__, ret );
			RETURN_INT( -1 );
		}
	}
#endif

	if ( close( dfile->open_descriptor ) < 0 ) {
		UNIX_ERROR( "close() failed" );
		ret = -1;
	}

	if ( dfile->format == Dfile_zipped ) {
		assert( dfile->zstream != (void *)0 );
		if ( inflateEnd( (z_stream *)dfile->zstream ) != Z_OK ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to free GZIP stream.\n", stderr );
		}
	}

	_dfile_free( dfile );

	RETURN_INT( ret );
}
