/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_write_open_file.c,v 1.2 2009/10/16 07:35:57 keith Exp $";

/*
** $Log: _dfile_write_open_file.c,v $
** Revision 1.2  2009/10/16 07:35:57  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/28 18:04:19  keith
** Initial revision
**
*/

/*
** This function opens an output file. I/O buffers are allocated during
** the first call.
*/

int _dfile_write_open_file( dfile_t *dfile )
{
	static const char	func[] = "_dfile_write_open_file";
	int	fd, open_mode;
	const char	*fname;
	dfile_buffer_t	*buffer;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	fname = (const char *)dfile->file_name;
	assert( fname != (const char *)0 );

	buffer = dfile->buffer_base;
	assert( buffer != (dfile_buffer_t *)0 );

	switch ( dfile->open_mode ) {
	case Dfile_trunc:
		open_mode = O_TRUNC;
		break;
	case Dfile_append:
		open_mode = O_APPEND;
		break;
	default:
		(void) FPUT_SRC_CODE( stderr );
		(void) fputs( "Unknown file open mode (", stderr );
		(void) fput_int( dfile->open_mode, stderr );
		(void) fputs( ") encountered.\n", stderr );
		RETURN_INT( -1 );
	}

	if ( strcmp( fname, "-" ) == 0 ) {
		/*
		** Standard output.
		*/
		fd = 1;
	} else {
		fd = open( fname, O_WRONLY|O_CREAT|open_mode, (mode_t)0666 );
		if ( fd < 0 ) {
			UNIX_ERROR( "open() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to open file [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "] for writing.\n", stderr );
			dfile->error = Dfile_open_failed;
			RETURN_INT( -1 );
		}
	}

	dfile->open_descriptor = fd;

	if ( _dfile_allocate_buffer( dfile, fname ) == -1 ) {
		RETURN_INT( -1 );
	}

	do {
		assert( buffer != (dfile_buffer_t *)0 );
		buffer->buf_ptr = buffer->buffer_area;
		buffer->buf_ptr_end = buffer->buffer_end;
		buffer = buffer->next;
	} while ( buffer != dfile->buffer_base );

	dfile->error = Dfile_ok;
	dfile->file_rec_cnt = 0UL;

	if ( Debug ) {
		(void) fprintf( stderr, "opened file [%s]\n", fname );
	}

	RETURN_INT( 0 );
}
