/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_refill_ascii_buffer.c,v 1.2 2009/10/16 07:35:56 keith Exp $";

/*
** $Log: _dfile_refill_ascii_buffer.c,v $
** Revision 1.2  2009/10/16 07:35:56  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/27 15:55:33  keith
** Initial revision
**
*/

/*
** This function refills a buffer with ASCII data. Buffer is returned with only
** complete records. Buffer ending record fragment is copied to an overflow
** buffer. Overflow buffer is used when next buffer is filled. It is expected
** that no data records will exceed buffer size.
*/

int _dfile_refill_ascii_buffer( dfile_t *dfile )
{
	static const char	func[] = "_dfile_refill_ascii_buffer";
	size_t	overflow_len;
	dfile_buffer_t	*buffer;
	char	*buffer_area;
	ssize_t	read_byte_cnt;
	size_t	available_cnt, usable_cnt;
	int	ret;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	overflow_len = dfile->overflow_length;
	assert( dfile->buffer_length > (size_t)0 );
	assert( dfile->buffer_length >= overflow_len );
	available_cnt = dfile->buffer_length - overflow_len;
	buffer = dfile->io_buffer;
	assert( buffer != (dfile_buffer_t *)0 );
	buffer_area = buffer->buffer_area;
	assert( buffer_area != (char *)0 );

	/*
	** Copy overflow portion to start of buffer.
	*/
	(void) memcpy( (void *)buffer_area, dfile->overflow, overflow_len );

	read_byte_cnt = read( dfile->open_descriptor, &buffer_area[ overflow_len ], available_cnt );
	if ( read_byte_cnt < (ssize_t)0 ) {
		dfile->error = Dfile_io_failed;
		buffer->processing_complete_flag = 'Y';
		UNIX_ERROR( "read() failed" );
		RETURN_INT( -1 );
	}

	if ( read_byte_cnt == (ssize_t)0 ) {
		if ( overflow_len > (size_t)0 ) {
			dfile->error = Dfile_partial_record;
			buffer->processing_complete_flag = 'Y';
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Partial record found at end of file.\n", stderr );
			RETURN_INT( -1 );
		}
		dfile->error = Dfile_end_of_file;
		RETURN_INT( -1 );
	}

	usable_cnt = overflow_len + (size_t)read_byte_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, "refilled buffer: previous overflow %u + read %d = %u usable\n", overflow_len, read_byte_cnt, usable_cnt );
	}

	buffer->buf_ptr = buffer_area;

	ret = dfile->overflow_func( (const char **)&buffer->buf_ptr_end, buffer_area, usable_cnt, dfile->rec_attribute );

	if ( ret == -1 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Buffer not large enough (", stderr );
		(void) fput_uint( dfile->buffer_length, stderr );
		(void) fputs( " bytes) to contain entire record.\n", stderr );
		dfile->error = Dfile_partial_record;
		buffer->processing_complete_flag = 'Y';
		RETURN_INT( -1 );
	}

	assert( &buffer_area[ usable_cnt ] >= buffer->buf_ptr_end );
	overflow_len = (size_t)( &buffer_area[ usable_cnt ] - buffer->buf_ptr_end );

	if ( _dfile_resize_overflow( &dfile->overflow, &dfile->overflow_size, overflow_len ) == -1 ) {
		dfile->error = Dfile_io_failed;
		buffer->processing_complete_flag = 'Y';
		RETURN_INT( -1 );
	}

	(void) memcpy( (void *)dfile->overflow, (const void *)buffer->buf_ptr_end, overflow_len );

	if ( Debug ) {
		(void) fprintf( stderr, "Saving %u overflow bytes.\n", overflow_len );
		(void) fprintf( stderr, "buf_ptr = %p, buf_ptr_end = %p\n", buffer->buf_ptr, buffer->buf_ptr_end );
	}

	buffer->buffer_rec_cnt = 0UL;
	dfile->overflow_length = overflow_len;

	RETURN_INT( 0 );
}
