/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_allocate_buffer.c,v 1.3 2009/10/16 07:35:53 keith Exp $";

/*
** $Log: _dfile_allocate_buffer.c,v $
** Revision 1.3  2009/10/16 07:35:53  keith
** Add GPL to source code.
**
** Revision 1.2  2009/05/19 05:31:03  keith
** Make changes to reduce pointer arithmetic.
**
** Revision 1.1  2009/02/20 05:45:31  keith
** Initial revision
**
*/

/*
** This function finds the preferred block size for file system I/O and
** allocates I/O buffer memory.
*/

int _dfile_allocate_buffer( dfile_t *dfile, const char *fname )
{
	static const char	func[] = "_dfile_allocate_buffer";
	const size_t	default_block_length = 1024;
	const size_t	pad_size = 4;
	dfile_buffer_t	*buffer;
	struct stat	fs;
	char	*buffer_area;
	size_t	buffer_length;

	assert( dfile != (dfile_t *)0 );
	assert( fname != (const char *)0 );

	DEBUG_FUNC_START;

	if ( fstat( dfile->open_descriptor, &fs ) < 0 ) {
		UNIX_ERROR( "fstat() failed" );
		RETURN_INT( -1 );
	}

	if ( default_block_length > (size_t)fs.st_blksize ) {
		dfile->block_length = default_block_length;
	} else {
		dfile->block_length = (size_t)fs.st_blksize;
	}

	buffer_length = dfile->block_length * (size_t)dfile->buffer_block_cnt;
	dfile->buffer_length = buffer_length;

	if ( Debug ) {
		(void) fprintf( stderr, "I/O block length %u * block count %u = buffer length %u\n", dfile->block_length, dfile->buffer_block_cnt, dfile->buffer_length );
	}

	buffer = dfile->buffer_base;
	do {
		assert( buffer != (dfile_buffer_t *)0 );
		/*
		** Pad buffer because variable length record parsing
		** routine may null terminate string at end of buffer.
		*/
		buffer_area = (char *)malloc( buffer_length + pad_size );
		if ( buffer_area == (char *)0 ) {
			UNIX_ERROR( "malloc() failed" );
			RETURN_INT( -1 );
		}

		buffer->buffer_area = buffer_area;
		buffer->buffer_end = &buffer_area[ buffer_length ];

		buffer = buffer->next;
	} while ( buffer != dfile->buffer_base );

	RETURN_INT( 0 );
}
