/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "where.h"
#include "dfile_partition.h"

static const char       rcsid[] = "$Id: main.c,v 1.2 2009/10/16 20:16:47 keith Exp $";

/*
** $Log: main.c,v $
** Revision 1.2  2009/10/16 20:16:47  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/08 21:11:57  keith
** Initial revision
**
*/

/*
** This program divvys records into files that represent partition units.
*/
int main( int argc, char **argv )
{
	static const char	func[] = "main";
	const char	*input_dfile_name, *output_dfile_name;
	const char	**divvy_tbl, *filter_file_name, *output_partition_tag;
	const char	*range_partition_file;
	char	*partition_field_name;
	unsigned short	divvy_tbl_cnt, hash_slot_cnt;
	unsigned short	partition_ndx;
	dfile_t	*input_dfile, *output_dfile, **output_dfile_tbl;
	size_t	alloc_size;
	int	thread_input_flag, thread_output_flag;
	dfile_tag_t	*common_tag_tbl, *input_tag_tbl, *output_tag_tbl;
	unsigned short	common_tag_tbl_cnt;
	unsigned short	input_tag_tbl_cnt, output_tag_tbl_cnt;
	dfile_cfg_t	output_cfg;
	char	**key_ptr, err_msg[256];
	const unsigned short	blocks_per_buffer_cnt = 4;
	unsigned short	ndx, buffer_cnt;
	void	*condition;
	where_result_t	where_result;
	dfile_bind_t	*new_bind, *output_bind;
	unsigned long	hash;
	size_t	bind_size;

	if ( get_args( argc, argv, &input_dfile_name, &output_dfile_name, &thread_input_flag, &thread_output_flag, &common_tag_tbl, &common_tag_tbl_cnt, &input_tag_tbl, &input_tag_tbl_cnt, &output_tag_tbl, &output_tag_tbl_cnt, &output_partition_tag, &filter_file_name, &partition_field_name, &hash_slot_cnt, &range_partition_file ) == -1 ) {
		return 5;
	}

	DEBUG_FUNC_START;

	if ( open_input( &input_dfile, &key_ptr, common_tag_tbl, common_tag_tbl_cnt, input_tag_tbl, input_tag_tbl_cnt, thread_input_flag, input_dfile_name, partition_field_name ) == -1 ) {
		RETURN_INT( 10 );
	}

	if ( range_partition_file == (const char *)0 ) {
		divvy_tbl_cnt = hash_slot_cnt;

		if ( format_hash_values( &divvy_tbl, divvy_tbl_cnt ) == -1 ) {
			RETURN_INT( 15 );
		}
	} else {
		if ( load_range_partition_file( &divvy_tbl, &divvy_tbl_cnt, range_partition_file ) == -1 ) {
			RETURN_INT( 20 );
		}
	}

	assert( divvy_tbl != (const char **)0 );

	if ( dfile_cfg( &output_cfg, output_dfile_name ) == -1 ) {
		RETURN_INT( 25 );
	}

	/*
	** Allocate pointer table for output files.
	*/
	alloc_size = (size_t)divvy_tbl_cnt * sizeof( dfile_t * );
	output_dfile_tbl = (dfile_t **)malloc( alloc_size );
	if ( output_dfile_tbl == (dfile_t **)0 ) {
		unix_error( "malloc() failed", __FILE__, __LINE__ );
		RETURN_INT( 30 );
	}

	(void) memset( (void *)output_dfile_tbl, 0, alloc_size );

	buffer_cnt = ( thread_output_flag ) ? 3 : 1;

	if ( filter_file_name != (const char *)0 ) {
		if ( where_compile_file( &condition, err_msg, sizeof( err_msg ), filter_file_name, input_dfile ) != WHERE_NOERR ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "where_compile_file() failed [", stderr );
			(void) fputs( err_msg, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( 35 );
		}
	}

	if ( assign_output_tags( &output_tag_tbl, &output_tag_tbl_cnt, output_tag_tbl, output_tag_tbl_cnt, common_tag_tbl, common_tag_tbl_cnt, output_partition_tag ) == -1 ) {
		RETURN_INT( 40 );
	}

	output_bind = (dfile_bind_t *)0;
	bind_size = sizeof( dfile_bind_t ) * (size_t)output_cfg.bind_cnt;

	while ( dfile_read( input_dfile ) == 0 ) {
		if ( filter_file_name != (const char *)0 ) {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), condition ) != WHERE_NOERR ) {
				fput_src_code( __FILE__, __LINE__, stderr );
				(void) fputs( "where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				RETURN_INT( 45 );
			}
			if ( where_result == Where_result_false ) {
				/*
				** Skip record.
				*/
				continue;
			}
		}

		if ( range_partition_file == (const char *)0 ) {
			/*
			** hash partition
			*/
			hash = strhkey( *key_ptr );
			partition_ndx = (unsigned short)( hash % (unsigned long)divvy_tbl_cnt );
			if ( Debug ) {
				(void) fprintf( stderr, "hash partition_ndx = %hu\n", partition_ndx );
			}
		} else {
			/*
			** range partition
			*/
			if ( find_partition( &partition_ndx, *key_ptr, divvy_tbl, divvy_tbl_cnt ) == -1 ) {
				RETURN_INT( 50 );
			}
		}

		if ( output_dfile_tbl[ partition_ndx ] == (dfile_t *)0 ) {
			assert( partition_ndx < divvy_tbl_cnt );
			/*
			** There is an assumption the first tag is the
			** partition tag.
			*/
			assert( strcmp( output_tag_tbl->tag, output_partition_tag ) == 0 );
			output_tag_tbl->tag_value = divvy_tbl[ partition_ndx ];

			new_bind = (dfile_bind_t *)malloc( bind_size );
			if ( new_bind == (dfile_bind_t *)0 ) {
				unix_error( "malloc() failed", __FILE__, __LINE__ );
				RETURN_INT( 52 );
			}

			(void) memcpy( (void *)new_bind, (void *)output_cfg.bind, bind_size );

			output_dfile = dfile_write_open( &output_cfg, new_bind, output_cfg.bind_cnt, output_tag_tbl, output_tag_tbl_cnt, blocks_per_buffer_cnt, buffer_cnt, Dfile_trunc );

			if ( output_dfile == (dfile_t *)0 ) {
				RETURN_INT( 55 );
			}

			if ( output_bind == (dfile_bind_t *)0 ) {
				output_bind = output_dfile->bind;
				map_bind( output_bind, output_dfile->bind_cnt, input_dfile->bind_hash_table );
			} else {
				(void) memcpy( (void *)output_dfile->bind, (void *)output_bind, bind_size );
			}

			output_dfile_tbl[ partition_ndx ] = output_dfile;
		}


		if ( dfile_write( output_dfile_tbl[ partition_ndx ] ) == -1 ) {
			RETURN_INT( 60 );
		}
	}

	if ( input_dfile->error != Dfile_all_data_processed ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Failed to read all data.\n", stderr );
		RETURN_INT( 65 );
	}

	(void) dfile_read_close( input_dfile );

	for ( ndx = (unsigned short)0; ndx < divvy_tbl_cnt; ++ndx ) {
		if ( output_dfile_tbl[ ndx ] == (dfile_t *)0 ) {
			/*
			** Create zero length file.
			** There is an assumption the first tag is the
			** partition tag.
			*/
			assert( strcmp( output_tag_tbl->tag, output_partition_tag ) == 0 );
			output_tag_tbl->tag_value = divvy_tbl[ ndx ];

			output_dfile_tbl[ ndx ] = dfile_write_open( &output_cfg, (dfile_bind_t *)0, (unsigned short)0, output_tag_tbl, output_tag_tbl_cnt, (unsigned short)1, (unsigned short)1, Dfile_trunc );

			if ( output_dfile_tbl[ ndx ] == (dfile_t *)0 ) {
				RETURN_INT( 70 );
			}
		}

		/*
		** Do not free bind memory.
		*/
		output_dfile_tbl[ ndx ]->bind = (dfile_bind_t *)0;

		if ( dfile_write_close( output_dfile_tbl[ ndx ] ) == -1 ) {
			RETURN_INT( 75 );
		}
	}

#ifdef DEBUG
	debug_audit_heap();
#endif

	RETURN_INT( 0 );
}
