/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <sys/shm.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_join.h"

static const char       rcsid[] = "$Id: attach_shared_memory.c,v 1.2 2009/10/16 20:07:58 keith Exp $";

/*
** $Log: attach_shared_memory.c,v $
** Revision 1.2  2009/10/16 20:07:58  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/08 05:50:48  keith
** Initial revision
**
*/

/*
** This function attaches to a pre-loaded shared memory segment containing
** data from a join file.
*/
int attach_shared_memory( shm_t *shm, const char *ipc_key_str, unsigned short *join_key_ndx_tbl, unsigned short key_tbl_cnt )
{
	static const char	func[] = "attach_shared_memory";
	const size_t	*header, *offset;
	key_t	ipc_key;
	char	*end_ptr;
	const int	hex = 16;
	int	shmid;
	const void	*shmptr;
	size_t	alloc_size, field_ndx, rec_ndx, rec_cnt;
	const char	***key, **field_ptr, *data;

	assert( shm != (shm_t *)0 );
	assert( ipc_key_str != (const char *)0 );
	assert( join_key_ndx_tbl != (unsigned short *)0 );

	DEBUG_FUNC_START;

	ipc_key = (key_t)strtol( ipc_key_str, &end_ptr, hex );

	if ( end_ptr == ipc_key_str || *end_ptr != (char)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "strtol() failed to convert hex value [", stderr );
		(void) fputs( ipc_key_str, stderr );
		(void) fputs( "] to an integer.\n", stderr );
		RETURN_INT( -1 );
	}

	shmid = shmget( ipc_key, (size_t)0, 0 );
	if ( shmid == -1 ) {
		UNIX_ERROR( "shmget() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to find shared memory segment [", stderr );
		(void) fputs( ipc_key_str, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	shmptr = (const void *)shmat( shmid, (const void *)0, SHM_RDONLY | SHM_RND );

	if ( shmptr == (const void *)-1 ) {
		UNIX_ERROR( "shmat() failed" );
		RETURN_INT( -1 );
	}

	header = (const size_t *)shmptr;

	/*
	** Header contains an array of size_t values.
	**   1. record_cnt
	**   2. record_size
	**   3. field_cnt
	**   4. field 1 offset, field 2 offset, ..., field n offset
	*/
	rec_cnt = *header;
	shm->rec_cnt = rec_cnt;
	++header;
	shm->rec_size = *header;
	++header;

	if ( shm->field_cnt != *header ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Field count from dfile config [", stderr );
		(void) fput_uint( shm->field_cnt, stderr );
		(void) fputs( "] does not match shared memory field count [", stderr );
		(void) fput_uint( *header, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	++header;

	shm->field_offset = header;
	offset = header;
	header += shm->field_cnt;
	data = (const char *)header;
	shm->data = data;

	if ( Debug ) {
		(void) fprintf( stderr, "rec_cnt = %u, rec_size = %u, field_cnt = %u\n", shm->rec_cnt, shm->rec_size, shm->field_cnt );
		(void) fputs( "Field Offsets:\n", stderr );
		for ( field_ndx = (size_t)0; field_ndx < shm->field_cnt; ++field_ndx ) {
			(void) fprintf( stderr, "\t%u. %u\n", field_ndx + 1, offset[ field_ndx ] );
		}
	}

	alloc_size = rec_cnt * sizeof( const char ** );
	key = (const char ***)malloc( alloc_size );

	if ( key == (const char ***)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}


	shm->key = key;

	alloc_size = rec_cnt * key_tbl_cnt * sizeof( const char * );
	field_ptr = (const char **)malloc( alloc_size );

	if ( field_ptr == (const char **)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	for ( rec_ndx = (size_t)0; rec_ndx < rec_cnt; ++rec_ndx ) {
		*key = field_ptr;

		for ( field_ndx = (size_t)0; field_ndx < key_tbl_cnt; ++field_ndx ) {
			*field_ptr = data + offset[ join_key_ndx_tbl[ field_ndx ] ];
			++field_ptr;
		}

		data += shm->rec_size;
		++key;
	}

	alloc_size = (size_t)key_tbl_cnt * sizeof( const char * );
	shm->input_key_tbl = (const char **)malloc( alloc_size );

	if ( shm->input_key_tbl == (const char **)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
