/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "sexpr.h"
#include "dfile_utility.h"
#include "dfile_join.h"

static const char       rcsid[] = "$Id: assign_cmd_line_information.c,v 1.2 2009/10/16 20:07:58 keith Exp $";

/*
** $Log: assign_cmd_line_information.c,v $
** Revision 1.2  2009/10/16 20:07:58  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 18:00:28  keith
** Initial revision
**
*/

static int load_filter( sexpr_t **, const char * );

/*
** This function assigns information for command line arguments.
*/
int assign_cmd_line_information( input_ctl_t *input_ctl, join_ctl_t **join_ctl_tbl, unsigned short *join_ctl_tbl_cnt, output_ctl_t *output_ctl, const char *input_dfile_name, const char *key_argument, const char *data_field_argument, const char *input_filter_file_name, const char *output_filter_file_name, dfile_tag_t *dfile_tag_tbl, unsigned short dfile_tag_tbl_cnt, const char *output_dfile_name, const char *join_dfile_name, char inner_outer_join_flag, char unique_join, const char *outer_join_status_field, const char *join_filter_file_name, const char *ipc_key, int thread_input_flag, int thread_output_flag )
{
	static const char	func[] = "assign_cmd_line_information";
	int	result;
	join_ctl_t	*join_ctl;
	char	**field_tbl;
	unsigned long	ndx, field_tbl_cnt;

	assert( input_ctl != (input_ctl_t *)0 );
	assert( join_ctl_tbl != (join_ctl_t **)0 );
	assert( join_ctl_tbl_cnt != (unsigned short *)0 );
	assert( input_dfile_name != (const char *)0 );
	assert( key_argument != (const char *)0 );
	assert( output_dfile_name != (const char *)0 );
	assert( join_dfile_name != (const char *)0 );
	assert( inner_outer_join_flag == 'I' || inner_outer_join_flag == 'O' );
	assert( unique_join == (char)0 || unique_join == 'F' || unique_join == 'L' );

	DEBUG_FUNC_START;

	(void) memset( (void *)input_ctl, 0, sizeof( input_ctl_t ) );
	(void) memset( (void *)output_ctl, 0, sizeof( output_ctl_t ) );

	input_ctl->dfile_name = input_dfile_name;
	input_ctl->blocks_per_buffer_cnt = (unsigned short)2;
	if ( thread_input_flag ) {
		input_ctl->buffer_cnt = (unsigned short)3;
	} else {
		input_ctl->buffer_cnt = (unsigned short)1;
	}
	input_ctl->tag = dfile_tag_tbl;
	input_ctl->tag_cnt = dfile_tag_tbl_cnt;

	output_ctl->dfile_name = output_dfile_name;
	output_ctl->blocks_per_buffer_cnt = (unsigned short)2;
	if ( thread_output_flag ) {
		output_ctl->buffer_cnt = (unsigned short)3;
	} else {
		output_ctl->buffer_cnt = (unsigned short)1;
	}
	output_ctl->tag = dfile_tag_tbl;
	output_ctl->tag_cnt = dfile_tag_tbl_cnt;
	output_ctl->dfile_open_mode = Dfile_trunc;

	if ( input_filter_file_name != (const char *)0 ) {
		result = load_filter( &input_ctl->record_filter, input_filter_file_name );
		if ( result != 0 ) {
			RETURN_INT( -1 );
		}
	}

	if ( output_filter_file_name != (const char *)0 ) {
		result = load_filter( &output_ctl->record_filter, output_filter_file_name );
		if ( result != 0 ) {
			RETURN_INT( -1 );
		}
	}

	join_ctl = (join_ctl_t *)malloc( sizeof( join_ctl_t ) );

	if ( join_ctl == (join_ctl_t *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	(void) memset( (void *)join_ctl, 0, sizeof( join_ctl_t ) );

	join_ctl->dfile_name = join_dfile_name;
	join_ctl->blocks_per_buffer_cnt = (unsigned short)2;
	join_ctl->buffer_cnt = (unsigned short)1;
	join_ctl->tag = dfile_tag_tbl;
	join_ctl->tag_cnt = dfile_tag_tbl_cnt;
	join_ctl->unique_join_flag = unique_join;
	join_ctl->inner_outer_join_flag = inner_outer_join_flag;
	join_ctl->status_field = outer_join_status_field;
	join_ctl->ipc_key = ipc_key;

	/*
	** Set initial read flag to true.
	*/
	join_ctl->initial_read_flag = 1;

	/*
	** Parse key field names from command line argument.
	*/
	if ( parse_list( (const char ***)&field_tbl, &field_tbl_cnt, key_argument, ',' ) == -1 ) {
		RETURN_INT( -1 );
	}

	for ( ndx = 0UL; ndx < field_tbl_cnt; ++ndx ) {
		strtoupper( field_tbl[ ndx ] );
	}

	join_ctl->key_field_tbl = (const char **)field_tbl;
	join_ctl->key_field_tbl_cnt = (unsigned short)field_tbl_cnt;

	if ( data_field_argument != (const char *)0 ) {
		if ( parse_list( (const char ***)&field_tbl, &field_tbl_cnt, data_field_argument, ',' ) == -1 ) {
			RETURN_INT( -1 );
		}

		for ( ndx = 0UL; ndx < field_tbl_cnt; ++ndx ) {
			strtoupper( field_tbl[ ndx ] );
		}

		join_ctl->copy_field_tbl = (const char **)field_tbl;
		join_ctl->copy_field_tbl_cnt = field_tbl_cnt;
	}

	if ( join_filter_file_name != (const char *)0 ) {
		result = load_filter( &join_ctl->record_filter, join_filter_file_name );
		if ( result != 0 ) {
			RETURN_INT( -1 );
		}
	}

	*join_ctl_tbl = join_ctl;
	*join_ctl_tbl_cnt = (unsigned short)1;

	if ( Debug ) {
		dump_input_ctl( input_ctl );
		dump_join_ctl( *join_ctl_tbl, *join_ctl_tbl_cnt );
		dump_output_ctl( output_ctl );
	}

	RETURN_INT( 0 );
}

static int load_filter( sexpr_t **ret_sexpr, const char *file_name )
{
	sexpr_t	*sexpr;
	int	result;

	result = sexpr_load( &sexpr, file_name );

	if ( result != 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "failed to load s-expression record filter from file [", stderr );
		(void) fputs( file_name, stderr );
		(void) fputs( "]: [", stderr );
		(void) fputs( sexpr_error( result ), stderr );
		(void) fputs( "]\n", stderr );
		return -1;
	}

	*ret_sexpr = sexpr;

	return 0;
}
