/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: parse_exec_args.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: parse_exec_args.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 06:18:13  keith
** Initial revision
**
*/

/*
** This function parses command line arguments of application to be run.
** Results will later be used in execvp() call.
*/
int parse_exec_args( const char *exec_str, char ***args )

{
	static const char	func[] = "parse_exec_args";
	char	*end_ptr, *beg_ptr, *start_quote, *new_str, **new_args;
	unsigned short	arg_cnt;
	size_t	alloc_size;

	assert( exec_str != (const char *) 0 );
	assert( args != (char ***) 0 );

	DEBUG_FUNC_START;

	arg_cnt = (unsigned short)0;
	/*
	** Allocate enough for one argument and trailing null.
	*/
	alloc_size = sizeof( char ** ) * (size_t)2;
	*args = (char **)malloc( alloc_size );
	if ( *args == (char **)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	/*
	** Copy original command line.
	*/
	beg_ptr = strdup( exec_str );
	if ( beg_ptr == (char *)0 ) {
		UNIX_ERROR( "strdup() failed" );
		RETURN_INT( -1 );
	}

	/*
	** Loop through each argument in command line.
	*/
	while ( ( end_ptr = strpbrk( beg_ptr, "\t '\"" ) ) != (char *) 0 ) {
		/*
		** Use switch for handling special characters used
		** in command lines.
		*/
		switch ( *end_ptr ) {
		case ' ':
		case '\t':
			if ( beg_ptr == end_ptr ) {
				/*
				** Extra space encountered that was
				** not needed to delimit command line
				** arguments--discard.
				*/
				++beg_ptr;
				continue;
			}
			break;
		case '\'':
		case '"':
			/*
			** Find matching quote or tick mark.
			** start_quote points to first character
			** after quote mark.
			*/
			start_quote = &end_ptr[ 1 ];
			do {
				if ( ( end_ptr = strchr( start_quote, (int) *end_ptr ) ) == (char *) 0 ) {
					/*
					** No matching mark.
					*/
					FPUT_SRC_CODE( stderr );
					(void) fputs( "No matching tick or quote mark found for execute field in schedule table.\n", stderr );
					RETURN_INT( -1 );
				}

				/*
				** Don't break out of loop if quote mark was
				** escaped with a back slash.
				*/
			} while ( end_ptr[ -1 ] == '\\' );

			/*
			** Increment beg_ptr to first character after
			** beginning quote or tick mark.
			*/
			++beg_ptr;
		}

		/*
		** Replace command line argument delimiter with string
		** terminating null character.
		*/
		*end_ptr = (char) 0;

		/*
		** Increase arg_cnt by two to include last agrument and
		** terminating null pointer.
		*/
		alloc_size = sizeof( char ** ) * ( arg_cnt + (size_t)3 );
		new_args = (char **)realloc( *args, alloc_size );
		if ( new_args == (char **)0 ) {
			UNIX_ERROR( "realloc() failed" );
			RETURN_INT( -1 );
		}

		*args = new_args;

		/*
		** Copy freshly parsed command line argument and
		** append it to args array.
		*/
		new_str = strdup( beg_ptr );
		if ( new_str == (char *) 0 ) {
			UNIX_ERROR( "strdup() failed" );
			RETURN_INT( -1 );
		}

		( *args )[ arg_cnt ] = new_str;
		++arg_cnt;

		/*
		** Make beg_ptr point to first character of next argument
		** to parse.
		*/
		beg_ptr = &end_ptr[ 1 ];
	}

	if ( *beg_ptr != (char) 0 ) {
		/*
		** Copy last parsed argument and append it to args array.
		*/
		new_str = strdup( beg_ptr );
		if ( new_str == (char *) 0 ) {
			UNIX_ERROR( "strdup() failed" );
			RETURN_INT( -1 );
		}
		( *args )[ arg_cnt ] = new_str;
		++arg_cnt;
	}

	/*
	** Null terminate args array.
	*/
	( *args )[ arg_cnt ] = (char *)0;

	if ( Debug ) {
		(void) fputs( "Command Arguments:\n", stderr );
		for ( arg_cnt = (unsigned short)0; ( *args )[ arg_cnt ] != (char *)0; ++arg_cnt ) {
			(void) fprintf( stderr, "%hu. [%s]\n", arg_cnt+1, ( *args )[ arg_cnt ] );
		}

	}

	RETURN_INT( 0 );
}
