/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: assign_fname.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: assign_fname.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/06 16:20:56  keith
** Initial revision
**
*/

/*
** This function assigns file names to capture stdout and stderr
** of an application about to be started. Stdout and stderr will be in format:
**    <prefix file path>.<sequence>.out
**    <prefix file path>.<sequence>.err
**
** Prefix file path is passed in as a command line argument. Sequence is
** is the slice number about to be processed.
*/
int assign_fname( const char **stdout_fname, const char **stderr_fname, const char *output_name, unsigned short job_ndx, unsigned short max_job_ndx, const char *slice_text )
{
	static const char	func[] = "assign_fname";
	static const char	period[] = ".";
	static char	stdout_area[PATH_MAX+1];
	static char	stderr_area[PATH_MAX+1];
	const	size_t	max_fname_len = PATH_MAX;
	size_t	len;
	char	sequence[24];
	int	seq_nbr_len;

	assert( stdout_fname != (const char **)0 );
	assert( stderr_fname != (const char **)0 );
	/*
	** output_name may be null.
	*/

	DEBUG_FUNC_START;

	if ( output_name == (const char *)0 ) {
		/*
		** Allow child process to inherit
		** parent's stdout and stderr.
		*/
		*stdout_fname = (const char *)0;
		*stderr_fname = (const char *)0;

		RETURN_INT( 0 );
	}

	/*
	** Figure max sequence number length.
	*/
	if ( max_job_ndx > (unsigned short)10000 ) {
		seq_nbr_len = 5;
	} else {
		if ( max_job_ndx > (unsigned short)1000 ) {
			seq_nbr_len = 4;
		} else {
			if ( max_job_ndx > (unsigned short)100 ) {
				seq_nbr_len = 3;
			} else {
				if ( max_job_ndx > (unsigned short)10 ) {
					seq_nbr_len = 2;
				} else {
					seq_nbr_len = 1;
				}
			}
		}
	}

	if ( snprintf( sequence, sizeof( sequence ), "%0*hu", seq_nbr_len, job_ndx ) <= 0 ) {
		UNIX_ERROR( "sprintf() failed to convert job_ndx to ASCII" );
		RETURN_INT( -1 );
	}

	/*
	** 20 wasn't derived as an exact figure. It's just some
	** additional cushion space for suffix '.out' and '.err'.
	*/
	len = strlen( output_name ) + strlen( sequence ) + (size_t)20;

	if ( len > max_fname_len ) {
		(void) fputs( __FILE__, stderr );
		(void) fprintf( stderr, "(%d)", __LINE__ );
		(void) fputs( ": Output file name using [", stderr );
		(void) fputs( output_name, stderr );
		(void) fputs( "] will exceed maximum length ", stderr );
		(void) fprintf( stderr, "%u", max_fname_len );
		(void) fputs( ".\n", stderr );
		RETURN_INT( -1 );
	}

	(void) strcpy( stdout_area, output_name );
	(void) strcat( stdout_area, period );

	if ( substitute_tokens( stdout_area, max_fname_len, sequence, slice_text ) == -1 ) {
		RETURN_INT( -1 );
	}

	(void) strcpy( stderr_area, stdout_area );

	(void) strcat( stdout_area, "out" );
	(void) strcat( stderr_area, "err" );


	*stdout_fname = stdout_area;
	*stderr_fname = stderr_area;

	if ( Debug ) {
		(void) fprintf( stderr, "stdout [%s]\n", *stdout_fname );
		(void) fprintf( stderr, "stderr [%s]\n", *stderr_fname );
	}

	RETURN_INT( 0 );
}
