/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "where.h"
#include "dfile_agfunc.h"

static const char       rcsid[] = "$Id: output_record.c,v 1.2 2009/10/16 19:43:49 keith Exp $";

/*
** $Log: output_record.c,v $
** Revision 1.2  2009/10/16 19:43:49  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/02 05:02:08  keith
** Initial revision
**
*/

/*
** This function processes an output record. It involves aggregate function
** processing, record filtering, and writing the output record.
*/
int output_record( dfile_t *dfile, unsigned long rec_cnt, func_t *func_tbl, unsigned short func_tbl_cnt, void *output_filter )
{
	static const char	func[] = "output_record";
	unsigned short	ndx;
	where_result_t	where_result;
	char	err_msg[ 256 ];

	assert( dfile != (dfile_t *)0 );
	assert( rec_cnt > 0UL );
	assert( func_tbl != (func_t *)0 );
	assert( func_tbl_cnt > (unsigned short)0 );

	DEBUG_FUNC_START;

	for ( ndx = (unsigned short)0; ndx < func_tbl_cnt; ++ndx ) {
		if ( eval_agfunc( &func_tbl[ ndx ], rec_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	if ( output_filter == (void *)0 ) {
		/*
		** No output filter.
		*/
		if ( dfile_write( dfile ) == -1 ) {
			RETURN_INT( -1 );
		}
	} else {
		if ( where_condition( &where_result, err_msg, sizeof( err_msg ), output_filter ) != WHERE_NOERR ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "output where_condition() failed [", stderr );
			(void) fputs( err_msg, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( -1 );
		}

		if ( where_result == Where_result_true ) {
			if ( dfile_write( dfile ) == -1 ) {
				RETURN_INT( -1 );
			}
		}
	}

	for ( ndx = (unsigned short)0; ndx < func_tbl_cnt; ++ndx ) {
		/*
		** Set initialize flag to true.
		*/
		func_tbl[ ndx ].initialize_flag = 1;

		switch( func_tbl[ ndx ].func_type ) {
		case Str_min:
		case Str_max:
			free( (void *)func_tbl[ ndx ].value_u.str );
		}
	}

	RETURN_INT( 0 );
}
