#!/bin/bash

# Call 'dar' for a complete backup every month and incremental ones
# each day

# Copyright (C) 2005  Cliss XXI
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.
#
# Author: Sylvain Beucler <beuc@beuc.net>


# ToDo:
# - rewrite in Perl/Python and handle spaces in the directories
# - if a backup is ran twice, do not create an incremental backup on
# itself, but either use the previous day's backup as reference or
# create a distinct, new backup for the current day without
# overwriting the current one
# - when overwriting last year's backup, find a way to make it clear
# which backups are from the current year, and which ones are not; -
# maybe switch from overwriting to logrotate'ing
# - if a different root_dir is saved to the same backup_dir, do
# something (issue a warning, adopt a new naming convention, etc) -
# but do not keep quiet

# Others products to have a look to:
# rsync-based:
# - http://colas.nahaboo.net/software/rsync-incr/
# - http://www.nongnu.org/duplicity/
# tar-based:
# - http://www.backup-manager.org/
# client-server:
# - http://sourceforge.net/projects/afbackup/
# - http://www.amanda.org/
# Others:
# http://directory.fsf.org/search/fsd-search.py?q=backup
# http://www.linuxlinks.com/Software/Backup/

function launch_dar() {
    incremental=$1
    incremental_opt=''
    if [ $incremental -eq 1 ]; then
	incremental_opt="--ref $backup_dir/prev_dar_cat"
    fi

    # If the backup dir under the root directory that we'll save,
    # exclude it
    prune_backup_opt=""
    if [ "${backup_dir#$root_dir}" != "$backup_dir" ]; then
	prune_backup_opt="--prune ${backup_dir#$root_dir}"
    fi

    dar -Q --noconf --no-warn \
	--on-fly-isolate "$backup_dir/cur_dar_cat" $incremental_opt \
	--fs-root $root_dir --prune proc --prune sys $prune_backup_opt \
	--empty-dir \
	-z6 -Z '.zip' -Z '*.gz' -Z '*.bz2' -Z '*.jpg' -Z '*.png' \
	-Z '*.mp3' -Z '*.mpg' \
	-s 1G -E 'par2create -qq -r5 %p/%b.%n.%e' \
	--create "$backup_dir/$month/$day_of_month" \
	> /dev/null
}


# What we backup:
root_dir=$1 
# Where we backup:
backup_dir=${2%%/} # Strip trailing slashes;
# no spaces allowed in the backup dir btw


# Usage
if [ -z "$root_dir" -o -z "$backup_dir" ]; then
    echo "Usage: $0 what_dir_to_backup where_to_backup"
    exit 1
fi

if [ ! -e "$root_dir" ]; then
    echo "The directory to backup ($root_dir) does not exist."
    exit 1
fi
if [ ! -e "$backup_dir" ]; then
    echo "The backup/destination directory ($backup_dir) does not exist."
    exit 1
fi
    


# When are we?
day_of_month=`date +%d`
month=`LC_ALL=C LANGUAGE=C LANG=C date +%b`
year=`date +%d`


# Clean-up (maybe there're still files from last year)
mkdir -p "$backup_dir/$month" 2> /dev/null
rm "$backup_dir/$month/$day_of_month*" 2> /dev/null


# Full backup each 1st day of the month
incremental=1
if [ $day_of_month -eq 1 -o ! -e $backup_dir/prev_dar_cat.1.dar ]; then
    $incremental=0
fi

# Launch the backup
launch_dar $incremental

# Verify the backup
dar -t $backup_dir/$month/$day_of_month > /dev/null

# Place the catalogue for the next call
mv "$backup_dir/cur_dar_cat.1.dar" "$backup_dir/prev_dar_cat.1.dar"
