<?php
/******************************************************************************
  daDrill, an offline BibTeX library browser
  Copyright (C) 2005  Sylvain Hall
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  MA  02110-1301, USA.
******************************************************************************/

// Imports
require_once("db-connection.lib.php");  // Connection info to the database
require_once("encode-decode.lib.php");  // Encoding/decoding of strings

/**
 * Constants for presentation styles in paper lists
 */
define ("PRESENTATION_REGULAR", 0);     // Regular: authors, year, title
define ("PRESENTATION_ABRIDGED", 1);    // Abridged: year, title
define ("PRESENTATION_ABRIDGED2", 2);   // Abridged: authors, title
define ("PRESENTATION_BIBTEX", 3);      // Preformatted BibTeX block
define ("PRESENTATION_CLEARTEXT", 4);   // No markup in reference


/**
 * Strips opening and closing accolades from a string
 */
function strip_accolades($s) // {{{
{
  $s = str_replace("{", "", $s);
  $s = str_replace("}", "", $s);
  return $s;
} // }}}

/**
 * Formats a paper entry according to a presentation style.  NOTE:  the "m" and
 * "n" comments around the title are for the daDrill Mozilla search plugin;
 * leave them there!
 * @param $db A valid link to a connected database
 * @param $a An associative array containing the database parameters of the 
 *   paper
 * @param $style 0 for the moment
 * @param $prefix A prefix to append to every field to retrieve
 * @return An HTML string of the formatted paper entry
 */
function format_paper_entry($db, $a, $style=PRESENTATION_REGULAR, $prefix="") // {{{
{
  $out = "";
  switch ($style)
  {
    case PRESENTATION_CLEARTEXT: // {{{
    {
      $out = "";
      if ($a["bibtexAuthor"] && $a["bibtexAuthor"] != "NULL") // Shows authors if not null
        $out .= format_author_list($db, db_decode($a["bibtexAuthor"]), PRESENTATION_CLEARTEXT);
      else if ($a["bibtexEditor"] && $a["bibtexEditor"] != "NULL")  // Otherwise shows editors
        $out .= format_author_list($db, db_decode($a["bibtexEditor"]), PRESENTATION_CLEARTEXT);
      if ($a["bibtexYear"] && $a["bibtexYear"] != "NULL") // Shows year if not null
        $out = $out."(".htmlentities(db_decode($a["bibtexYear"]))."). ";
      if ($a["bibtexTitle"] && $a["bibtexTitle"] != "NULL") // Shows title if not null
        $out .= strip_accolades(db_decode($a["bibtexTitle"]));
      else if ($a["bibtexBooktitle"] && $a["bibtexBooktitle"] != "NULL") // Otherwise shows book title
        $out .= strip_accolades(db_decode($a["bibtexBooktitle"]));
      break;
    }
    // }}}
    case PRESENTATION_ABRIDGED2: // {{{
    {
      $out = "<span class=\"reference\">\n";
      if ($a["bibtexAuthor"] && $a["bibtexAuthor"] != "NULL") // Shows authors if not null
        $out = $out."<span class=\"author\">".format_author_list($db, db_decode($a["bibtexAuthor"]))."</span>.\n";
      else if ($a["bibtexEditor"] && $a["bibtexEditor"] != "NULL")  // Otherwise shows editors
        $out = $out."<span class=\"author\">".format_author_list($db, db_decode($a["bibtexEditor"]))."</span>.\n";
      if ($a["bibtexTitle"] && $a["bibtexTitle"] != "NULL") // Shows title if not null
        $out = $out."<!--m--><a class=\"title\" href=\"paper.php?key=".$a["key"]."\">".htmlentities(strip_accolades(db_decode($a["bibtexTitle"])))."</a><!--n-->.\n";
      else if ($a["bibtexBooktitle"] && $a["bibtexBooktitle"] != "NULL") // Otherwise shows book title
        $out = $out."<!--m--><a class=\"title\" href=\"paper.php?key=".$a["key"]."\">".htmlentities(strip_accolades(db_decode($a["bibtexBooktitle"])))."</a><!--n-->.\n";
      $out = $out."</span>\n\n";
      break;
    }
    // }}}
    case PRESENTATION_BIBTEX: // {{{
    {
      $out = "@".$a["typeBibtex"]."{".$a["identifierBibtex"].",\n";
      if ($a["bibtexTitle"] != "")
      {
        $out = $out."  title       = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexTitle"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexAuthor"] != "")
      {
        $out = $out."  author      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexAuthor"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexYear"] != "")
      {
        $out = $out."  year        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexYear"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexCrossref"] != "")
      {
        $out = $out."  crossref    = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexCrossref"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexPages"] != "")
      {
        $out = $out."  pages       = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexPages"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexAddress"] != "")
      {
        $out = $out."  address     = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexAddress"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexAffiliation"] != "")
      {
        $out = $out."  affiliation = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexAffiliation"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexAnnote"] != "")
      {
        $out = $out."  annote      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexAnnote"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexBooktitle"] != "")
      {
        $out = $out."  booktitle   = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexBooktitle"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexChapter"] != "")
      {
        $out = $out."  chapter     = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexChapter"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexCopyright"] != "")
      {
        $out = $out."  copyright   = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexCopyright"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexEditor"] != "")
      {
        $out = $out."  editor      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexEditor"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexEdition"] != "")
      {
        $out = $out."  edition     = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexEdition"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexEe"] != "")
      {
        $out = $out."  ee          = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexEe"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexHowpublished"] != "")
      {
        $out = $out."  howpublished= {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexHowpublished"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexInstitution"] != "")
      {
        $out = $out."  institution = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexInstitution"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexIsbn"] != "")
      {
        $out = $out."  isbn        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexIsbn"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexIssn"] != "")
      {
        $out = $out."  issn        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexIssn"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexJournal"] != "")
      {
        $out = $out."  journal     = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexJournal"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexKey"] != "")
      {
        $out = $out."  key         = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexKey"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexKeywords"] != "")
      {
        $out = $out."  keywords    = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexKeywords"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexLanguage"] != "")
      {
        $out = $out."  language    = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexLanguage"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexLccn"] != "")
      {
        $out = $out."  lccn        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexLccn"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexLocation"] != "")
      {
        $out = $out."  location    = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexLocation"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexMonth"] != "")
      {
        $out = $out."  month       = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexMonth"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexNumber"] != "")
      {
        $out = $out."  number      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexNumber"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexOrganization"] != "")
      {
        $out = $out."  organization= {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexOrganization"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexPrice"] != "")
      {
        $out = $out."  price       = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexPrice"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexPublisher"] != "")
      {
        $out = $out."  publisher   = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexPublisher"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexSchool"] != "")
      {
        $out = $out."  school      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexSchool"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexSeries"] != "")
      {
        $out = $out."  series      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexSeries"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexSize"] != "")
      {
        $out = $out."  size        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexSize"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexType"] != "")
      {
        $out = $out."  type        = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexType"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexVolume"] != "")
      {
        $out = $out."  volume      = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexVolume"])), 50, "\n"))."},\n";
      }
      if ($a["bibtexUrl"] != "")
      {
        $out = $out."  url         = {".str_replace("\n", "\n                 ",
          wordwrap(utf2latex(db_decode($a["bibtexUrl"])), 50, "\n"))."},\n";
      }
      
      // Strips out last comma
      $out = substr($out, 0, strlen($out) - 2);
      $out = $out."\n}";      
      break;
    } // }}}
    default:
    {
      $out = "<div class=\"reference\">\n";
      if ($a["hasSummary"])
        $out .= "<a href=\"".paper_url($a["key"])."#comment-".$a["hasSummary"]."\"><div class=\"summary-available\" title=\""._("Reading summary")."\"><span class=\"text\">Summary available</span></div></a>";
      else
        $out .= "<div class=\"no-summary-available\"></div>";
      if ($a["bibtexAuthor"] && $a["bibtexAuthor"] != "NULL") // Shows authors if not null
        $out .= "<span class=\"author\">".format_author_list($db, db_decode($a["bibtexAuthor"]))."</span>.\n";
      else if ($a["bibtexEditor"] && $a["bibtexEditor"] != "NULL")  // Otherwise shows editors
        $out .= "<span class=\"author\">".format_author_list($db, db_decode($a["bibtexEditor"]))."</span>.\n";
      if ($a["bibtexYear"] && $a["bibtexYear"] != "NULL") // Shows year if not null
        $out .= "(<a href=\"specific-query.php?year=".htmlentities(db_decode($a["bibtexYear"]))."\" class=\"year\">".db_decode($a["bibtexYear"])."</a>).\n";
      if ($a["bibtexTitle"] && $a["bibtexTitle"] != "NULL") // Shows title if not null
        $out .= "<!--m--><a class=\"title\" href=\"".paper_url($a["key"])."\">".htmlentities(strip_accolades(db_decode($a["bibtexTitle"])))."</a><!--n-->.\n";
      else if ($a["bibtexBooktitle"] && $a["bibtexBooktitle"] != "NULL") // Otherwise shows book title
        $out .= "<!--m--><a class=\"title\" href=\"".paper_url($a["key"])."\">".htmlentities(strip_accolades(db_decode($a["bibtexBooktitle"])))."</a><!--n-->.\n";
      $out .= "</div>\n\n";      
      break;
    }
  }
  return $out;
} // }}}

/**
 * Process author list and back-references authors listed in the database
 * @param $db A valid link to a connected database
 * @param $l The string of the author list.  It is supposed to be decoded (i.e.
 *  the MySQL escape sequences are removed.
 * @return The HTML string with links for authors in the database
 */
function format_author_list($db, $l, $style = '') // {{{
{
  global $config;
  
  if (!$db)
  {
    die("Invalid database link");
  }
  $out = "";
  $authors = explode(" and ", $l);
  for ($i = 0; $i < count($authors); $i++)
  {
    switch ($style)
    {
      case PRESENTATION_CLEARTEXT:
        $name_parts = explode_author_name($authors[$i]);
        // No htmlentities
        // If more than 2 authors, shows et al.
        if ($i < $config["maxAuthorsEtAl"])
          $out .= trim($name_parts[2]).", ";
        if ($i == $config["maxAuthorsEtAl"])
          $out .= "et al, ";
        break;
      default:
        $result = db_query($db, "SELECT * FROM ".$config["ddDBPrefix"]."author WHERE TRIM(CONCAT(IF(firstName <> '', CONCAT(firstName, ' '), ''), IF(vonPart <> '', CONCAT(vonPart, ' '), ''), IF(lastName <> '', CONCAT(lastName, ' '), ''), IF(jrPart <> '', CONCAT(jrPart, ' '), ''))) = '".db_encode(trim($authors[$i]))."';");
        if (!$result)
          echo db_error($db);
        $row2 = db_fetch_assoc_array($result);
        if ($row2)
        {
          // This author is in the database
          //$out = $out."<a href=\"".author_url($row2["authorKey"])."\">".htmlentities(trim($authors[$i]))."</a>, ";
          $out = $out."<a href=\"".author_url(trim($authors[$i]))."\">".htmlentities(trim($authors[$i]))."</a>, ";
        }
        else
        {
          // This author is not in the database
          $out = $out.htmlentities(trim($authors[$i])).", ";
        }
        break;
    }
  }
  // Trims last comma and space FROM ".$config["ddDBPrefix"]."author list
  $out = substr($out, 0, strlen($out) - 2);
  if ($style == PRESENTATION_CLEARTEXT)
  {
    $out .= ". ";
  }
  return $out;
 } // }}}

/**
 * Explodes an author name into its first name, von part, last name and jr. part
 * according to BibTeX's rules
 * @param $s The string of the author's name. It is supposed to be decoded (i.e.
 *  the MySQL escape sequences are removed.
 * @return An array with 4 indexes, numbered 0-3, corresponding to the 4 parts
 *   of the name
 */
function explode_author_name($name) // {{{
{
  $n = trim($name);
  $matches = array();
  $exploded = array();
  $exploded[0] = $exploded[1] = $exploded[2] = $exploded[3] = "";
  $vons = "/ von der | van der | von | van | du | de | de la /";
  $jrs = "/ Jr\\.| Sr\\.| II| III/";
  if (preg_match($vons, $n, $matches))
  {
    // There is a von part; add it and remove it from the name
    $n = str_replace($matches[0], " ", $n);
    $exploded[1] = trim($matches[0]);
  }
  $n = trim($n);
  if (preg_match($jrs, $n, $matches))
  {
    // There is a Jr. part; add it and remove it from the name
    $n = str_replace($matches[0], " ", $n);
    $exploded[3] = trim($matches[0]);
  }
  // The last remaining token is the last name, the rest is the first name
  $n = trim($n);
  $exp = array();
  $exp = explode(" ", $n);
  if (count($exp) > 1)
  {
    $exploded[2] = trim($exp[count($exp) - 1]);
    $n = str_replace($exp[count($exp) - 1], "", $n);
    $exploded[0] = trim($n);
  }
  else
  {
    // If only one token remains, we assume it is the last name
    $exploded[2] = trim($n);
  }
  return $exploded;
} // }}}

/**
 * Rebuilds the author name from the exploded list of all 4 parts
 * @param $first First name
 * @param $von Von part
 * @param $last Last name
 * @param $jr Jr. part
 * @return A string with the full name, without extra spaces
 */
function format_author_name($first, $von, $last, $jr) // {{{
{
  $name = "";
  if (trim($first) != "")
    $name = $name.$first." ";
  if (trim($von) != "")
    $name = $name.$von." ";
  if (trim($last) != "")
    $name = $name.$last." ";
  if (trim($jr) != "")
    $name = $name.$jr." ";
  return trim($name);
} // }}}

// :folding=explicit:
?>
