<?php
/******************************************************************************
  daDrill, an offline BibTeX library browser
  Copyright (C) 2005  Sylvain Hall
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
  MA  02110-1301, USA.
******************************************************************************/

/*
  Determines the co-authors of an author.
  Also determines the "friends":  the people that this author cites.
*/

require_once ("lib/init.lib.php");
require_once ("lib/GraphViz.php");

// Verifies if a GET request has been sent
if (!$_GET["key"] && !isset($_POST))
{
  die("No paper key passed to the page.");
}

/**
 * Maximum depth for co-author graph (you can go up to 7 or 8 without
 * any problem).  Set it to 0 to disable this functionality.
 */
define("COAUTHORS_DEPTH", 6);

/**
 * Maximum depth for co-citers graph.  This computes much more slowly and
 * produces a very large graph, so keep it to small values (1 or 2), if you
 * want it at all.  Set it to 0 to disable this functionality.
 */
define("CITEES_DEPTH", 0);

/**
 * Whether or not to exclude papers with type "misc" or "techreport" from the
 * graph.
 */
define("EXCLUDE_MISC", true);
define("EXCLUDE_TECHREPORT", false);

/**
 * Other constants for graph drawing
 */
$show_labels = false;     // Whether or not to show labels in nodes
$graph_font_size = 8;             // The size of the font to display the graph
$graph_font_family = "Tahoma";    // The font name to display the graph
$graph_fill_starting = "#ffff00"; // Color of the "start" nodes of the search
$graph_fill_coauthor_immediate = "#ffff66";  // Color of the immediate co-authors
$graph_fill_coauthor_indirect = "#ffffaa";   // Color of the indirect co-authors
$graph_fill_citee_immediate = "#ffffff";  // Color of the immediate cited people
$graph_fill_citee_indirect = "#ffffff";   // Color of the indirectly cited people

// Arrays used for the search algorithm (later)
$nodes = array();
$new_nodes = array();
$depths = array();
$common_papers = array();
$rev_nodes = array();

// Create a new graph instance
$graph2 = new Image_GraphViz(false);

// Get the authors that will be used as a "seed" for the search
if (isset($_GET["key"]))
{
  $select_request = "authorKey = ".$_GET["key"]." OR ";
}
if (isset($_POST["num-authors"]))
{
  for ($i = 0; $i < $_POST["num-authors"]; $i++)
  {
    if (isset($_POST["seed-author-$i"]) && $_POST["seed-author-$i"] == "on")
    {
      $select_request .= "authorKey = $i OR ";
    }
  }
}

// Get info from these authors
$query_string = "SELECT * from ".$config["ddDBPrefix"]."author WHERE ".$select_request."FALSE;";
$result = db_query($db, $query_string);
if (!$result)
{
  echo _("No result; maybe key does not exist?  See below if MySQL says something.");
  echo db_error($db);
  exit();
}
while($row = db_fetch_assoc_array($result))
{
  $author_firstname = $row["firstName"];
  $author_vonpart = $row["vonPart"];
  $author_lastname = $row["lastName"];
  $author_jrpart = $row["jrPart"];
  $author_fullname = $author_firstname
    .($author_vonpart != "" ? " ".$author_vonpart : "")
    .($author_lastname != "" ? " ".$author_lastname : "")
    .($author_jrpart != "" ? " ".$author_jrpart : "");
  // Adds the author in question in the graph
  $graph2->addNode(
    $row["authorKey"], // Node name
    array(
      "URL" => author_url($row["authorKey"]),
      "label" => ($show_labels ? unaccent($author_fullname) : ""),
      "shape" => "box",
      "height" => ($show_labels ? "" : ".3"),
      "width" => ($show_labels ? "" : ".3"),
      "style" => "filled",
      "fontsize" => $graph_font_size,
      "fillcolor" => $graph_fill_starting
    ));
  $new_nodes[$author_fullname] = $row["authorKey"];
}

// Multiple passes in the DB
$future_new_nodes = array();
$upper_bound = max(COAUTHORS_DEPTH, CITEES_DEPTH);
for ($depth = 1; $depth <= $upper_bound && count($new_nodes) > 0; $depth++)
{
  foreach ($new_nodes as $authorname => $key)
  {
    // Adds this author to the processed authors
    $nodes[$authorname] = $key;
    $rev_nodes[$key] = $authorname;
    
    // Finds co-authors
    if ($depth < COAUTHORS_DEPTH)
    {
      list($coauthors, $coauthors_keys) = getCoAuthors($authorname, array("exclude-misc" => EXCLUDE_MISC, "exclude-techreport" => EXCLUDE_TECHREPORT));
      $toadd_ck = $coauthors_keys;
      // Add all co-authors to be considered as candidates in the next cycle
      foreach ($toadd_ck as $new_name => $cok)
        $future_new_nodes[$new_name] = $cok;
      foreach ($coauthors as $new_name => $collaborations)
      {
        if (!isset($nodes[$new_name]) && !isset($new_nodes[$new_name]))
        {
          // This co-author is not already in the graph:  add it
          $graph2->addNode(
            $coauthors_keys[$new_name],
            array(
              "URL"   => author_url($coauthors_keys[$new_name]),
              "label" => ($show_labels ? unaccent($new_name).": ".$depth : ""),
              "shape" => ($show_labels ? "oval" : "circle"),
              "height" => ($show_labels ? "" : ".3"),
              "width" => ($show_labels ? "" : ".3"),
              "style" => "filled",
              "fontsize" => $graph_font_size,
              "fillcolor" => gradient($depth, 1, 6, array("style" => "linear", "start" => "#DDFF00", "finish" => "#008800"))
              //"fillcolor" => ($depth === 2 ? $graph_fill_coauthor_immediate : $graph_fill_coauthor_indirect)
            ));
          $depths[$new_name] = $depth;
        // Add this new author to be considered in the next cycle
        if (!isset($new_nodes[$new_name]))
          $future_new_nodes[$new_name] = $coauthors_keys[$new_name];
        }
        // Adds a link between the current person and this co-author
        if (!isset($common_papers[$key."&#45;&gt;".$coauthors_keys[$new_name]]) && !isset($common_papers[$coauthors_keys[$new_name]."&#45;&gt;".$key]))
        {
          $graph2->addEdge(
            array($key => $coauthors_keys[$new_name]),
            array(
              "color" => $linkcolor,
              "fontsize" => $graph_font_size,
              "dir" => "none",  // Undirected edge, of course
              "label" => $collaborations
              )
            );
          $common_papers[$key."&#45;&gt;".$coauthors_keys[$new_name]] = $collaborations;
        }
      }
    }
    
    // Finds people cited
    if ($depth < CITEES_DEPTH)
    {
      list($citees, $num_citations, $num_self_citations) = getCitees($authorname);
      foreach($citees as $new_name => $citee_key)
      {
        if (!isset($nodes[$new_name]))
        {
          // This citee is not already in the graph:  add it
          $graph2->addNode(
            $citee_key, // Node name
            array(
              "URL"   => author_url($citee_key), // URL
              "label" => ($show_labels ? unaccent($new_name)." (".$depth.")" : ""),
              "shape" => "box",
              "style" => "filled",
              "fontsize" => $graph_font_size,
              ($depth === 1 ? $graph_fill_citee_immediate : $graph_fill_citee_indirect)
            ));
        // Adds a link between the current person and this citee
        $graph2->addEdge(
          array($key => $citee_key),
          array(
            "color" => $linkcolor,
            "fontsize" => $graph_font_size,
            "style" => "dashed",
            "label" => $num_citations[$new_name]."/".$num_self_citations[$new_name]
            )
          );
          // Add this new author to be considered in the next cycle
          if (!isset($nodes[$new_name]) && !isset($new_nodes[$new_name]))
            $future_new_nodes[$new_name] = $coauthors_keys[$new_name];
        }
      }
    }
  }
  // The new authors gathered in this cycle become the authors to consider
  // in the next loop
  $new_nodes = $future_new_nodes;
  $future_new_nodes = array();
}

// Sends the image as an output
if (isset($_GET["format"]) && $_GET["format"] != "svg") // Can be png, svg, gif, jpg
{
  $graph2->image($_GET["format"]);
}
else // SVG default
{
  // In the case of an SVG file, trap the output file with our special function...
  $output = $graph2->image_out();
  // ...then parse the file to add titles to nodes.  The title
  // is the small bubble that pops when the mouse goes over a node, and in
  // our case we make it show a clear-text formatted version of the
  // article title.
  if (!$show_labels)
  {
    foreach($nodes as $key => $value)
    {
      // Escape slashes for regex
      $key = str_replace("/", "\\/", $key);
      $output = preg_replace("/<title>($value)<\\/title>([^<]*?)<a(.*?)>/m", "<title>\$1</title>\$2<a\$3 title=\"".unaccent($key)." (".(isset($depths[$key]) ? $depths[$key] : "0").")\">", $output);
    }
  }
  foreach($common_papers as $key => $collabos)
  {
    list($first, $second) = explode("&#45;&gt;", $key);
    $output = preg_replace("/<title>($key)<\\/title>(.*?)<text(.*?)>(.*?)<\\/text>/s", "<title>\$1</title>\$2<a xlink:href=\"".author_url($first."&amp;with=".$second)."\"><text\$3 title=\"".unaccent($rev_nodes[$first])." and ".unaccent($rev_nodes[$second])."\">\$4</text></a>", $output);
  }
  
  // ...and finally output the resulting file
  header('Content-Type: image/svg+xml');
  header('Content-Length: ' . strlen($output));
  echo $output;
}
?>
