/*
 * $RCSfile: Integer.java,v $
 *
 * Copyright (c) 1999-2002. The Res Medicinae developers. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.number;

import org.resmedicinae.resmedlib.*;
import org.resmedicinae.resmedlib.digit.*;

/**
 * This class represents an integer.<br><br>
 *
 * An integer is an unbroken number with only a whole part.<br><br>
 *
 * For now, this class just encapsulates an integer of the java language.
 *
 * @version $Revision: 1.11 $ $Date: 2002/12/10 17:27:45 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Integer extends org.resmedicinae.resmedlib.Number {
    
    //
    // Static values.
    //

    /** The minimum value. */
    public static final int MINIMUM = java.lang.Integer.MIN_VALUE;

    /** The maximum value. */
    public static final int MAXIMUM = java.lang.Integer.MAX_VALUE;

    /** The default value. */
    public static final int DEFAULT = 0;

    //
    // Constructable.
    //
    
    /**
     * Constructs this integer.
     */
    public Integer() {

        super();
    }

    /**
     * Constructs this integer.<br><br>
     *
     * Normally, constructors are NOT used in the ResMedLib framework,
     * because no counterparts (destructors) exist in the java language.
     * The aim of this prohibition is to ensure a proper startup and shutdown
     * (lifecycle) of every item in the system.<br><br>
     *
     * However, when it comes to lower abstraction (language) levels of our
     * ontology, we need to make some exceptions:<br>
     * Whenever an integer like "0", "1", "2" is given,
     * the java language constructs a new integer object anyway.
     * So, it doesn't matter if also we are creating integers which do not
     * get properly destroyed by the lifecycle, just by the garbage collector
     * after an indefinable time.
     * Most often, such integers are needed/created as static variables to define
     * constants, e.g. attribute ids. That's why we need this integer constructor.
     *
     * @param i the primitive integer value
     */
    public Integer(int i) {

        super();
        setJavaObject(new java.lang.Integer(i));
    }

    //
    // Encapsulated java integer.
    //
    
    /**
     * Creates an encapsulated java integer.
     *
     * @return the encapsulated java integer
     */
    public java.lang.Object createJavaObject() {

        return new java.lang.Integer(createJavaPrimitive());
    }

    //
    // Java primitive.
    //
    
    /**
     * Creates the java primitive.
     *
     * @return the java primitive
     */
    public int createJavaPrimitive() {

        return Integer.DEFAULT;
    }

    /**
     * Destroys the java primitive.
     *
     * @param p the java primitive
     */
    public void destroyJavaPrimitive(int p) {
    }

    /**
     * Sets the java primitive.
     *
     * @param p the java primitive
     * @exception NullPointerException if the java object is null
     */
    public void setJavaPrimitive(int p) throws NullPointerException {
    }

    /**
     * Returns the java primitive.
     *
     * @return the java primitive
     * @exception NullPointerException if the java object is null
     */
    public int getJavaPrimitive() throws NullPointerException {

        int p = Integer.DEFAULT;
        java.lang.Integer j = (java.lang.Integer) getJavaObject();
        
        if (j != null) {

            p = j.intValue();

        } else {
        
            throw new NullPointerException("Could not get java primitive. The java object is null.");
        }            

        return p;
    }

    //
    // Algebraic methods.
    //

    /**
     * Tests if this integer is equal to the given integer.
     *
     * @param i1 the integer
     * @return true - if this integer is equal to the given integer;
     * false - otherwise
     * @exception NullPointerException if the integer is null
     * @exception NullPointerException if the java integer is null
     * @exception NullPointerException if this java integer is null
     */
    public boolean isEqualTo(org.resmedicinae.resmedlib.number.Integer i1) throws NullPointerException {

        boolean result = org.resmedicinae.resmedlib.digit.Boolean.FALSE;

        if (i1 != null) {

            java.lang.Integer j1 = (java.lang.Integer) i1.getJavaObject();

            if (i1 != null) {

                java.lang.Integer j2 = (java.lang.Integer) getJavaObject();

                if (j2 != null) {

                    if (j2.intValue() == j1.intValue()) {

                        result = org.resmedicinae.resmedlib.digit.Boolean.TRUE;
                    }
                
                } else {
                    
                    throw new NullPointerException("Could not test integer. This java integer is null.");
                }

            } else {
                
                throw new NullPointerException("Could not test integer. The java integer is null.");
            }
                
        } else {
            
            throw new NullPointerException("Could not test integer. The integer is null.");
        }

        return result;        
    }

    /**
     * Tests if this integer is equal to the given integer.
     *
     * @param p1 the primitive integer
     * @return true - if this integer is equal to the given integer;
     * false - otherwise
     * @exception NullPointerException if the integer is null
     */
    public boolean isEqualTo(int p1) throws NullPointerException {

        boolean result = org.resmedicinae.resmedlib.digit.Boolean.FALSE;
        
        java.lang.Integer j2 = (java.lang.Integer) getJavaObject();
        
        if (j2 != null) {
            
            if (j2.intValue() == p1) {

                result = org.resmedicinae.resmedlib.digit.Boolean.TRUE;
            }
        
        } else {
            
            throw new NullPointerException("Could not test integer. This java integer is null.");
        }
            
        return result;        
    }

    /**
     * Tests if this integer is smaller than or equal to the given integer.
     *
     * @param i1 the integer
     * @return true - if this integer is smaller than or equal to the given integer;
     * false - otherwise
     * @exception NullPointerException if the integer is null
     * @exception NullPointerException if the java integer is null
     * @exception NullPointerException if this java integer is null
     */
    public boolean isSmallerThanOrEqualTo(org.resmedicinae.resmedlib.number.Integer i1) throws NullPointerException {

        boolean result = org.resmedicinae.resmedlib.digit.Boolean.FALSE;

        if (i1 != null) {

            java.lang.Integer j1 = (java.lang.Integer) i1.getJavaObject();

            if (j1 != null) {

                java.lang.Integer j2 = (java.lang.Integer) getJavaObject();

                if (j2 != null) {

                    if (j2.intValue() <= j1.intValue()) {

                        result = org.resmedicinae.resmedlib.digit.Boolean.TRUE;
                    }
                
                } else {
                    
                    throw new NullPointerException("Could not test integer. This java integer is null.");
                }

            } else {
                
                throw new NullPointerException("Could not test integer. The java integer is null.");
            }
                
        } else {
            
            throw new NullPointerException("Could not test integer. The integer is null.");
        }

        return result;        
    }

    /**
     * Tests if this integer is smaller than or equal to the given integer.
     *
     * @param p1 the primitive integer
     * @return true - if this integer is smaller than or equal to the given integer;
     * false - otherwise
     * @exception NullPointerException if the integer is null
     */
    public boolean isSmallerThanOrEqualTo(int p1) throws NullPointerException {

        boolean result = org.resmedicinae.resmedlib.digit.Boolean.FALSE;
        
        java.lang.Integer j2 = (java.lang.Integer) getJavaObject();
        
        if (j2 != null) {
            
            if (j2.intValue() <= p1) {

                result = org.resmedicinae.resmedlib.digit.Boolean.TRUE;
            }
        
        } else {
            
            throw new NullPointerException("Could not test integer. This java integer is null.");
        }
            
        return result;        
    }
}

