/*
 * $RCSfile: SwingView.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.view.swing;

import java.awt.BorderLayout;
import java.awt.LayoutManager;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.lang.*;
import java.util.*;
import javax.swing.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.logger.*;
import org.resmedicinae.resmedlib.component.view.*;

/**
 * This class represents a swing view.
 *
 * Design patterns:
 * - View in Model View Controller
 *
 * @version $Revision: 1.13 $ $Date: 2002/08/09 21:12:41 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class SwingView extends ResPanel implements View {

    //
    // Attributes.
    //

    /** The logger. */
    private Logger logger;

    //
    // Root pane specific attributes.
    //

    /** The title. */
    private String title;

    /** The icon. */
    private String icon;

    /** The menu bar. */
    private ResMenuBar menuBar;

    /** The tool bar. */
    private ResToolBar toolBar;

    /** The status bar. */
    private ResStatusBar statusBar;

    /** The layout manager. */
    private LayoutManager layoutManager;

    //
    // Display specific attributes.
    //

    /**
     * The display mode which can be one of the following:
     * - window
     * - modal dialog
     * - modeless dialog
     * - frame
     * - internal frame
     * - tab page
     */
    private int displayMode;

    //
    // Window specific attributes.
    //

    /** The flag indicating whether or not this view is resizable. */
    private boolean resizable;

    /**
     * The window bounds that the view will be shown at.
     */
    private Rectangle windowBounds;

    /**
     * The dialog bounds that the view will be shown at.
     */
    private Rectangle dialogBounds;

    /**
     * The frame bounds that the view will be shown at.
     */
    private Rectangle frameBounds;

    /**
     * The internal frame bounds that the view will be shown at.
     */
    private Rectangle internalFrameBounds;

    //
    // Control specific attributes.
    //
    
    /** The close control. */
    private Control closeControl;

    /** The controller. */
    private Controller controller;
    
    /** The model. */
    private Object model;

    //
    // Logger.
    //

    /**
     * Returns the logger.
     *
     * @return the logger
     */
    public Logger getLogger() {

        return this.logger;
    }

    /**
     * Sets the logger.
     *
     * @param logger the logger
     */
    public void setLogger(Logger logger) {

        this.logger = logger;
    }

    //
    // Title.
    //

    /**
     * Returns the title.
     *
     * @return the title
     */
    public String getTitle() {

        return this.title;
    }

    /**
     * Sets the title.
     *
     * @param title the title
     */
    public void setTitle(String title) {

        this.title = title;
    }

    /**
     * Creates a title.
     *
     * @return the title
     */
    public String createTitle() {

        return "Res Medicinae";
    }

    //
    // Icon.
    //

    /**
     * Returns the icon.
     *
     * @return the icon
     */
    public String getIcon() {

        return this.icon;
    }

    /**
     * Sets the icon.
     *
     * @param icon the icon
     */
    public void setIcon(String icon) {

        this.icon = icon;
    }

    /**
     * Creates an icon.
     *
     * @return the icon
     */
    public String createIcon() {

        return "DefaultIcon";
    }

    //
    // Menu bar.
    //

    /**
     * Returns the menu bar.
     *
     * @return the menu bar
     */
    public ResMenuBar getMenuBar() {

        return this.menuBar;
    }

    /**
     * Sets the menu bar.
     *
     * @param menuBar the menu bar
     */
    public void setMenuBar(ResMenuBar menuBar) {

        this.menuBar = menuBar;
    }

    /**
     * Creates a menu bar.
     *
     * @return the menu bar
     * @exception NullPointerException if the menu bar is null
     */
    public ResMenuBar createMenuBar() throws Exception {

        return null;
    }

    /**
     * Destroys the menu bar.
     *
     * @param mb the menu bar
     */
    public void destroyMenuBar(ResMenuBar mb) throws Exception {

        if (mb != null) {

            mb.finalizz();
        }
    }

    //
    // Tool bar.
    //

    /**
     * Returns the tool bar.
     *
     * @return the tool bar
     */
    public ResToolBar getToolBar() {

        return this.toolBar;
    }

    /**
     * Sets the tool bar.
     *
     * @param toolBar the tool bar
     */
    public void setToolBar(ResToolBar toolBar) {

        this.toolBar = toolBar;
    }

    /**
     * Creates a tool bar.
     *
     * @return the tool bar
     * @exception NullPointerException if the tool bar is null
     */
    public ResToolBar createToolBar() throws Exception, NullPointerException {

        ResToolBar tb = new ResToolBar();

        if (tb != null) {

            tb.initialize();

        } else {

            throw new NullPointerException("Could not create tool bar. The tool bar is null.");
        }

        return tb;
    }

    /**
     * Destroys the tool bar.
     *
     * @param tb the tool bar
     */
    public void destroyToolBar(ResToolBar tb) throws Exception {

        if (tb != null) {

            tb.finalizz();
        }
    }

    //
    // Status bar.
    //

    /**
     * Returns the status bar.
     *
     * @return the status bar
     */
    public ResStatusBar getStatusBar() {

        return this.statusBar;
    }

    /**
     * Sets the status bar.
     *
     * @param statusBar the status bar
     */
    public void setStatusBar(ResStatusBar statusBar) {

        this.statusBar = statusBar;
    }

    /**
     * Creates a status bar.
     *
     * @return the status bar
     * @exception NullPointerException if the status bar is null
     */
    public ResStatusBar createStatusBar() throws NullPointerException {

        ResStatusBar sb = new ResStatusBar();

        if (sb != null) {

            sb.initialize();

        } else {

            throw new NullPointerException("Could not create status bar. The status bar is null.");
        }

        return sb;
    }

    /**
     * Destroys the status bar.
     *
     * @param sb the status bar
     */
    public void destroyStatusBar(ResStatusBar sb) throws Exception {

        if (sb != null) {

            sb.finalizz();
        }
    }

    //
    // Layout manager.
    //

    /**
     * Returns the layout manager.
     *
     * @return the layout manager
     */
    public LayoutManager getLayoutManager() {

        return this.layoutManager;
    }

    /**
     * Sets the layout manager.
     *
     * @param layoutManager the layout manager
     */
    public void setLayoutManager(LayoutManager layoutManager) {

        this.layoutManager = layoutManager;
    }

    /**
     * Creates a layout manager.
     *
     * @return the layout manager
     */
    public LayoutManager createLayoutManager() {

        return new BorderLayout();
    }

    //
    // Resizable.
    //

    /**
     * Returns the is resizable flag.
     *
     * @return the is resizable flag
     */
    public boolean getResizable() {

        return this.resizable;
    }

    /**
     * Sets the is resizable flag.
     *
     * @param resizable the is resizable flag
     */
    public void setResizable(boolean resizable) {

        this.resizable = resizable;
    }

    /**
     * Creates an resizable flag.
     *
     * @return the resizable flag
     */
    public boolean createResizable() {

        return true;
    }

    //
    // Window bounds.
    //

    /**
     * Returns the rectangle storing the window bounds.
     *
     * @return the rectangle storing the window bounds
     */
    public Rectangle getWindowBounds() {

        return this.windowBounds;
    }

    /**
     * Sets the rectangle storing the window bounds.
     *
     * @param windowBounds the rectangle storing the window bounds
     */
    public void setWindowBounds(Rectangle windowBounds) {

        this.windowBounds = windowBounds;
    }

    /**
     * Creates a rectangle storing the window bounds.
     *
     * @return the rectangle storing the window bounds
     */
    public Rectangle createWindowBounds() {

        // Determine width.
        int w = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());

        // Use centered x position by default.
        int x = 0;

        // Determine height.
        int h = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());

        // Use centered y position by default.
        int y = 0;

        return new Rectangle(x, y, w, h);
    }

    /**
     * Destroys the rectangle storing the window bounds.
     *
     * @param r the rectangle storing the window bounds
     */
    public void destroyWindowBounds(Rectangle r) {
    }

    //
    // Dialog bounds.
    //

    /**
     * Returns the rectangle storing the dialog bounds.
     *
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle getDialogBounds() {

        return this.dialogBounds;
    }

    /**
     * Sets the rectangle storing the dialog bounds.
     *
     * @param dialogBounds rectangle storing the dialog bounds
     */
    public void setDialogBounds(Rectangle dialogBounds) {

        this.dialogBounds = dialogBounds;
    }

    /**
     * Creates a rectangle storing the dialog bounds.
     *
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle createDialogBounds() {

        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = 640;

        if (w > screenWidth) {

            w = screenWidth;
        }

        // Use centered x position by default.
        int x = (screenWidth - w) / 2;

        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = 480;

        if (h > screenHeight) {

            h = screenHeight;
        }

        // Use centered y position by default.
        int y = (screenHeight - h) / 2;

        return new Rectangle(x, y, w, h);
    }

    /**
     * Destroys the rectangle storing the dialog bounds.
     *
     * @param r the rectangle storing the dialog bounds
     */
    public void destroyDialogBounds(Rectangle r) {
    }

    //
    // Frame bounds.
    //

    /**
     * Returns the rectangle storing the frame bounds.
     *
     * @return the rectangle storing the frame bounds
     */
    public Rectangle getFrameBounds() {

        return this.frameBounds;
    }

    /**
     * Sets the rectangle storing the frame bounds.
     *
     * @param frameBounds the rectangle storing the frame bounds
     */
    public void setFrameBounds(Rectangle frameBounds) {

        this.frameBounds = frameBounds;
    }

    /**
     * Creates a rectangle storing the frame bounds.
     *
     * @return the rectangle storing the frame bounds
     */
    public Rectangle createFrameBounds() {

        // Determine width.
        int w = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());

        // Use centered x position by default.
        int x = 0;

        // Determine height.
        int h = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());

        // Use centered y position by default.
        int y = 0;

        return new Rectangle(x, y, w, h);
    }

    /**
     * Destroys the rectangle storing the frame bounds.
     *
     * @param r the rectangle storing the frame bounds
     */
    public void destroyFrameBounds(Rectangle r) {
    }

    //
    // Internal frame bounds.
    //

    /**
     * Returns the rectangle storing the internal frame bounds.
     *
     * @return the rectangle storing the internal frame bounds
     */
    public Rectangle getInternalFrameBounds() {

        return this.internalFrameBounds;
    }

    /**
     * Sets the rectangle storing the internal frame bounds.
     *
     * @param internalFrameBounds the rectangle storing the internal frame bounds
     */
    public void setInternalFrameBounds(Rectangle internalFrameBounds) {

        this.internalFrameBounds = internalFrameBounds;
    }

    /**
     * Creates a rectangle storing the internal frame bounds.
     *
     * @return the rectangle storing the internal frame bounds
     */
    public Rectangle createInternalFrameBounds() {

        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = 640;

        if (w > screenWidth) {

            w = screenWidth;
        }

        // Use centered x position by default.
        int x = 10;

        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = 480;

        if (h > screenHeight) {

            h = screenHeight;
        }

        // Use centered y position by default.
        int y = 10;

        return new Rectangle(x, y, w, h);
    }

    /**
     * Destroys the rectangle storing the internal frame bounds.
     *
     * @param r the rectangle storing the internal frame bounds
     */
    public void destroyInternalFrameBounds(Rectangle r) {
    }

    //
    // Close control.
    //

    /**
     * Returns the control issued when the window containing this view is closed.
     *
     * @return the control issued when the window containing this view is closed
     */
    public Control getCloseControl() {

        return this.closeControl;
    }

    /**
     * Sets the control issued when the window containing this view is closed.
     *
     * @param closeControl the control issued when the window containing this view is closed
     */
    public void setCloseControl(Control closeControl) {

        this.closeControl = closeControl;
    }

    /**
     * Creates a control issued when the window containing this view is closed.
     *
     * @return the control issued when the window containing this view is closed
     */
    public Control createCloseControl() {

        Control c = new Control();
        
        if (c != null) {
        
            c.setId(Controller.DESTROY_CONTROLLER_CONTROL_ID);
//??            c.setSender((Controller) getController());
            
        } else {
            
            throw new NullPointerException("Could not create close control. The control is null.");
        }
        
        return c;
    }

    //
    // Display mode.
    //

    /**
     * Returns the display mode.
     *
     * @return the display mode
     */
    public int getDisplayMode() {

        return this.displayMode;
    }

    /**
     * Sets the display mode
     *
     * Removes child application component from this application view
     * to allow the creation of new ones.
     *
     * @param displayMode the display mode
     */
    public void setDisplayMode(int displayMode) {

        this.displayMode = displayMode;
    }

    /**
     * Creates a display mode.
     *
     * @return the display mode
     */
    public int createDisplayMode() {

        return DisplayFactory.INTERNAL_FRAME_DISPLAY;
    }

    //
    // Controller.
    //

    /**
     * Returns the controller.
     *
     * @return the controller
     */
    public Controller getController() {

        return this.controller;
    }

    /**
     * Sets the controller.
     *
     * @param controller the controller
     */
    public void setController(Controller controller) {

        this.controller = controller;
    }

    //
    // Model.
    //

    /**
     * Returns the model.
     *
     * @return the model
     */
    public Object getModel() {

        return this.model;
    }

    /**
     * Sets the model.
     *
     * @param model the model
     */
    public void setModel(Object model) {

        this.model = model;
    }

    //
    // Child application component.
    //

    /**
     * Returns the child application component.
     *
     * @return the child application component
     */
    public JComponent getChildApplicationComponent() {

        return null;
    }

    /**
     * Sets the child application component.
     *
     * @param c the child application component
     */
    public void setChildApplicationComponent(JComponent c) {
    }

    //
    // Composable.
    //
    
    /**
     * Composes this item.
     *
     * @param l the components
     * @exception NullPointerException if the components list is null
     */
    public void compose(List l) throws Exception, NullPointerException {

        if (l != null) {
            
        } else {
            
            throw new NullPointerException("Could not compose item. The components list is null.");
        }
    }

    /**
     * Decomposes this item.
     *
     * @param l the components
     * @exception NullPointerException if the components list is null
     */
    public void decompose(List l) throws Exception, NullPointerException {

        if (l != null) {
            
        } else {
            
            throw new NullPointerException("Could not decompose item. The components list is null.");
        }
    }

    //
    // Configurable.
    //

    /**
     * Configures this application.
     *
     * @param m the configuration manager
     * @exception NullPointerException if the configuration manager is null
     */
    public void configure(ConfigurationManager m) throws NullPointerException {

        if (m != null) {

            configureSystemSettings(m.getSystemConfiguration());
            configureLocalSettings(m.getLocalConfiguration());
            configureUserSettings(m.getUserConfiguration());

        } else {

            throw new NullPointerException("Could not configure swing view. The configuration manager is null.");
        }
    }

    /**
     * Deconfigures this application.
     *
     * @param m the configuration manager
     * @exception NullPointerException if the configuration manager is null
     */
    public void deconfigure(ConfigurationManager m) throws NullPointerException {

        if (m != null) {

            deconfigureUserSettings(m.getUserConfiguration());
            deconfigureLocalSettings(m.getLocalConfiguration());
            deconfigureSystemSettings(m.getSystemConfiguration());

        } else {

            throw new NullPointerException("Could not deconfigure swing view. The configuration manager is null.");
        }
    }

    /**
     * Configures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void configureSystemSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            setDisplayMode(c.getDisplayMode(createDisplayMode()));
            setWindowBounds(c.getWindowBounds(createWindowBounds()));
            setDialogBounds(c.getDialogBounds(createDialogBounds()));
            setFrameBounds(c.getFrameBounds(createFrameBounds()));
            setInternalFrameBounds(c.getInternalFrameBounds(createInternalFrameBounds()));
            setResizable(c.getResizable(createResizable()));
            setTitle(c.getTitle(createTitle()));

        } else {

            throw new NullPointerException("Could not configure system settings. The system configuration is null.");
        }
    }

    /**
     * Deconfigures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void deconfigureSystemSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            c.setTitle(getTitle());
            c.setResizable(getResizable());
            c.setWindowBounds(getWindowBounds());
            c.setDialogBounds(getDialogBounds());
            c.setFrameBounds(getFrameBounds());
            c.setInternalFrameBounds(getInternalFrameBounds());
            c.setDisplayMode(getDisplayMode());

        } else {

            throw new NullPointerException("Could not deconfigure system settings. The system configuration is null.");
        }
    }

    /**
     * Configures the local settings.
     *
     * @param c the local configuration
     * @exception NullPointerException if the local configuration is null
     */
    protected void configureLocalSettings(Configuration c) throws NullPointerException {
    }

    /**
     * Deconfigures the local settings.
     *
     * @param c the local configuration
     * @exception NullPointerException if the local configuration is null
     */
    protected void deconfigureLocalSettings(Configuration c) throws NullPointerException {
    }

    /**
     * Configures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void configureUserSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            setDisplayMode(c.getDisplayMode(createDisplayMode()));
            setWindowBounds(c.getWindowBounds(createWindowBounds()));
            setDialogBounds(c.getDialogBounds(createDialogBounds()));
            setFrameBounds(c.getFrameBounds(createFrameBounds()));
            setInternalFrameBounds(c.getInternalFrameBounds(createInternalFrameBounds()));
            setResizable(c.getResizable(createResizable()));
            setTitle(c.getTitle(createTitle()));

        } else {

            throw new NullPointerException("Could not configure user settings. The user configuration is null.");
        }
    }

    /**
     * Deconfigures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void deconfigureUserSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            c.setTitle(getTitle());
            c.setResizable(getResizable());
            c.setWindowBounds(getWindowBounds());
            c.setDialogBounds(getDialogBounds());
            c.setFrameBounds(getFrameBounds());
            c.setInternalFrameBounds(getInternalFrameBounds());
            c.setDisplayMode(getDisplayMode());

        } else {

            throw new NullPointerException("Could not deconfigure user settings. The user configuration is null.");
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() throws Exception {

        setIcon(createIcon());
        setMenuBar(createMenuBar());
        setToolBar(createToolBar());
        setStatusBar(createStatusBar());
        setLayoutManager(createLayoutManager());
        setCloseControl(createCloseControl());

        //?? Put the following methods to an extra lifecycle method "assemble"?
        setLayout(getLayoutManager());
        add(getToolBar(), BorderLayout.NORTH);
        add(getStatusBar(), BorderLayout.SOUTH);
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() throws Exception {

        //?? Put the following methods to an extra lifecycle method "disassemble"?
        remove(getToolBar());
        remove(getStatusBar());
        setLayout(null);

        setCloseControl(null);
        setLayoutManager(null);
        destroyStatusBar(getStatusBar());
        setStatusBar(null);
        destroyToolBar(getToolBar());
        setToolBar(null);
        destroyMenuBar((ResMenuBar) getMenuBar());
        setMenuBar(null);
        setIcon(null);
    }
    
    //
    // Control issuing.
    //
    
    /**
     * Issues a control to this view's controller.
     *
     * @param c the control to be issued
     */
    public void issue(Control control) {
    }
}

