/*
 * $RCSfile: SwingDisplayManager.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All Rights Reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.view.swing;

import java.awt.Container;
import java.awt.Dialog;
import java.awt.Frame;
import java.awt.Window;
import java.lang.*;
import java.util.*;
import javax.swing.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.logger.*;
import org.resmedicinae.resmedlib.component.view.*;

/**
 * This class represents a swing view manager.
 *
 * It manages views in a Swing environment.
 * Views can be displayed in:
 * - Window
 * - Dialog
 * - Frame
 * - Internal Frame
 * - Tab Page
 *
 * @version $Revision: 1.11 $ $Date: 2002/06/12 18:49:40 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class SwingDisplayManager extends DisplayManager {

    //
    // Attributes.
    //

    /**
     * The root panes container.
     *
     * It keeps track of all open Windows/Dialogs/Frames by keeping references
     * to their root panes.
     * Don't bother tracking message boxes because they're modal and don't
     * interact with controllers directly.
     *
     * This is an ordered list: the last opened rootpane is at the end.
     */
    private List rootPanes;

    /** The adapter listening to window events. */
    private DisplayAdapter displayAdapter;

    //
    // Look and feel.
    //

    /**
     * Returns the look and feel.
     *
     * @return the look and feel
     */
    public LookAndFeel getLookAndFeel() {

        return UIManager.getLookAndFeel();
    }

    /**
     * Sets the look and feel.
     *
     * @param lookAndFeel the look and feel
     */
    public void setLookAndFeel(LookAndFeel lookAndFeel) throws UnsupportedLookAndFeelException {

        UIManager.setLookAndFeel(lookAndFeel);
    }

    /**
     * Creates a look and feel.
     *
     * @return the look and feel
     */
    public LookAndFeel createLookAndFeel() throws Exception {

        return new javax.swing.plaf.metal.MetalLookAndFeel();
    }

    /**
     * Destroys the look and feel.
     */
    public void destroyLookAndFeel() throws Exception {
    }

    //
    // Root panes.
    //

    /**
     * Returns the root panes container.
     *
     * @return the root panes container
     */
    public List getRootPanes() {

        return this.rootPanes;
    }

    /**
     * Sets the root panes container.
     *
     * @param rootPanes the root panes container
     */
    public void setRootPanes(List rootPanes) {

        this.rootPanes = rootPanes;
    }

    /**
     * Creates a root panes container.
     *
     * @return the root panes container
     */
    public List createRootPanes() {

        return new LinkedList();
    }

    /**
     * Destroys the root panes container.
     *
     * @param the root panes container
     * @exception NullPointerException if the root panes list is null
     */
    public void destroyRootPanes(List l) throws NullPointerException {

        if (l != null) {

            Object o = null;

            for (int i = 0; i < l.size(); i++) {

                o = l.get(i);

                l.remove(o);
            }

        } else {

            throw new NullPointerException("Could not destroy root panes. The root panes list is null.");
        }
    }

    //
    // Display adapter.
    //

    /**
     * Returns the adapter listening to window events.
     *
     * @return the adapter listening to window events
     */
    public DisplayAdapter getDisplayAdapter() {

        return this.displayAdapter;
    }

    /**
     * Sets the adapter listening to window events.
     *
     * @param displayAdapter the adapter listening to window events
     */
    public void setDisplayAdapter(DisplayAdapter displayAdapter) {

        this.displayAdapter = displayAdapter;
    }

    /**
     * Creates an adapter listening to window events.
     *
     * @return the adapter listening to window events
     */
    public DisplayAdapter createDisplayAdapter() {

        return null;
    }

    /**
     * Destroys the adapter listening to window events.
     *
     * @param the adapter listening to window events
     * @exception NullPointerException if the adapter is null
     */
    public void destroyDisplayAdapter(DisplayAdapter da) throws NullPointerException {
    }

    //
    // Configurable.
    //

    /**
     * Configures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void configureSystemSettings(Configuration c) throws Exception, NullPointerException {

        super.configureSystemSettings(c);

        if (c != null) {

            setLookAndFeel(c.getLookAndFeel(createLookAndFeel()));

        } else {

            throw new NullPointerException("Could not configure system settings. The system configuration is null.");
        }
    }

    /**
     * Deconfigures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void deconfigureSystemSettings(Configuration c) throws Exception, NullPointerException {

        try {

            if (c != null) {

                c.setLookAndFeel(getLookAndFeel());

            } else {

                throw new NullPointerException("Could not deconfigure system settings. The system configuration is null.");
            }

        } finally {

            super.deconfigureSystemSettings(c);
        }
    }

    /**
     * Configures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void configureUserSettings(Configuration c) throws Exception, NullPointerException {

        super.configureUserSettings(c);

        if (c != null) {

            setLookAndFeel(c.getLookAndFeel(createLookAndFeel()));

        } else {

            throw new NullPointerException("Could not configure user settings. The user configuration is null.");
        }
    }

    /**
     * Deconfigures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void deconfigureUserSettings(Configuration c) throws Exception, NullPointerException {

        try {

            if (c != null) {

                c.setLookAndFeel(getLookAndFeel());

            } else {

                throw new NullPointerException("Could not deconfigure user settings. The user configuration is null.");
            }

        } finally {

            super.deconfigureUserSettings(c);
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() {

        setRootPanes(createRootPanes());
        setDisplayAdapter(createDisplayAdapter());
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() {

        destroyDisplayAdapter(getDisplayAdapter());
        setDisplayAdapter(null);
        destroyRootPanes(getRootPanes());
        setRootPanes(null);
    }

    //
    // Creation and destruction.
    //

    /**
     * Creates a display for the view.
     *
     * If a root pane already exists for the given application view,
     * then the view's display is brought to front.
     * Otherwise, a new display will be created for the application view and
     * its root pane will be registered here.
     *
     * @param v the swing view
     * @param pv the parent view
     * @exception NullPointerException if the display factory is null
     */
    public void createDisplay(View v, View pv) throws NullPointerException, Exception {

        // Get topmost view container to ensure that only application views are used following.
        SwingView sv = (SwingView) getRootContainer((SwingView) v);
        DisplayFactory f = createDisplayFactory(sv, (SwingView) pv);

        if (f != null) {

            // Try to find the root pane that holds the view.
            ResRootPane r = findRootPaneFor(sv);

/*??
            if (r != null) {

                // If holding root pane exists, let the factory bring its display to front.
                f.bringDisplayToFront(r);

            } else {
*/

                // Let factory create the display and return its root pane.
                r = f.createDisplay(sv, (SwingView) pv);

                registerRootPane(r);
/*??
            }
*/

        } else {

            throw new NullPointerException("Could not create view display. The display factory is null.");
        }
    }

    /**
     * Destroys the display of the view.
     *
     * The root pane of the given swing view will be unregistered here.
     *
     * @param v the swing view
     * @param pv the parent view
     * @exception NullPointerException if the display factory is null
     */
    public void destroyDisplay(View v, View pv) throws NullPointerException, Exception {

        // Get topmost view container to ensure that only application views are used following.
        SwingView sv = (SwingView) getRootContainer((SwingView) v);
        DisplayFactory f = createDisplayFactory(sv, (SwingView) pv);
        // Try to find the root pane that holds the view.
        ResRootPane r = findRootPaneFor(sv);

        if (f != null) {

            // Let factory destroy the display.
            f.destroyDisplay(sv, (SwingView) pv, r);

        } else {

            throw new NullPointerException("Could not destroy view display. The display factory is null.");
        }

        unregisterRootPane(r);
    }

    /**
     * Destroys all open displays.
     *
     * @exception NullPointerException if the root panes list is null
     * @exception NullPointerException if the root panes list iterator is null
     * @exception NullPointerException if the display factory is null
     */
    public void destroyAllDisplays() throws NullPointerException {

        List l = getRootPanes();

        if (l != null) {

            Iterator iter = l.iterator();

            if (iter != null) {

                Object o = null;

                while (iter.hasNext()) {

                    o = iter.next();

/*??
                    DisplayFactory f = createDisplayFactory(av);

                    if (f != null) {

                        f.destroyDisplay(findView((ResRootPane) o), (ResRootPane) o);

                    } else {

                        throw new NullPointerException("Could not destroy view display. The display factory is null.");
                    }

                    unregisterRootPane(r);
*/
                }

            } else {

                throw new NullPointerException("Could not destroy view displays. The root panes list iterator is null.");
            }

        } else {

            throw new NullPointerException("Could not destroy view displays. The root panes list is null.");
        }
    }

    //
    // Root pane management.
    //

    /**
     * Registers root pane.
     *
     * @param r the root pane
     * @exception NullPointerException if the root pane list ist null
     */
    private void registerRootPane(ResRootPane r) throws NullPointerException {

        List l = getRootPanes();

        if (l != null) {

            l.add(r);

        } else {

            throw new NullPointerException("Could not register root pane. The root pane list is null.");
        }
    }

    /**
     * Unregisters root pane.
     *
     * @param r the root pane
     * @exception NullPointerException if the root pane list ist null
     */
    private void unregisterRootPane(ResRootPane r) throws NullPointerException {

        List l = getRootPanes();

        if (l != null) {

            l.remove(r);

        } else {

            throw new NullPointerException("Could not unregister root pane. The root pane list is null.");
        }
    }

    //
    // Factory management.
    //

    /**
     * Creates a display factory.
     *
     * The type of the created factory depends on the view's display mode.
     *
     * Following display modes exist: Window, Dialog, others (switchable)
     *
     * @param v the swing view
     * @param pv the parent swing view
     * @return the display factory
     * @exception NullPointerException if the application view is null
     * @see createSwitchableDisplayFactory
     */
    protected DisplayFactory createDisplayFactory(SwingView v, SwingView pv) throws Exception, NullPointerException {

        DisplayFactory f = null;

        if (v != null) {

            int mode = v.getDisplayMode();

            // If no root pane exists, create a window/dialog/switchable display (top level frame/internal frame/tab page) with a new root pane.
            if (mode == DisplayFactory.WINDOW_DISPLAY) {

                f = new WindowDisplayFactory();

            } else if ((mode == DisplayFactory.MODAL_DIALOG_DISPLAY) || (mode == DisplayFactory.MODELESS_DIALOG_DISPLAY)) {

                f = new DialogDisplayFactory();

                if (f != null) {

                    ((DialogDisplayFactory) f).setParentWindow(findParentWindow());
                }

            } else {

                // The standard mode is a switchable display which can mean TopLevelFrame/InternalFrame/TabbedPane.
                f = createSwitchableDisplayFactory(v, pv);
            }

        } else {

            throw new NullPointerException("Could not create display factory. The application view is null.");
        }

        return f;
    }

    /**
     * Creates a switchable display factory.
     *
     * If the view's application controller has a parent application controller,
     * then use its view's display mode to decide which display to create.
     * If the view's controller (application) has no parent application controller,
     * then create a frame display since the view is standalone.
     *
     * Following display modes exist: TopLevelFrame, InternalFrame, TabPage
     *
     * @param v the swing view to be displayed
     * @param pv the parent swing view
     * @return the display factory
     * @exception Exception if the display mode does not exist
     * @see createDisplayFactory
     */
    protected DisplayFactory createSwitchableDisplayFactory(SwingView v, SwingView pv) throws Exception {

        DisplayFactory f = null;

        if (pv != null) {

            int mode = pv.getDisplayMode();

            if (mode == DisplayFactory.FRAME_DISPLAY) {

                f = new FrameDisplayFactory();

            } else if (mode == DisplayFactory.INTERNAL_FRAME_DISPLAY) {

                f = new InternalFrameDisplayFactory();

            } else if (mode == DisplayFactory.TAB_PAGE_DISPLAY) {

                f = new TabPageDisplayFactory();

            } else {

                throw new Exception("Could not create switchable display. The display mode does not exist.");
            }

        } else {

            // If no parent view exists, then this view is the root and hence a frame.
            f = new FrameDisplayFactory();
        }

        return f;
    }

    //
    // Helping methods.
    //

    /**
     * Looks for the topmost Container, stopping before any Window structure.
     *
     * @param v the view to be displayed
     * @return the topmost container of the view
     * @exception NullPointerException if the view is null
     */
    protected Container getRootContainer(SwingView v) throws NullPointerException {

        Container c = (SwingView) v;

        if (v != null) {

            ResRootPane r = (ResRootPane) ((SwingView) v).getRootPane();
            Container cp = null;

            // Does the view already belong to a root pane?
            // Then find the content pane so we know when to stop below.
            if (r != null) {

                cp = r.getContentPane();
            }

            // Determine the parent container of the view.
            Container parent = c.getParent();

            while ((parent != null) && (parent != cp)) {

                // Remember the parent as the current topmost container.
                c = parent;
                // Determine the next higher parent container of the parent.
                parent = parent.getParent();
            }

        } else {

            throw new NullPointerException("Could not find topmost container. The view is null.");
        }

        return c;
    }

    /**
     * Finds the root pane that contains the view.
     *
     * @param v the swing view
     * @return the root pane that contains the view
     * @exception NullPointerException if the root pane list is null
     */
    protected ResRootPane findRootPaneFor(SwingView v) throws NullPointerException {

        ResRootPane r = null;
        List l = getRootPanes();

        if (l != null) {

            Iterator iter = l.iterator();
            ResRootPane pr = findFirstParentRootPaneFor(v);

            while (iter.hasNext()) {

                r = (ResRootPane) iter.next();

                if (r == pr) {

                    break;
                }
/*??
                if ((r != null) && (r.isAncestorOf((SwingView) v))) {

                    break;
                }
*/
            }

        } else {

            throw new NullPointerException("Could not find root pane for the swing view. The root pane list is null.");
        }

        return r;
    }

    /**
     * Finds the first parent which is a root pane container of the given view.
     *
     * @param v the swing view
     * @return the first parent which is a root pane container of the given view
     * @exception NullPointerException if the root pane list is null
     */
    private ResRootPane findFirstParentRootPaneFor(SwingView v) throws NullPointerException {

        ResRootPane r = null;
        Container c = (SwingView) v;

        while (c != null) {

            if (c instanceof ResRootPane) {

                r = (ResRootPane) c;

                break;
            }

            c = c.getParent();
        }

        return r;
    }

    /**
     * Finds the parent window.
     *
     * Necessary for displaying dialogs as they request a java.awt.component
     * being handed over as parent window.
     *
     * Search is done in the following order:
     * - currently focussed window
     * - last opened window (if no focussed window has been found)
     * - shared null frame (if no opened window has been found)
     *
     * @return the parent window
     */
    protected Window findParentWindow() throws Exception {

        Window w = null;
        ResRootPane r = getFocussedRootPane();

        if (r != null) {

            // The currently focussed window.
            w = (Window) r.getParent();

        } else {

            List l = getRootPanes();

            if ((l != null) && (l.size() > 0)) {

                // The last opened window.
                w = (Window) ((ResRootPane) ((LinkedList) l).getLast()).getParent();

            } else {

//??                w = getSharedNullFrame();
            }
        }

        return w;
    }

    /**
     * Returns the rootpane that is owned by the currently focussed window.
     *
     * @return the rootpane that is owned by the currently focussed window
     */
    protected ResRootPane getFocussedRootPane() throws Exception {

        ResRootPane r = null;
        Iterator iter = getRootPanes().iterator();

        while (iter.hasNext()) {

            r = (ResRootPane) iter.next();

            if (r != null) {

                // Determine the parent window of the root pane.
                Window w = (Window) r.getParent();

                if (isFocusOwner(w)) {

                    break;
                }
            }
        }

        return r;
    }

    /**
     * Returns a flag indicating whether or not the focus lies on the window.
     *
     * Swing loses focus from the current component when a menu is pulled down
     * so for a Window to have focus we say it needs to either contain the
     * focus owner OR parent a Window (not Dialog/Frame) that owns the focus.
     *
     * @param w the window
     * @return a flag indicating whether or not the focus lies on the window
     * @exception Exception if the window is null
     */
    protected boolean isFocusOwner(Window w) throws Exception {

        boolean focus = false;

        if (w != null) {

            if (w.getFocusOwner() != null) {

                // The current window has the focus.
                focus = true;

            } else {

                Window[] children = w.getOwnedWindows();

                if (children != null) {

                    Window child = null;

                    for (int i = 0; i < children.length; i++) {

                        child = children[i];

                        //?? What about Window here, if it isn't a menu but the splash screen window, for example??
                        if (!(child instanceof Dialog) && !(child instanceof Frame)) {

                            if (isFocusOwner(child)) {

                                // A child window (e.g. menu) of the current window has the focus.
                                focus = true;
                            }
                        }
                    }
                }
            }

        } else {

            throw new Exception("Could not check window focus. The window is null.");
        }

        return focus;
    }

    //
    // Message display.
    //

    /**
     * Shows a message dialog.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
    public void showMessage(Level lev, String msg, Throwable t) throws Exception {

        java.awt.Component c = findParentWindow();
        String title = "Undefined Title";
        int type = ResOptionPane.INFORMATION_MESSAGE;

        if (lev.equals(Level.SEVERE)) {

            type = ResOptionPane.ERROR_MESSAGE;

        } else if (lev.equals(Level.WARNING)) {

            type = ResOptionPane.WARNING_MESSAGE;

        } else if (lev.equals(Level.INFO) || lev.equals(Level.CONFIG) || lev.equals(Level.FINE) || lev.equals(Level.FINER) || lev.equals(Level.FINEST)) {

            type = ResOptionPane.INFORMATION_MESSAGE;
        }

        ResOptionPane.showMessageDialog(c, msg + "\n" + t, title, type);
    }
}

