/*
 * $RCSfile: ResTabPage.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.view.swing;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.lang.*;
import javax.accessibility.*;
import javax.swing.*;

/**
 * This class represents a tab page.
 *
 * @version $Revision: 1.2 $ $Date: 2002/05/02 10:18:34 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class ResTabPage extends ResPanel implements Accessible, RootPaneContainer, ApplicationDisplay {

    /** The root pane bound property name. */
    public final static String ROOT_PANE_PROPERTY = "rootPane";
    /** The content pane bound property name. */
    public final static String CONTENT_PANE_PROPERTY = "contentPane";
    /** The layered pane bound property name. */
    public final static String LAYERED_PANE_PROPERTY = "layeredPane";
    /** The glass pane bound property name. */
    public final static String GLASS_PANE_PROPERTY = "glassPane";
    /** Bound property name. */
//??    public final static String MENU_BAR_PROPERTY = "JMenuBar";
    /** Bound property name. */
//??    public final static String TITLE_PROPERTY = "title";

    /**
     * The JRootPane instance that manages the content pane and
     * optional menu bar for this tab page, as well as the glass pane.
     */
    private JRootPane rootPane;

    /**
     * Constructs this object.
     */
    public ResTabPage() {

        setRootPane(createRootPane());
    }

    /**
     * Return the root pane.
     *
     * The rootPane has two children: the glassPane and the layeredPane.
     *
     * @return the root pane
     */
    public JRootPane getRootPane() {

        return this.rootPane;
    }

    /**
     * Sets the root pane.
     *
     * @param rootPane the root pane
     */
    public void setRootPane(JRootPane rootPane) {

        if (this.rootPane != null) {

            remove(this.rootPane);
        }

        if (rootPane != null) {

            add(rootPane, BorderLayout.CENTER);
        }

        JRootPane old = getRootPane();
        this.rootPane = rootPane;

        firePropertyChange(ROOT_PANE_PROPERTY, old, rootPane);
    }

    /**
     * Returns the content pane.
     *
     * @return the content pane
     * @see #setContentPane
     */
    public Container getContentPane() {

        Container c = null;
        JRootPane r = getRootPane();

        if (r != null) {

            c = r.getContentPane();
        }

        return c;
    }

    /**
     * Sets the content pane.
     *
     * The content pane is the primary container for application specific
     * components, i.e. for the contents of the root pane. Applications should
     * add children to the contentPane, set its layout manager and so on.
     *
     * The contentPane my not be null.
     *
     * Generally implemented with
     * <code>getRootPane().setContentPane(contentPane);</code>
     *
     * @param contentPane the content pane
     * @exception java.awt.IllegalComponentStateException (a runtime
     * exception) if the content pane parameter is null
     * @see JRootPane#getContentPane
     * @see #getContentPane
     */
    public void setContentPane(Container contentPane) {

        Container old = getContentPane();
        JRootPane r = getRootPane();

        if (r != null) {

            r.setContentPane(contentPane);
            firePropertyChange(CONTENT_PANE_PROPERTY, old, contentPane);
        }
    }

    /**
     * Returns the layered pane.
     *
     * @return the layered pane
     * @see #setLayeredPane
     */
    public JLayeredPane getLayeredPane() {

        JLayeredPane l = null;
        JRootPane r = getRootPane();

        if (r != null) {

            l = r.getLayeredPane();
        }

        return l;
    }

    /**
     * Sets the layered pane.
     *
     * A container that manages the content pane and in some cases a menu bar.
     * The layered pane can be used by descendants that want to add a child
     * to the RootPaneContainer that isn't layout managed.  For example
     * an internal dialog or a drag and drop effect component.
     *
     * The layeredPane may not be null.
     *
     * Generally implemented with
     * <code>getRootPane().setLayeredPane(layeredPane);</code>
     *
     * @param layeredPane the layered pane
     * @exception java.awt.IllegalComponentStateException (a runtime
     * exception) if the layered pane parameter is null
     * @see #getLayeredPane
     * @see JRootPane#getLayeredPane
     */
    public void setLayeredPane(JLayeredPane layeredPane) {

        JLayeredPane old = getLayeredPane();
        JRootPane r = getRootPane();

        if (r != null) {

            r.setLayeredPane(layeredPane);
            firePropertyChange(LAYERED_PANE_PROPERTY, old, layeredPane);
        }
    }

    /**
     * Returns the glass pane.
     *
     * @return the glass pane
     * @see #setGlassPane
     */
    public Component getGlassPane() {

        Component g = null;
        JRootPane r = getRootPane();

        if (r != null) {

            g = r.getGlassPane();
        }

        return g;
    }

    /**
     * Sets the glass pane.
     *
     * The glass pane is always the first child of the root pane and
     * the root panes layout manager ensures that it's always as big as
     * the rootPane. By default, it is transparent and not visible.
     * It can be used to temporarily grab all keyboard and mouse input
     * by adding listeners and then making it visible.
     * By default, it is not visible.
     *
     * The glass pane may not be null.
     *
     * Generally implemented with
     * <code>getRootPane().setGlassPane(glassPane);</code>
     *
     * @param glassPane the glass pane
     * @see #getGlassPane
     * @see JRootPane#setGlassPane
     */
    public void setGlassPane(Component glassPane) {

        Component old = getGlassPane();
        JRootPane r = getRootPane();

        if (r != null) {

            r.setGlassPane(glassPane);
            firePropertyChange(GLASS_PANE_PROPERTY, old, glassPane);
        }
    }

    /**
     * Creates root pane.
     *
     * @return the root pane
     */
    protected JRootPane createRootPane() {

        return new ResRootPane();
    }
}

