/*
 * $RCSfile: Logger.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.logger;

import java.lang.*;
import org.resmedicinae.resmedlib.component.component.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.context.*;

/**
 * This class represents a logger.
 *
 * @version $Revision: 1.13 $ $Date: 2002/08/08 21:55:16 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Logger extends Component {

    //
    // Constants.
    //

    /** The role constant. */
    public static final String ROLE = "logger";

    /** The constant indicating a console output handler. */
    public static final String CONSOLE_OUTPUT = "console";

    //
    // Attributes.
    //

    /** The named subsystem. */
    private String namedSubsystem;

    /** The current logger output. */
    private String loggerOutput;

    /** The log level. */
    private Level level;

    /** The encapsulated jdk logger. */
    private java.util.logging.Logger jdkLogger;

    //
    // Role.
    //

    /**
     * Returns the role.
     *
     * @return the role
     */
    public String getRole() {

        return Logger.ROLE;
    }

    //
    // Named subsystem.
    //

    /**
     * Returns the named subsystem.
     *
     * @return the named subsystem
     */
    public String getNamedSubsystem() {

        return this.namedSubsystem;
    }

    /**
     * Sets the named subsystem.
     *
     * @param namedSubsystem the named subsystem
     */
    public void setNamedSubsystem(String namedSubsystem) {

        this.namedSubsystem = namedSubsystem;
    }

    //
    // Logger Output.
    //

    /**
     * Returns the current logger output.
     *
     * @return the current logger output
     */
    public String getLoggerOutput() {

        return this.loggerOutput;
    }

    /**
     * Sets the current logger output.
     *
     * @param loggerOutput the current logger output
     */
    public void setLoggerOutput(String loggerOutput) {

        this.loggerOutput = loggerOutput;
    }

    /**
     * Creates a logger output.
     *
     * @return the logger output
     */
    public String createLoggerOutput() {

        return "console";
    }

    /*
     * Destroys the logger output.
     */
    public void destroyLoggerOutput() {
    }

    //
    // Level.
    //

    /**
     * Returns the log level.
     *
     * @return the log level
     * @exception NullPointerException if the jdk logger is null
     */
    public Level getLevel() throws NullPointerException {

        return this.level;
    }

    /**
     * Sets the log level.
     *
     * @param level the log level
     * @exception NullPointerException if the jdk logger is null
     */
    public void setLevel(Level level) throws NullPointerException {

        this.level = level;

        // Forward the level to the embedded jdk logger.
        java.util.logging.Logger log = getJdkLogger();

        if (log != null) {

            log.setLevel(level);
        }
    }

    /**
     * Creates a log level.
     *
     * @return the log level
     */
    public Level createLevel() {

        return (Level) Level.ALL;
    }

    /*
     * Destroys the log level.
     *
     * @param l the log level
     */
    public void destroyLevel(Level l) {
    }

    //
    // Encapsulated jdk logger.
    //

    /**
     * Returns the jdk logger.
     *
     * @return the jdk logger
     */
    public java.util.logging.Logger getJdkLogger() {

        return this.jdkLogger;
    }

    /**
     * Sets the jdk logger.
     *
     * @param jdkLogger the jdk logger
     */
    public void setJdkLogger(java.util.logging.Logger jdkLogger) {

        this.jdkLogger = jdkLogger;
    }

    /**
     * Creates a jdk logger.
     *
     * @return the jdk logger
     */
    public java.util.logging.Logger createJdkLogger() {

        return java.util.logging.Logger.getLogger(getNamedSubsystem());
    }

    /**
     * Destroys the jdk logger.
     *
     * @param l the jdk logger
     */
    public void destroyJdkLogger(java.util.logging.Logger l) {
    }

    //
    // Handler container.
    //

    /**
     * Returns the handlers.
     *
     * @return the handlers
     * @exception NullPointerException if the jdk logger is null
     */
    public java.util.logging.Handler[] getHandlers() throws NullPointerException {

        java.util.logging.Handler[] h = null;
        java.util.logging.Logger l = getJdkLogger();

        if (l != null) {

            h = l.getHandlers();

        } else {

            throw new NullPointerException("Could not get handlers. The jdk logger is null.");
        }

        return h;
    }

    /*
     * Removes the handlers so that the handler container is empty.
     *
     * @handlers the handlers
     * @exception NullPointerException if the handlers container is null
     */
    public void destroyHandlers(java.util.logging.Handler[] handlers) throws Exception, NullPointerException {

        if (handlers != null) {

            java.util.logging.Handler h = null;

            for (int i = 0; i < handlers.length; i++) {

                h = handlers[i];

                destroyHandler(h);
                removeHandler(h);
            }

        } else {

            throw new NullPointerException("Could not destroy handlers. The handlers container is null.");
        }
    }

    //
    // Handler.
    //

    /**
     * Adds the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the jdk logger is null
     */
    public void addHandler(java.util.logging.Handler h) throws NullPointerException {

        java.util.logging.Logger l = getJdkLogger();

        if (l != null) {

            l.addHandler(h);

        } else {

            throw new NullPointerException("Could not add handler. The jdk logger is null.");
        }
    }

    /**
     * Removes the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the jdk logger is null
     */
    public void removeHandler(java.util.logging.Handler h) throws NullPointerException {

        java.util.logging.Logger l = getJdkLogger();

        if (l != null) {

            l.removeHandler(h);

        } else {

            throw new NullPointerException("Could not remove handler. The jdk logger is null.");
        }
    }

    /**
     * Creates a handler.
     *
     * @return the handler
     * @exception Exception if the output is not known
     * @exception NullPointerException if the handler is null
     */
    public java.util.logging.Handler createHandler() throws Exception, NullPointerException {

        java.util.logging.Handler h = null;
        String out = getLoggerOutput();

        if (out.equals(Logger.CONSOLE_OUTPUT)) {

            h = new java.util.logging.ConsoleHandler();

        } else {

            throw new Exception("Could not create handler. The output is not known.");
        }

        if (h != null) {

            // Set log level for handler specifying which log messages should be handled by it.
            h.setLevel(getLevel());

        } else {

            throw new NullPointerException("Could not create handler. The handler is null.");
        }

        return h;
    }

    /**
     * Destroys the handler.
     *
     * @param h the handler
     * @exception NullPointerException if the handler is null
     */
    public void destroyHandler(java.util.logging.Handler h) throws NullPointerException {

        if (h != null) {

            // Set log level for handler specifying which log messages should be handled by it.
            h.setLevel(Level.OFF);

        } else {

            throw new NullPointerException("Could not destroy handler. The handler is null.");
        }
    }

    //
    // Loggable.
    //

    /**
     * Logs a message with associated throwable information.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
    public void log(Level lev, String msg, Throwable t) throws Exception {

        java.util.logging.Logger l = getJdkLogger();

        if (l != null) {

            l.log(lev, msg, t);

        } else {

            throw new NullPointerException("Could not log message. The jdk logger is null.");
        }
    }

    //
    // Contextualizable.
    //

    /**
     * Contextualizes this component.
     *
     * @param c the context
     * @exception NullPointerException if the context is null
     */
    public void contextualize(Context c) throws Exception {

        if (c != null) {

            setNamedSubsystem((String) c.getEntry("named_subsystem"));

        } else {

            throw new NullPointerException("Could not set named subsystem. The context is null.");
        }
    }

    /**
     * Decontextualizes this component.
     *
     * @param c the context
     * @exception NullPointerException if the context is null
     */
    public void decontextualize(Context c) throws Exception {

        if (c != null) {

            setNamedSubsystem(null);

        } else {

            throw new NullPointerException("Could not set named subsystem. The context is null.");
        }
    }

    //
    // Configurable.
    //

    /**
     * Configures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void configureSystemSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            setLoggerOutput(c.getLoggerOutput(createLoggerOutput()));
            setLevel(c.getLevel(createLevel()));

        } else {

            throw new NullPointerException("Could not configure system settings. The system configuration is null.");
        }
    }

    /**
     * Deconfigures the system settings.
     *
     * @param c the system configuration
     * @exception NullPointerException if the system configuration is null
     */
    protected void deconfigureSystemSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            c.setLevel(getLevel());
            c.setLoggerOutput(getLoggerOutput());

        } else {

            throw new NullPointerException("Could not deconfigure system settings. The system configuration is null.");
        }
    }

    /**
     * Configures the local settings.
     *
     * @param c the local configuration
     * @exception NullPointerException if the local configuration is null
     */
    protected void configureLocalSettings(Configuration c) throws NullPointerException {
    }

    /**
     * Deconfigures the local settings.
     *
     * @param c the local configuration
     * @exception NullPointerException if the local configuration is null
     */
    protected void deconfigureLocalSettings(Configuration c) throws NullPointerException {
    }

    /**
     * Configures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void configureUserSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            setLoggerOutput(c.getLoggerOutput(createLoggerOutput()));
            setLevel(c.getLevel(createLevel()));

        } else {

            throw new NullPointerException("Could not configure user settings. The user configuration is null.");
        }
    }

    /**
     * Deconfigures the user settings.
     *
     * @param c the user configuration
     * @exception NullPointerException if the user configuration is null
     */
    protected void deconfigureUserSettings(Configuration c) throws NullPointerException {

        if (c != null) {

            c.setLevel(getLevel());
            c.setLoggerOutput(getLoggerOutput());

        } else {

            throw new NullPointerException("Could not deconfigure user settings. The user configuration is null.");
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() throws Exception {

        setJdkLogger(createJdkLogger());

        addHandler(createHandler());
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() throws Exception {

        destroyHandlers(getHandlers());

        destroyJdkLogger(getJdkLogger());
        setJdkLogger(null);
    }
}

