/*
 * $RCSfile: ConfigurationManager.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.configuration;

import java.io.*;
import java.lang.*;
import java.util.*;
import java.util.prefs.*;
import org.resmedicinae.resmedlib.component.component.*;

/**
 * This class represents a configuration manager.
 *
 * It is to manage different kinds of configuration entities, e.g.:
 * - SystemConfiguration
 * - LocalConfiguration
 * - UserConfiguration
 *
 * The last has highest priority, i.e. user properties will overwrite
 * any other properties of the same kind that were set before.
 *
 * @version $Revision: 1.9 $ $Date: 2002/06/12 18:49:37 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class ConfigurationManager extends Item {

    //
    // Configuration attributes.
    //

    /** The system configuration. */
    private Configuration systemConfiguration;

    /** The local configuration. */
    private Configuration localConfiguration;

    /** The user configuration. */
    private Configuration userConfiguration;

    //
    // System configuration.
    //

    /*
     * Returns the system configuration.
     *
     * return the system configuration
     */
    public Configuration getSystemConfiguration() {

        return this.systemConfiguration;
    }

    /*
     * Sets the system configuration.
     *
     * @param systemConfiguration the system configuration
     */
    public void setSystemConfiguration(Configuration systemConfiguration) {

        this.systemConfiguration = systemConfiguration;
    }

    //
    // Local configuration.
    //

    /*
     * Returns the local configuration.
     *
     * return the local configuration
     */
    public Configuration getLocalConfiguration() {

        return this.localConfiguration;
    }

    /*
     * Sets the local configuration.
     *
     * @param localConfiguration the local configuration
     */
    public void setLocalConfiguration(Configuration localConfiguration) {

        this.localConfiguration = localConfiguration;
    }

    //
    // User configuration.
    //

    /*
     * Returns the user configuration.
     *
     * return the user configuration
     */
    public Configuration getUserConfiguration() {

        return this.userConfiguration;
    }

    /*
     * Sets the user configuration.
     *
     * @param userConfiguration the user configuration
     */
    public void setUserConfiguration(Configuration userConfiguration) {

        this.userConfiguration = userConfiguration;
    }

    //
    // Location.
    //

    /**
     * Sets the configuration locations.
     *
     * @param args the command line arguments
     * @exception NullPointerException if the system configuration is null
     * @exception NullPointerException if the local configuration is null
     * @exception NullPointerException if the user configuration is null
     */
    public void setLocations(String[] args) throws NullPointerException {

        Configuration sys = getSystemConfiguration();
        Configuration loc = getLocalConfiguration();
        Configuration usr = getUserConfiguration();

        if (sys != null) {

            sys.setLocation(args[0]);

        } else {

            throw new NullPointerException("Could not set the system configuration location. The system configuration is null.");
        }

        if (loc != null) {

            loc.setLocation(args[1]);

        } else {

            throw new NullPointerException("Could not set the local configuration location. The local configuration is null.");
        }

        if (usr != null) {

            usr.setLocation(args[2]);

        } else {

            throw new NullPointerException("Could not set the user configuration location. The user configuration is null.");
        }
    }

    //
    // Loadable.
    //

    /**
     * Loads the configuration data.
     *
     * @param c the configuration
     * @exception NullPointerException if the configuration is null
     */
    public void load(Configuration c) throws NullPointerException, FileNotFoundException, IOException, InvalidPreferencesFormatException {

        if (c != null) {

            c.load();

        } else {

            throw new NullPointerException("Could not load configuration data. The configuration is null.");
        }
    }

    /**
     * Saves the configuration data.
     *
     * @param c the configuration
     * @exception NullPointerException if the configuration is null
     */
    public void save(Configuration c) throws NullPointerException, FileNotFoundException, IOException, InvalidPreferencesFormatException, BackingStoreException {

        if (c != null) {

            c.save();

        } else {

            throw new NullPointerException("Could not save configuration data. The configuration is null.");
        }
    }
}

