/*
 * $RCSfile: Configuration.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.configuration;

// Do not add java.awt.* here! It contains equally named classes that would conflict, e.g. Component.
import java.awt.Rectangle;
import java.io.*;
import java.lang.*;
import java.util.prefs.*;
import javax.swing.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.component.*;
import org.resmedicinae.resmedlib.component.context.*;
import org.resmedicinae.resmedlib.component.logger.*;
import org.resmedicinae.resmedlib.component.view.*;

/**
 * This class represents a configuration.
 *
 * It wraps <code>java.util.prefs.Preferences</code>, which offers a standard
 * mechanism for dealing with preference data in an application.
 *
 * This configuration tries to read (import) its values from a file.
 * If a file can not be found, the default values given in the get method
 * of each parameter will be used.
 *
 * Note the following length limitations:
 * - Maximum length of string allowed as a key (80 characters)
 * - Maximum length of a node name (80 characters)
 * - Maximum length of string allowed as a value (8192 characters)
 * That means, for storing a tree as one property, approximately 100 nodes
 * of the length 80 can be used. If the tree, i.e. length of all tree nodes
 * (absolute paths) added together is larger than that, the tree will have
 * to be split to store its branches in more than just one property.
 *
 * @version $Revision: 1.9 $ $Date: 2002/06/12 18:49:37 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Configuration extends Component {

    //
    // Preference keys for the controller.
    //

    /** The application's slogan. */
    static final String SLOGAN = "slogan";
    /** The application's description. */
    static final String DESCRIPTION = "description";
    /** The log level. */
    static final String LOG_LEVEL = "log_level";
    /** The logger output. */
    static final String LOGGER_OUTPUT = "logger_output";
    /** The view context. */
    static final String VIEW_CONTEXT = "view_context";
    /** The look and feel. */
    static final String LOOK_AND_FEEL = "look_and_feel";
    /** The display mode used to display this view. */
    static final String DISPLAY_MODE = "display_mode";

    //
    // Window parameters.
    //

    /** The x coordinate of the window. */
    static final String WINDOW_X = "window_x";
    /** The y coordinate of the window. */
    static final String WINDOW_Y = "window_y";
    /** The window width. */
    static final String WINDOW_WIDTH = "window_width";
    /** The window height. */
    static final String WINDOW_HEIGHT = "window_height";

    //
    // Dialog parameters.
    //

    /** The x coordinate of the dialog. */
    static final String DIALOG_X = "dialog_x";
    /** The y coordinate of the dialog. */
    static final String DIALOG_Y = "dialog_y";
    /** The dialog width. */
    static final String DIALOG_WIDTH = "dialog_width";
    /** The dialog height. */
    static final String DIALOG_HEIGHT = "dialog_height";

    //
    // Frame parameters.
    //

    /** The x coordinate of the frame. */
    static final String FRAME_X = "frame_x";
    /** The y coordinate of the frame. */
    static final String FRAME_Y = "frame_y";
    /** The frame width. */
    static final String FRAME_WIDTH = "frame_width";
    /** The frame height. */
    static final String FRAME_HEIGHT = "frame_height";

    //
    // Internal frame parameters.
    //

    /** The x coordinate of the internal frame. */
    static final String INTERNAL_FRAME_X = "internal_frame_x";
    /** The y coordinate of the internal frame. */
    static final String INTERNAL_FRAME_Y = "internal_frame_y";
    /** The internal frame width. */
    static final String INTERNAL_FRAME_WIDTH = "internal_frame_width";
    /** The internal frame height. */
    static final String INTERNAL_FRAME_HEIGHT = "internal_frame_height";

    //
    // Frame parameters.
    //

    /** The window title. */
    static final String TITLE = "title";
    /** The resizable flag. */
    static final String RESIZABLE = "resizable";

    //
    // Attributes.
    //

    /** The reference to the actual preferences object. */
    private Preferences preferences;

    /**
     * The location which can for example point to a:
     * - file
     * - database
     * - remote service
     */
    private String location;

    /** The file to import and export preferences from/to. */
    private File file;

    //
    // Preferences.
    //

    /**
     * Returns the preferences.
     *
     * @return the preferences
     */
    public Preferences getPreferences() {

        return this.preferences;
    }

    /**
     * Sets the preferences.
     *
     * @param preferences the preferences
     */
    public void setPreferences(Preferences preferences) {

        this.preferences = preferences;
    }

    //
    // Location.
    //

    /*
     * Returns the location.
     *
     * return the location
     */
    public String getLocation() {

        return this.location;
    }

    /*
     * Sets the location.
     *
     * @param location the location
     */
    public void setLocation(String location) {

        this.location = location;
    }

    //
    // File.
    //

    /*
     * Returns the file.
     *
     * return the file
     */
    public File getFile() {

        return this.file;
    }

    /*
     * Sets the file.
     *
     * @param file the file
     */
    public void setFile(File file) {

        this.file = file;
    }

    //
    // Contextualizable.
    //

    /**
     * Contextualizes this component.
     *
     * @param c the context
     * @exception NullPointerException if the location is null
     */
    public void contextualize(Context c) throws NullPointerException {

        if (c != null) {

            setPreferences((Preferences) c.getEntry("preferences"));
            setLocation((String) c.getEntry("location"));

        } else {

            throw new NullPointerException("Could not contextualize configuration. The context is null.");
        }
    }

    /**
     * Decontextualizes this component.
     *
     * @param c the context
     * @exception NullPointerException if the location is null
     */
    public void decontextualize(Context c) throws NullPointerException {

        if (c != null) {

            setPreferences(null);
            setLocation(null);

        } else {

            throw new NullPointerException("Could not contextualize configuration. The context is null.");
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() {

        setFile(new File(""/*??getLocation()*/));
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() {

        setFile(null);
    }

    //
    // Loadable.
    //

    /*
     * Load configuration data.
     */
    public void load() throws FileNotFoundException, IOException, InvalidPreferencesFormatException {

        InputStream is = new FileInputStream(getFile());

        getPreferences().importPreferences(is);
    }

    /*
     * Save configuration data.
     */
    public void save() throws FileNotFoundException, IOException, InvalidPreferencesFormatException, BackingStoreException {

        OutputStream os = new FileOutputStream(getFile());

        getPreferences().exportSubtree(os);
    }

    //
    // Slogan.
    //

    /*
     * Returns the slogan.
     *
     * @param s the default slogan
     * @return the slogan
     */
    public String getSlogan(String s) {

        return getPreferences().get(Configuration.SLOGAN, s);
    }

    /*
     * Sets the slogan.
     *
     * @param slogan the slogan
     */
    public void setSlogan(String slogan) {

        getPreferences().put(Configuration.SLOGAN, slogan);
    }

    //
    // Description.
    //

    /*
     * Returns the description.
     *
     * @param s the default description
     * @return the description
     */
    public String getDescription(String s) {

        return getPreferences().get(Configuration.DESCRIPTION, s);
    }

    /*
     * Sets the description.
     *
     * @param description the description
     */
    public void setDescription(String description) {

        getPreferences().put(Configuration.DESCRIPTION, description);
    }

    //
    // Log level.
    //

    /**
     * Returns the log level.
     *
     * @param def the default log level
     * @return the log level
     */
    public Level getLevel(Level def) {

        Level l = Level.ALL;
        String s = getPreferences().get(Configuration.LOG_LEVEL, def.toString());

        if (s.equals("OFF")) {

            l = Level.OFF;

        } else if (s.equals("FINEST")) {

            l = Level.FINEST;

        } else if (s.equals("FINER")) {

            l = Level.FINER;

        } else if (s.equals("FINE")) {

            l = Level.FINE;

        } else if (s.equals("CONFIG")) {

            l = Level.CONFIG;

        } else if (s.equals("INFO")) {

            l = Level.INFO;

        } else if (s.equals("WARNING")) {

            l = Level.WARNING;

        } else if (s.equals("SEVERE")) {

            l = Level.SEVERE;

        } else {

            l = Level.ALL;
        }

        return l;
    }

    /**
     * Sets the log level.
     *
     * @param level the log level
     */
    public void setLevel(Level level) {

        getPreferences().put(Configuration.LOG_LEVEL, level.toString());
    }

    //
    // Logger output.
    //

    /**
     * Returns the logger output.
     *
     * @param s the default logger output
     * @return the logger output
     */
    public String getLoggerOutput(String s) {

        return getPreferences().get(Configuration.LOGGER_OUTPUT, s);
    }

    /**
     * Sets the logger output.
     *
     * @param loggerOutput the logger output
     */
    public void setLoggerOutput(String loggerOutput) {

        getPreferences().put(Configuration.LOGGER_OUTPUT, loggerOutput);
    }

    //
    // View context.
    //

    /**
     * Returns the view context.
     *
     * @param i the default view context
     * @return the view context
     */
    public int getViewContext(int i) {

        return getPreferences().getInt(Configuration.VIEW_CONTEXT, i);
    }

    /**
     * Sets the view context.
     *
     * @param viewContext the view context
     */
    public void setViewContext(int viewContext) {

        getPreferences().putInt(Configuration.VIEW_CONTEXT, viewContext);
    }

    //
    // Look and feel.
    //

    /**
     * Returns the look and feel.
     *
     * @param def the default look and feel
     * @return the look and feel
     * @exception NullPointerException if the default look and feel is null
     * @exception NullPointerException if the look and feel id string is null
     * @exception Exception if the look and feel is invalid
     */
    public LookAndFeel getLookAndFeel(LookAndFeel def) throws Exception, NullPointerException {

        LookAndFeel laf = null;
        String id = "";

        if (def != null) {

            id = getPreferences().get(Configuration.LOOK_AND_FEEL, def.getID());

            if (id != null) {

                if (id.equals("Metal")) {

                    laf = new javax.swing.plaf.metal.MetalLookAndFeel();

                } else if (id.equals("Motif")) {

                    laf = new com.sun.java.swing.plaf.motif.MotifLookAndFeel();

/*??
                } else if (id.equals("Mac")) {

                    laf = new com.sun.java.swing.plaf.mac.MacLookAndFeel();
*/

                } else if (id.equals("Windows")) {

                    laf = new com.sun.java.swing.plaf.windows.WindowsLookAndFeel();

                } else {

                    throw new Exception("Could not get look and feel. The look and feel is invalid.");
                }

            } else {

                throw new NullPointerException("Could not get look and feel. The look and feel id string is null.");
            }

        } else {

            throw new NullPointerException("Could not get look and feel id string. The default look and feel is null.");
        }

        return laf;
    }

    /**
     * Sets the look and feel.
     *
     * @param laf the look and feel
     * @exception NullPointerException if the look and feel is null
     */
    public void setLookAndFeel(LookAndFeel laf) throws NullPointerException {

        if (laf != null) {

            getPreferences().put(Configuration.LOOK_AND_FEEL, laf.getID());

        } else {

            throw new NullPointerException("Could not set look and feel id string. The look and feel is null.");
        }
    }

    //
    // Display mode.
    //

    /**
     * Returns the display mode used to display this view.
     *
     * @param i the default display mode
     * @return the display mode used to display this view
     */
    public int getDisplayMode(int i) {

        return getPreferences().getInt(Configuration.DISPLAY_MODE, i);
    }

    /**
     * Sets the display mode used to display this view.
     *
     * @param displayMode the display mode used to display this view
     */
    public void setDisplayMode(int displayMode) {

        getPreferences().putInt(Configuration.DISPLAY_MODE, displayMode);
    }

    //
    // Window.
    //

    /**
     * Returns the window bounds.
     *
     * @param def the default window bounds
     * @return the window bounds
     * @exception NullPointerException if the default window bounds is null
     */
    public Rectangle getWindowBounds(Rectangle def) throws NullPointerException {

        Rectangle r = null;

        if (def != null) {

            int x = getWindowX((int) Math.floor(def.getX()));
            int y = getWindowY((int) Math.floor(def.getY()));
            int w = getWindowWidth((int) Math.floor(def.getWidth()));
            int h = getWindowHeight((int) Math.floor(def.getHeight()));

            r = new Rectangle(x, y, w, h);

        } else {

            throw new NullPointerException("Could not get window bounds. The default window bounds is null.");
        }

        return r;
    }

    /**
     * Sets the window bounds.
     *
     * @param windowBounds the window bounds
     * @exception NullPointerException if the window bounds is null
     */
    public void setWindowBounds(Rectangle windowBounds) throws NullPointerException {

        if (windowBounds != null) {

            setWindowX((int) Math.floor(windowBounds.getX()));
            setWindowY((int) Math.floor(windowBounds.getY()));
            setWindowWidth((int) Math.floor(windowBounds.getWidth()));
            setWindowHeight((int) Math.floor(windowBounds.getHeight()));

        } else {

            throw new NullPointerException("Could not set window bounds. The window bounds is null.");
        }
    }

    /**
     * Returns the x coordinate of the window.
     *
     * @param x the default x coordinate of the window
     * @return the x coordinate of the window
     */
    public int getWindowX(int x) {

        return getPreferences().getInt(Configuration.WINDOW_X, x);
    }

    /**
     * Sets the x coordinate of the window.
     *
     * @param x the x coordinate of the window
     */
    public void setWindowX(int x) {

        getPreferences().putInt(Configuration.WINDOW_X, x);
    }

    /**
     * Returns the y coordinate of the window.
     *
     * @param y the default y coordinate of the window
     * @return the y coordinate of the window
     */
    public int getWindowY(int y) {

        return getPreferences().getInt(Configuration.WINDOW_Y, y);
    }

    /**
     * Sets the y coordinate of the window.
     *
     * @param y the y coordinate of the window
     */
    public void setWindowY(int y) {

        getPreferences().putInt(Configuration.WINDOW_Y, y);
    }

    /**
     * Returns the window width.
     *
     * @param w the default window width
     * @return the window width
     */
    public int getWindowWidth(int w) {

        return getPreferences().getInt(Configuration.WINDOW_WIDTH, w);
    }

    /**
     * Sets the window width.
     *
     * @param w the window width
     */
    public void setWindowWidth(int w) {

        getPreferences().putInt(Configuration.WINDOW_WIDTH, w);
    }

    /**
     * Returns the window height.
     *
     * @param h the default window height
     * @return the window height
     */
    public int getWindowHeight(int h) {

        return getPreferences().getInt(Configuration.WINDOW_HEIGHT, h);
    }

    /**
     * Sets the window height.
     *
     * @param h the window height
     */
    public void setWindowHeight(int h) {

        getPreferences().putInt(Configuration.WINDOW_HEIGHT, h);
    }

    //
    // Dialog.
    //

    /**
     * Returns the dialog bounds.
     *
     * @param def the default dialog bounds
     * @return the dialog bounds
     * @exception NullPointerException if the default dialog bounds is null
     */
    public Rectangle getDialogBounds(Rectangle def) throws NullPointerException {

        Rectangle r = null;

        if (def != null) {

            int x = getDialogX((int) Math.floor(def.getX()));
            int y = getDialogY((int) Math.floor(def.getY()));
            int w = getDialogWidth((int) Math.floor(def.getWidth()));
            int h = getDialogHeight((int) Math.floor(def.getHeight()));

            r = new Rectangle(x, y, w, h);

        } else {

            throw new NullPointerException("Could not get dialog bounds. The default dialog bounds is null.");
        }

        return r;
    }

    /**
     * Sets the dialog bounds.
     *
     * @param r the dialog bounds
     * @exception NullPointerException if the dialog bounds is null
     */
    public void setDialogBounds(Rectangle r) throws NullPointerException {

        if (r != null) {

            setDialogX((int) Math.floor(r.getX()));
            setDialogY((int) Math.floor(r.getY()));
            setDialogWidth((int) Math.floor(r.getWidth()));
            setDialogHeight((int) Math.floor(r.getHeight()));

        } else {

            throw new NullPointerException("Could not set window bounds. The dialog bounds is null.");
        }
    }

    /**
     * Returns the x coordinate of the dialog.
     *
     * @param x the default x coordinate of the dialog
     * @return the x coordinate of the dialog
     */
    public int getDialogX(int x) {

        return getPreferences().getInt(Configuration.DIALOG_X, x);
    }

    /**
     * Sets the x coordinate of the dialog.
     *
     * @param x the x coordinate of the dialog
     */
    public void setDialogX(int x) {

        getPreferences().putInt(Configuration.DIALOG_X, x);
    }

    /**
     * Returns the y coordinate of the dialog.
     *
     * @param y the default y coordinate of the dialog
     * @return the y coordinate of the dialog
     */
    public int getDialogY(int y) {

        return getPreferences().getInt(Configuration.DIALOG_Y, y);
    }

    /**
     * Sets the y coordinate of the dialog.
     *
     * @param y the y coordinate of the dialog
     */
    public void setDialogY(int y) {

        getPreferences().putInt(Configuration.DIALOG_Y, y);
    }

    /**
     * Returns the dialog width.
     *
     * @param w the default dialog width
     * @return the dialog width
     */
    public int getDialogWidth(int w) {

        return getPreferences().getInt(Configuration.DIALOG_WIDTH, w);
    }

    /**
     * Sets the dialog width.
     *
     * @param w the dialog width
     */
    public void setDialogWidth(int w) {

        getPreferences().putInt(Configuration.DIALOG_WIDTH, w);
    }

    /**
     * Returns the dialog height.
     *
     * @param h the default dialog height
     * @return the dialog height
     */
    public int getDialogHeight(int h) {

        return getPreferences().getInt(Configuration.DIALOG_HEIGHT, h);
    }

    /**
     * Sets the dialog height.
     *
     * @param h the dialog height
     */
    public void setDialogHeight(int h) {

        getPreferences().putInt(Configuration.DIALOG_HEIGHT, h);
    }

    //
    // Frame.
    //

    /**
     * Returns the frame bounds.
     *
     * @param def the default frame bounds
     * @return the frame bounds
     * @exception NullPointerException if the default frame bounds is null
     */
    public Rectangle getFrameBounds(Rectangle def) throws NullPointerException {

        Rectangle r = null;

        if (def != null) {

            int x = getFrameX((int) Math.floor(def.getX()));
            int y = getFrameY((int) Math.floor(def.getY()));
            int w = getFrameWidth((int) Math.floor(def.getWidth()));
            int h = getFrameHeight((int) Math.floor(def.getHeight()));

            r = new Rectangle(x, y, w, h);

        } else {

            throw new NullPointerException("Could not get frame bounds. The default frame bounds is null.");
        }

        return r;
    }

    /**
     * Sets the frame bounds.
     *
     * @param r the frame bounds
     * @exception NullPointerException if the frame bounds is null
     */
    public void setFrameBounds(Rectangle r) throws NullPointerException {

        if (r != null) {

            setFrameX((int) Math.floor(r.getX()));
            setFrameY((int) Math.floor(r.getY()));
            setFrameWidth((int) Math.floor(r.getWidth()));
            setFrameHeight((int) Math.floor(r.getHeight()));

        } else {

            throw new NullPointerException("Could not set frame bounds. The frame bounds is null.");
        }
    }

    /**
     * Returns the x coordinate of the frame.
     *
     * @param x the default x coordinate of the frame
     * @return the x coordinate of the frame
     */
    public int getFrameX(int x) {

        return getPreferences().getInt(Configuration.FRAME_X, x);
    }

    /**
     * Sets the x coordinate of the frame.
     *
     * @param x the x coordinate of the frame
     */
    public void setFrameX(int x) {

        getPreferences().putInt(Configuration.FRAME_X, x);
    }

    /**
     * Returns the y coordinate of the frame.
     *
     * @param y the default y coordinate of the frame
     * @return the y coordinate of the frame
     */
    public int getFrameY(int y) {

        return getPreferences().getInt(Configuration.FRAME_Y, y);
    }

    /**
     * Sets the y coordinate of the frame.
     *
     * @param y the y coordinate of the frame
     */
    public void setFrameY(int y) {

        getPreferences().putInt(Configuration.FRAME_Y, y);
    }

    /**
     * Returns the frame width.
     *
     * @param w the default frame width
     * @return the frame width
     */
    public int getFrameWidth(int w) {

        return getPreferences().getInt(Configuration.FRAME_WIDTH, w);
    }

    /**
     * Sets the frame width.
     *
     * @param w the frame width
     */
    public void setFrameWidth(int w) {

        getPreferences().putInt(Configuration.FRAME_WIDTH, w);
    }

    /**
     * Returns the frame height.
     *
     * @param h the default frame height
     * @return the frame height
     */
    public int getFrameHeight(int h) {

        return getPreferences().getInt(Configuration.FRAME_HEIGHT, h);
    }

    /**
     * Sets the frame height.
     *
     * @param h the frame height
     */
    public void setFrameHeight(int h) {

        getPreferences().putInt(Configuration.FRAME_HEIGHT, h);
    }

    //
    // Internal Frame.
    //

    /**
     * Returns the internal frame bounds.
     *
     * @param def the default internal frame bounds
     * @return the internal frame bounds
     * @exception NullPointerException if the default internal frame bounds is null
     */
    public Rectangle getInternalFrameBounds(Rectangle def) throws NullPointerException {

        Rectangle r = null;

        if (def != null) {

            int x = getInternalFrameX((int) Math.floor(def.getX()));
            int y = getInternalFrameY((int) Math.floor(def.getY()));
            int w = getInternalFrameWidth((int) Math.floor(def.getWidth()));
            int h = getInternalFrameHeight((int) Math.floor(def.getHeight()));

            r = new Rectangle(x, y, w, h);

        } else {

            throw new NullPointerException("Could not get internal frame bounds. The default internal frame bounds is null.");
        }

        return r;
    }

    /**
     * Sets the internal frame bounds.
     *
     * @param r the internal frame bounds
     * @exception NullPointerException if the internal frame bounds is null
     */
    public void setInternalFrameBounds(Rectangle r) throws NullPointerException {

        if (r != null) {

            setInternalFrameX((int) Math.floor(r.getX()));
            setInternalFrameY((int) Math.floor(r.getY()));
            setInternalFrameWidth((int) Math.floor(r.getWidth()));
            setInternalFrameHeight((int) Math.floor(r.getHeight()));

        } else {

            throw new NullPointerException("Could not set internal frame bounds. The internal frame bounds is null.");
        }
    }

    /**
     * Returns the x coordinate of the internal frame.
     *
     * @param x the default x coordinate of the internal frame
     * @return the x coordinate of the internal frame
     */
    public int getInternalFrameX(int x) {

        return getPreferences().getInt(Configuration.INTERNAL_FRAME_X, x);
    }

    /**
     * Sets the x coordinate of the internal frame.
     *
     * @param x the x coordinate of the internal frame
     */
    public void setInternalFrameX(int x) {

        getPreferences().putInt(Configuration.INTERNAL_FRAME_X, x);
    }

    /**
     * Returns the y coordinate of the internal frame.
     *
     * @param y the default y coordinate of the internal frame
     * @return the y coordinate of the internal frame
     */
    public int getInternalFrameY(int y) {

        return getPreferences().getInt(Configuration.INTERNAL_FRAME_Y, y);
    }

    /**
     * Sets the y coordinate of the internal frame.
     *
     * @param y the y coordinate of the internal frame
     */
    public void setInternalFrameY(int y) {

        getPreferences().putInt(Configuration.INTERNAL_FRAME_Y, y);
    }

    /**
     * Returns the internal frame width.
     *
     * @param w the default internal frame width
     * @return the internal frame width
     */
    public int getInternalFrameWidth(int w) {

        return getPreferences().getInt(Configuration.INTERNAL_FRAME_WIDTH, w);
    }

    /**
     * Sets the internal frame width.
     *
     * @param w the internal frame width
     */
    public void setInternalFrameWidth(int w) {

        getPreferences().putInt(Configuration.INTERNAL_FRAME_WIDTH, w);
    }

    /**
     * Returns the internal frame height.
     *
     * @param h the default internal frame height
     * @return the internal frame height
     */
    public int getInternalFrameHeight(int h) {

        return getPreferences().getInt(Configuration.INTERNAL_FRAME_HEIGHT, h);
    }

    /**
     * Sets the internal frame height.
     *
     * @param h the internal frame height
     */
    public void setInternalFrameHeight(int h) {

        getPreferences().putInt(Configuration.INTERNAL_FRAME_HEIGHT, h);
    }

    //
    // Resizable.
    //

    /*
     * Returns the resizable flag.
     *
     * @param def the default resizable flag
     * @return the resizable flag
     */
    public boolean getResizable(boolean def) {

        return getPreferences().getBoolean(Configuration.RESIZABLE, def);
    }

    /*
     * Sets the resizable flag.
     *
     * @param resizable the resizable flag
     */
    public void setResizable(boolean resizable) {

        getPreferences().putBoolean(Configuration.RESIZABLE, resizable);
    }

    //
    // Title.
    //

    /*
     * Returns the title.
     *
     * @param def the default title
     * @return the title
     */
    public String getTitle(String def) {

        return getPreferences().get(Configuration.TITLE, def);
    }

    /*
     * Sets the title.
     *
     * @param title the title
     */
    public void setTitle(String title) {

        getPreferences().put(Configuration.TITLE, title);
    }
}

