/*
 * $RCSfile: Item.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All Rights Reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.component;

import java.lang.*;
import java.util.*;

/**
 * This class represents a hierarchical item.
 *
 * An item can contain a number of child items.
 * It can also be a child itself, having a parent item.
 *
 * @version $Revision: 1.5 $ $Date: 2002/06/12 18:49:37 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Item extends Object {

    //
    // Constants.
    //

    /** The role constant. */
    public static final String ROLE = "Unspecified role";

    //
    // Attributes.
    //

    /** The role identifying this item. */
    private String role;

    /** The parent. */
    private Item parent;

    /** The children. */
    private List children;

    //
    // Role.
    //

    /**
     * Returns the role.
     *
     * @return the role
     */
    public String getRole() {

        return "Unspecified role";
    }

    //
    // Parent.
    //

    /**
     * Returns the parent.
     *
     * @return the parent
     */
    public Item getParent() {

        return this.parent;
    }

    /**
     * Sets the parent.
     *
     * @param parent the parent
     */
    public void setParent(Item parent) {

        this.parent = parent;
    }

    /**
     * Creates a parent.
     *
     * @return the parent
     */
    public Item createParent() {

        return null;
    }

    /**
     * Destroys the parent.
     *
     * @param parent the parent
     */
    public void destroyParent(Item parent) {
    }

    //
    // Children list.
    //

    /**
     * Returns the children list.
     *
     * @return the children list
     */
    public List getChildren() {

        return this.children;
    }

    /**
     * Sets the children list.
     *
     * @param children the children list
     */
    public void setChildren(List children) {

        this.children = children;
    }

    /**
     * Creates a children list.
     *
     * @return the children list
     */
    public List createChildren() {

        return new Vector();
    }

    /**
     * Destroys the children list.
     *
     * @param l the children list
     */
    public void destroyChildren(List l) {

        if (l != null) {

            Iterator iter = l.iterator();

            if (iter != null) {

                Item item;

                while (iter.hasNext()) {

                    item = (Item) iter.next();

                    l.remove(item);
                }

            } else {

                throw new NullPointerException("Could not destroy the children list. The list iterator is null.");
            }

        } else {

            throw new NullPointerException("Could not destroy the children list. The list is null.");
        }
    }

    //
    // Child management.
    //

    /**
     * Returns the child with the specified role.
     *
     * @param r the role
     * @exception NullPointerException if the children container is null
     * @exception NullPointerException if the list iterator is null
     * @exception NullPointerException if a list item is null
     * @exception NullPointerException if a list item's role is null
     */
    public Item get(String r) throws NullPointerException {

        Item i = null;
        List l = getChildren();

        if (l != null) {

            Iterator iter = l.iterator();

            if (iter != null) {

                Item item = null;
                String role;

                while (iter.hasNext()) {

                    item = (Item) iter.next();

                    if (item != null) {

                        role = item.getRole();

                        if (role != null) {

                            if (role.equals(r)) {

                                i = item;
                                break;
                            }

                        } else {

                            throw new NullPointerException("Could not get item. A list item's role is null.");
                        }

                    } else {

                        throw new NullPointerException("Could not get item. A list item is null.");
                    }
                }

            } else {

                throw new NullPointerException("Could not get item. The list iterator is null.");
            }

        } else {

            throw new NullPointerException("Could not get item. The children container is null.");
        }

        return i;
    }

    /**
     * Returns the child with the specified index.
     *
     * @param index the index
     * @exception NullPointerException if the children container is null
     */
    public Item get(int index) throws NullPointerException {

        Item i = null;
        List l = getChildren();

        if (l != null) {

            i = (Item) l.get(index);

        } else {

            throw new NullPointerException("Could not get item. The children container is null.");
        }

        return i;
    }

    /**
     * Adds the item to be a child of this item.
     *
     * Removes item from old parent.
     * Adds item to this item being the new parent.
     *
     * Appends the item to the list of children.
     *
     * @param i the item
     */
    public void add(Item i) {

        // Determine the list size to hand over last position as new index.
        add(i, getChildCount());
    }

    /**
     * Adds the item to be a child of this item.
     *
     * Removes item from old parent.
     * Adds item to this item being the new parent.
     *
     * Inserts the item at the specified index into the list of children.
     *
     * @param i the item
     * @param index the index
     */
    public void add(Item i, int index) {

        // Replace flag is set to false, by default.
        // An existing item at index will not be replaced.
        add(i, index, false);
    }

    /**
     * Adds the item to be a child of this item.
     *
     * Removes item from old parent.
     * Adds item to this item being the new parent.
     *
     * Inserts the item at the specified index into the list of children.
     * Replaces existing item at index, if replace flag is set to true.
     *
     * @param i the item
     * @param index the index
     * @param replace the replace flag
     * @exception NullPointerException if the item is null
     * @exception NullPointerException if the children container is null
     */
    public void add(Item i, int index, boolean replace) throws NullPointerException {

        if (i != null) {

            // Remove item from old parent.
            Item parent = (Item) i.getParent();

            if (parent != null) {

                parent.remove(i);
            }

            // Add item to this item being the new parent.
            List l = getChildren();

            if (l != null) {

                if (replace == false) {

                    ((Vector) l).insertElementAt(i, index);

                } else {

                    // Replace the item at index with the given item.
                    l.set(index, i);
                }

                // Set new parent for item.
                i.setParent(this);

            } else {

                throw new NullPointerException("Could not add item. The children container is null.");
            }

        } else {

            throw new NullPointerException("Could not add item. The item is null.");
        }
    }

    /**
     * Removes the item from this parent item.
     *
     * @param i the item
     * @exception NullPointerException if the item is null
     * @exception NullPointerException if the children container is null
     */
    public void remove(Item i) throws NullPointerException {

        if (i != null) {

            // Remove item from this parent item.
            List l = getChildren();

            if (l != null) {

                l.remove(i);

                // Set parent of item to null.
                i.setParent(null);

            } else {

                throw new NullPointerException("Could not remove item. The children container is null.");
            }

        } else {

            throw new NullPointerException("Could not remove item. The item is null.");
        }
    }

    /**
     * Returns the number of children.
     *
     * @return the number of children
     * @exception NullPointerException if the children container is null
     */
    public int getChildCount() throws NullPointerException {

        int count = 0;
        List l = getChildren();

        if (l != null) {

            count = l.size();

        } else {

            throw new NullPointerException("Could not get child count. The children container is null.");
        }

        return count;
    }

    //
    // Parent management.
    //

    /**
     * Returns the root.
     *
     * @return the root
     * @exception NullPointerException if the parent item is null
     */
    public Item getRoot() throws NullPointerException {

        Item root = null;
        Item parent = this;

        // Loop through parents of this item until no more parent is found (null).
        // Return the last valid parent (root) whose parent is null.
        do {

            root = parent;

            if (parent != null) {

                parent = (Item) parent.getParent();

            } else {

                throw new NullPointerException("Could not get root. This parent item is null.");
            }

        } while (parent != null);

        return root;
    }

    //
    // Initializable.
    //

    /**
     * Initializes this item.
     */
    public void initialize() throws Exception {

        setChildren(createChildren());
    }

    /**
     * Finalizes this item.
     */
    public void finalizz() throws Exception {

        destroyChildren(getChildren());
        setChildren(null);
    }
}

