/*
 * $RCSfile: Component.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All Rights Reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib.component.component;

import java.lang.*;
import java.util.prefs.*;
import org.resmedicinae.resmedlib.component.component.*;
import org.resmedicinae.resmedlib.component.configuration.*;
import org.resmedicinae.resmedlib.component.context.*;
import org.resmedicinae.resmedlib.component.logger.*;

/**
 * This class represents a component.
 *
 * A component is an object that has well-defined public methods useable by outside entities.
 * Such outside entities are sometimes called "container".
 * They have a contract with the component which says that the public methods
 * are called in a specific order, called "component lifecycle".
 *
 * By calling the lifecycle methods, a container ensures the proper startup and shutdown
 * of the component and of the underlying Framework, to work correctly.
 * One or more methods can be called by the container, but the order must be the same:
 *  contextualize(Context c)
 *  compose(ComponentManager m)
 *  configure(ConfigurationManager m)
 *  initialize()
 *  start()
 *      suspend()
 *      recontextualize(Context c)
 *      recompose(ComponentManager m)
 *      reinitialize()
 *      resume()
 *  stop()
 *  finalize()
 *  deconfigure(ConfigurationManager m)
 *  decompose(ComponentManager m)
 *  decontextualize(Context c)
 *
 * The component should indicate which functionality (public methods) it offers
 * by implementing special interfaces (concerns). A concern is something like an aspect,
 * some behaviour the component has, some role/task that the component is capable to fulfill.
 * For each of the lifecycle methods listed above, there is a concern with a similar name:
 *  Contextualizable
 *  Composeable
 *  Configurable
 *  Initializable
 *  Startable
 *  Suspendable
 *
 * The component also contains all functionality to manage the lifecycle of any child components.
 * In other words, this component can be used by other containers but also
 * act as a container itself, having children. It such is able to bear or kill
 * child components with the corresponding methods <code>bear</code> and <code>kill</code>.
 *
 * @version $Revision: 1.10 $ $Date: 2002/08/08 21:55:15 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Component extends Item {

    //
    // Lifecycle attributes.
    //

    /** The context. */
    private Context context;

    /** The component manager. */
    private ComponentManager componentManager;

    /** The configuration manager. */
    private ConfigurationManager configurationManager;

    //
    // Context.
    //

    /**
     * Returns the context.
     *
     * @return the context
     */
    public Context getContext() {

        return this.context;
    }

    /**
     * Sets the context.
     *
     * @param context the context
     */
    public void setContext(Context context) {

        this.context = context;
    }

    //
    // Component Manager.
    //

    /**
     * Returns the component manager.
     *
     * @return the component manager
     */
    public ComponentManager getComponentManager() {

        return this.componentManager;
    }

    /**
     * Sets the component manager.
     *
     * @param componentManager the component manager
     */
    public void setComponentManager(ComponentManager componentManager) {

        this.componentManager = componentManager;
    }

    //
    // Configuration Manager.
    //

    /**
     * Returns the configuration manager.
     *
     * @return the configuration manager
     */
    public ConfigurationManager getConfigurationManager() {

        return this.configurationManager;
    }

    /**
     * Sets the configuration manager.
     *
     * @param configurationManager the configuration manager
     */
    public void setConfigurationManager(ConfigurationManager configurationManager) {

        this.configurationManager = configurationManager;
    }

    //
    // Logger.
    //

    /**
     * Returns the logger.
     *
     * @return the logger
     */
    public Logger getLogger() {

        Logger logger = null;
        ComponentManager m = getComponentManager();
        
        if (m != null) {
            
            logger = (Logger) m.get(Logger.ROLE);
        }
        
        return logger;
    }

    /**
     * Sets the logger.
     *
     * @param logger the logger
     * @exception NullPointerException if the component manager is null
     */
    public void setLogger(Logger logger) throws NullPointerException {

        ComponentManager m = getComponentManager();
        
        if (m != null) {
            
            m.add(logger);
            
        } else {
            
            throw new NullPointerException("Could not set logger. The component manager is null.");
        }
    }

    //
    // Contextualizable.
    //

    /**
     * Contextualizes this component.
     *
     * @param c the context
     */
    public void contextualize(Context c) throws Exception {
        
        setContext(c);
    }

    /**
     * Decontextualizes this component.
     *
     * @param c the context
     */
    public void decontextualize(Context c) throws Exception {
        
        setContext(null);
    }

    //
    // Composable.
    //

    /**
     * Composes this component.
     *
     * @param m the component manager
     */
    public void compose(ComponentManager m) throws Exception {
        
        setComponentManager(m);
    }

    /**
     * Decomposes this component.
     *
     * @param m the component manager
     */
    public void decompose(ComponentManager m) throws Exception {
        
        setComponentManager(null);
    }

    //
    // Configurable.
    //

    /**
     * Configures this component.
     *
     * @param m the configuration manager
     * @exception NullPointerException if the configuration manager is null
     */
    public void configure(ConfigurationManager m) throws Exception, NullPointerException {

        setConfigurationManager(m);

        if (m != null) {

            configureSystemSettings(m.getSystemConfiguration());
            configureLocalSettings(m.getLocalConfiguration());
            configureUserSettings(m.getUserConfiguration());

        } else {

            throw new NullPointerException("Could not configure application controller. The configuration manager is null.");
        }
    }

    /**
     * Deconfigures this component.
     *
     * @param m the configuration manager
     * @exception NullPointerException if the configuration manager is null
     */
    public void deconfigure(ConfigurationManager m) throws Exception, NullPointerException {

        if (m != null) {

            deconfigureUserSettings(m.getUserConfiguration());
            deconfigureLocalSettings(m.getLocalConfiguration());
            deconfigureSystemSettings(m.getSystemConfiguration());

        } else {

            throw new NullPointerException("Could not deconfigure application controller. The configuration manager is null.");
        }

        setConfigurationManager(null);
    }

    /**
     * Configures the system settings.
     *
     * @param c the system configuration
     */
    protected void configureSystemSettings(Configuration c) throws Exception {
    }

    /**
     * Deconfigures the system settings.
     *
     * @param c the system configuration
     */
    protected void deconfigureSystemSettings(Configuration c) throws Exception {
    }

    /**
     * Configures the local settings.
     *
     * @param c the local configuration
     */
    protected void configureLocalSettings(Configuration c) throws Exception {
    }

    /**
     * Deconfigures the local settings.
     *
     * @param c the local configuration
     */
    protected void deconfigureLocalSettings(Configuration c) throws Exception {
    }

    /**
     * Configures the user settings.
     *
     * @param c the user configuration
     */
    protected void configureUserSettings(Configuration c) throws Exception {
    }

    /**
     * Deconfigures the user settings.
     *
     * @param c the user configuration
     */
    protected void deconfigureUserSettings(Configuration c) throws Exception {
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     */
    public void initialize() throws Exception {
        
        super.initialize();
    }

    /**
     * Finalizes this component.
     */
    public void finalizz() throws Exception {
        
        super.finalizz();
    }

    //
    // Startable.
    //

    /**
     * Starts this component.
     */
    public void start() throws Exception {
    }

    /**
     * Stops this component.
     */
    public void stop() throws Exception {
    }

    //
    // Suspendable.
    //

    /**
     * Suspends this component.
     */
    public void suspend() throws Exception {
    }

    /**
     * Resumes this component.
     */
    public void resume() throws Exception {
    }

    /**
     * Recontextualizes this component.
     */
    public void recontextualize() throws Exception {
    }

    /**
     * Recomposes this component.
     */
    public void recompose() throws Exception {
    }

    /**
     * Reconfigures this component.
     */
    public void reconfigure() throws Exception {
    }

    //
    // Loggable.
    //

    /**
     * Logs a message with no arguments.
     *
     * @param lev the level
     * @param msg the message
     */
    public void log(Level lev, String msg) throws Exception {

        log(lev, msg, null);
    }

    /**
     * Logs a message with associated throwable information.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
    public void log(Level lev, String msg, Throwable t) throws Exception {

        Logger l = getLogger();

        if (l != null) {

            if (t != null) {

                l.log(lev, msg, t);
                t.printStackTrace();

            } else {

                l.log(lev, msg, null);
            }

        } else {

            System.out.println("Could not log message. The logger is null. Will use system console for message output.");

            if (t != null) {

                System.out.println(this + " log\n" + lev + ": " + msg + "\n" + t);
                t.printStackTrace();

            } else {

                System.out.println(this + " log\n" + lev + ": " + msg);
            }
        }
    }

    //
    // Child birth and death.
    //

    /**
     * Bears the component.
     *
     * Takes the component through its birth process.
     * A component gets born by calling the following methods:
     *  [constructor(): called before, outside this method, without any parameters, to hand over the instance to this method]
     *  contextualize(Context c): handing over specific parameters from container to component
     *  compose(ComponentManager m): reuse general components of the parent container to save time and memory
     *  configure(ConfigurationManager m): configure component settings with a configuration manager
     *  initialize(): allocates necessary memory, e.g. a List attribute: this.list = new Vector();
     *  start(): activates the component, e.g. a loop of a server component that is waiting for requests; should call "run" method of java.lang.Runnable
     *      suspend()
     *      recontextualize(Context c)
     *      recompose(ComponentManager m)
     *      reinitialize()
     *      resume()
     *
     * These lifecycle methods are only called, if the component
     * has implemented the necessary interfaces (concerns):
     *  Contextualizable
     *  Composable
     *  Configurable
     *  Initializable
     *  Startable
     *  Suspendable
     *
     * Design patterns:
     * - Template method
     *
     * @param c the component
     * @exception NullPointerException if the component is null
     */
    protected void bear(Component c) throws Exception, NullPointerException {

        if (c != null) {

            log(Level.INFO, "Contextualize the component.");
            c.contextualize(getContext());

            log(Level.INFO, "Compose the component.");
            c.compose(getComponentManager());

            log(Level.INFO, "Configure the component.");
            c.configure(getConfigurationManager());

            log(Level.INFO, "Initialize the component.");
            c.initialize();
            
            log(Level.INFO, "Start the component.");
            c.start();

        } else {

            throw new NullPointerException("Could not bear component. The component is null.");
        }
    }

    /**
     * Kills the component.
     *
     * Takes the component through its death process.
     * A component gets killed by calling the following methods:
     *  stop(): deactivates the component
     *  finalize(): deallocates memory
     *  deconfigure(ConfigurationManager m): writes settings into a configuration file or database
     *  decompose(ComponentManager m)
     *  decontextualize
     *  [destructor(): called afterwards, outside this method, if not used anymore, by the Java garbage collector]
     *
     * These lifecycle methods are only called, if the component
     * has implemented the necessary interfaces (concerns):
     *  Contextualizable
     *  Composable
     *  Configurable
     *  Initializable
     *  Startable
     *  Suspendable
     *
     * Design patterns:
     * - Template method
     *
     * @param c the component
     * @exception NullPointerException if the component is null
     */
    protected void kill(Component c) throws Exception, NullPointerException {

        if (c != null) {

            log(Level.INFO, "Stop the component.");
            c.stop();
            
            log(Level.INFO, "Finalize the component.");
            c.finalizz();
            
            log(Level.INFO, "Deconfigure the component.");
            c.deconfigure(getConfigurationManager());

            log(Level.INFO, "Decompose the component.");
            c.decompose(getComponentManager());

            log(Level.INFO, "Decontextualize the component.");
            c.decontextualize(getContext());

        } else {

            throw new NullPointerException("Could not kill component. The component is null.");
        }
    }
}

