/*
 * $RCSfile: Item.java,v $
 *
 * Copyright (c) 1999-2002. The Res Medicinae developers. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib;

/**
 * This class represents a general abstract item.<br><br>
 *
 * It is the super-super class of all level classes in the framework's ontology:<br>
 *  <ul>
 *      <li>Network</li>
 *      <li>Family</li>
 *      <li>System</li>
 *      <li>Block</li>
 *      <li>Region</li>
 *      <li>Component</li>
 *      <li>Part</li>
 *      <li>Chain</li>
 *      <li>Term</li>
 *      <li>Sign</li>
 *      <li>Number</li>
 *      <li>Digit</li>
 *  </ul>
 *
 * An item consists of a number of child items. It can also be a child itself,
 * belonging to a parent item.<br><br>
 *
 * Since humans don't know where this hierarchy stops in the real world
 * - to microcosm as well as towards macrocosm -, a black box has to be assumed
 * at some poorg.resmedicinae.resmedlib.number.Integer towards microcosm. In the case of programming, this is mostly
 * the basic items of programming language, namely Boolean and Numbers.
 * They are coded in binaries within the machine and that is the final abstraction.
 * Towards macrocosm, it depends on the modeller how far he/she wants to go.
 * At the time of writing this, the ResMedLib framework's highest class in its
 * ontology is Network, consisting of Families which consist of Systems.<br><br>
 *
 * An item has well-defined public methods useable by outside entities. An outside
 * entity which contains items is called their container. Containers have a contract
 * with the items they contain, in that public methods are called in a specific order,
 * called a lifecycle.<br><br>
 *
 * By calling the lifecycle methods, a container ensures the proper startup and
 * shutdown of each item, to work correctly. One or more methods can be called
 * by the container, but the order must be the same.<br>
 *
 * An item has the following lifecycle:
 *  <ul>
 *      <li>constructor(): called without any parameters on keyword <code>new</code></li>
 *      <li>initialize(): creates items (attributes) that are specific to this item; allocates necessary memory</li>
 *      <li>finalize(): destroys items (attributes) that are specific to this item; deallocates memory</li>
 *      <li>destructor(): called without any parameters; not available for Java where a garbage collector destroys objects incidentally!</li>
 *  </ul>
 *
 * @version $Revision: 1.27 $ $Date: 2002/12/10 17:27:32 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Item extends java.lang.Object {

    //
    // Attributes.
    //

    /** The children. */
    private java.util.Map children;

    /** The java tree node. */
    private javax.swing.tree.DefaultMutableTreeNode javaTreeNode;

    /** The encapsulated java object. */
    private java.lang.Object javaObject;

    /** The identifier. */
    private org.resmedicinae.resmedlib.term.String identifier;

    /** The parent. */
    private Item parent;

    //
    // Constructor.
    //

    /**
     * Constructs this item.
     *
     * Actually, constructors should NOT be used in the ResMedLib framework
     * or any derived systems. The offered lifecycle methods are sufficient.
     * However, since the children structure (container) is not really
     * a child item itself but rather a helper attribute to store children,
     * it is created in this constructor.
     * In some way, it can be seen as meta property of (every) item as it
     * makes items hierarchical, just like in the universe, the real world.
     *
     * The items created here are destroyed in the finalize method
     * as Java does not provide destructors for its objects.
     */
    public Item() {

        setChildren(createChildren());

        // This java tree node is equivalent to the above children structure (container)
        // in that it can contain children.
        // It is only used as long as Java objects are used in the ResMedLib framework,
        // to build up a system (dependency) tree using java tree nodes
        // so that such trees can easily be displayed in javax.swing.JTree objects.
        setJavaTreeNode(createJavaTreeNode());

        // As long as the ResMedLib framework is built on Java, every
        // item needs to be capable of encapsulating a pure Java object.
        //
        // This method is not called in the initialize lifecycle method
        // since then one would especially need to call initialize
        // even for every primitive item (like String), being created.
        setJavaObject(createJavaObject());
    }

    //
    // Children structure.
    //

    /**
     * Creates a children structure.
     *
     * @return the children structure
     */
    public java.util.Map createChildren() {

        return new java.util.HashMap();
    }

    /**
     * Destroys the children structure.
     *
     * @param s the children structure
     */
    public void destroyChildren(java.util.Map s) {
    }

    /**
     * Sets the children structure.
     *
     * @param children the children structure
     */
    public void setChildren(java.util.Map children) {

        this.children = children;
    }

    /**
     * Returns the children structure.
     *
     * @return the children structure
     */
    public java.util.Map getChildren() {

        return this.children;
    }

    //
    // Java tree node.
    //
    
    /**
     * Creates a java tree node.
     *
     * @return the java tree node
     * @exception NullPointerException if the java tree node is null
     */
    public javax.swing.tree.DefaultMutableTreeNode createJavaTreeNode() throws NullPointerException {

        javax.swing.tree.DefaultMutableTreeNode tn = new javax.swing.tree.DefaultMutableTreeNode();

        if (tn != null) {

            tn.setUserObject("Item");

        } else {

            throw new NullPointerException("Could not create java tree node. The java tree node is null.");
        }

        return tn;
    }

    /**
     * Destroys the java tree node.
     *
     * @param tn the java tree node
     */
    public void destroyJavaTreeNode(javax.swing.tree.DefaultMutableTreeNode tn) {
    }

    /**
     * Sets the java tree node.
     *
     * @param tn the java tree node
     */
    public void setJavaTreeNode(javax.swing.tree.DefaultMutableTreeNode tn) {

        this.javaTreeNode = tn;
    }

    /**
     * Returns the java tree node.
     *
     * @return the java tree node
     */
    public javax.swing.tree.DefaultMutableTreeNode getJavaTreeNode() {

        return this.javaTreeNode;
    }

    //
    // Encapsulated java object.
    //
    
    /**
     * Creates an encapsulated java object.
     *
     * @return the encapsulated java object
     * @exception NullPointerException if the encapsulated java object is null
     */
    public java.lang.Object createJavaObject() {

        return null;
    }

    /**
     * Destroys the encapsulated java object.
     *
     * @param o the encapsulated java object
     */
    public void destroyJavaObject(java.lang.Object o) {
    }

    /**
     * Sets the encapsulated java object.
     *
     * @param o the encapsulated java object
     */
    public void setJavaObject(java.lang.Object o) {

        this.javaObject = o;
    }

    /**
     * Returns the encapsulated java object.
     *
     * @return the encapsulated java object
     */
    public java.lang.Object getJavaObject() {

        return this.javaObject;
    }

    //
    // Identifier.
    //
    
    public org.resmedicinae.resmedlib.term.String getIdentifier() {

        return identifier;
    }

    public void setIdentifier(org.resmedicinae.resmedlib.term.String identifier) {

        this.identifier = identifier;
    }

    //
    // Parent.
    //

    /**
     * Creates a parent.
     *
     * @return the parent
     */
    public Item createParent() {

        return null;
    }

    /**
     * Destroys the parent.
     *
     * @param parent the parent
     */
    public void destroyParent(Item parent) {
    }

    /**
     * Sets the parent.
     *
     * @param parent the parent
     */
    public void setParent(Item parent) {

        this.parent = parent;
    }

    /**
     * Returns the parent.
     *
     * @return the parent
     */
    public Item getParent() {

        return this.parent;
    }

    //
    // Child management.
    //

    /**
     * Adds the item to become a child of this item.
     *
     * @param n the name
     * @param i the item
     * @exception NullPointerException if the children structure is null
     * @exception NullPointerException if the name is null
     * @exception NullPointerException if the item is null
     */
    public void set(org.resmedicinae.resmedlib.term.String n, Item i) throws NullPointerException {

        java.util.Map s = getChildren();

        if (s != null) {

            if (n != null) {

                if (i != null) {

                    s.put(n.getJavaObject(), i);

                    addTreeNode(i);

                    //?? Remove this line later! Children shouldn't know about their parent. Security!
                    i.setParent(this);

                } else {

                    java.lang.System.out.println("DEBUG: Could not set item. The item is null.");
                }

            } else {

                throw new NullPointerException("Could not set item. The name is null.");
            }

        } else {

            throw new NullPointerException("Could not set item. The children structure is null.");
        }
    }

    /**
     * Removes the child item from this item.
     *
     * @param n the name
     * @exception NullPointerException if the children structure is null
     * @exception NullPointerException if the name is null
     */
    public void remove(org.resmedicinae.resmedlib.term.String n) throws NullPointerException {

        java.util.Map s = getChildren();

        if (s != null) {

            if (n != null) {

                removeTreeNode(get(n));

                s.remove(n.getJavaObject());

            } else {

                throw new NullPointerException("Could not remove item. The name is null.");
            }

        } else {

            throw new NullPointerException("Could not remove item. The children structure is null.");
        }
    }

    /**
     * Returns the child with the given name.
     *
     * @param n the name
     * @exception NullPointerException if the children structure is null
     * @exception NullPointerException if the name is null
     */
    public Item get(org.resmedicinae.resmedlib.term.String n) throws NullPointerException {

        Item i = null;
        java.util.Map s = getChildren();

        if (s != null) {

            if (n != null) {

                i = (Item) s.get(n.getJavaObject());

            } else {

                throw new NullPointerException("Could not get item. The name is null.");
            }

        } else {

            throw new NullPointerException("Could not get item. The children structure is null.");
        }

        return i;
    }

    /**
     * Returns the number of children.
     *
     * @return the number of children
     * @exception NullPointerException if the children structure is null
     */
    public org.resmedicinae.resmedlib.number.Integer getChildCount() throws NullPointerException {

        org.resmedicinae.resmedlib.number.Integer c = null;
        java.util.Map s = getChildren();

        if (s != null) {

            c = new org.resmedicinae.resmedlib.number.Integer(s.size());

        } else {

            throw new NullPointerException("Could not get child count. The children structure is null.");
        }

        return c;
    }

    //
    // Java tree node management.
    //

    /**
     * Adds the java tree node of the item.
     *
     * @param item the item
     * @exception NullPointerException if the java tree node is null
     * @exception NullPointerException if the item is null
     */
    public void addTreeNode(Item item) throws NullPointerException {

        javax.swing.tree.DefaultMutableTreeNode tn = getJavaTreeNode();

        if (tn != null) {
    
            if (item != null) {

                tn.add(item.getJavaTreeNode());

            } else {
    
                throw new NullPointerException("Could not set item. The item is null.");
            }
    
        } else {
    
            throw new NullPointerException("Could not set item. The java tree node is null.");
        }
    }

    /**
     * Removes the java tree node of the item.
     *
     * @param item the item
     * @exception NullPointerException if the java tree node is null
     * @exception NullPointerException if the item is null
     */
    public void removeTreeNode(Item item) throws NullPointerException {

        javax.swing.tree.DefaultMutableTreeNode tn = getJavaTreeNode();

        if (tn != null) {
    
            if (item != null) {

                tn.remove(item.getJavaTreeNode());

            } else {
    
                throw new NullPointerException("Could not set item. The item is null.");
            }

        } else {

            throw new NullPointerException("Could not set item. The java tree node is null.");
        }
    }

    //
    // Parent management.
    //

    /**
     * Returns the root.
     *
     * @return the root
     * @exception NullPointerException if the initial parent item is null
     */
    public Object getRoot() throws NullPointerException {

        Item root = null;
        Item parent = this;

        // Loop through parents of this item until no more parent is found (null).
        // Return the last valid parent (root) whose parent is null.
        do {

            root = parent;

            if (parent != null) {

                parent = (Item) parent.getParent();

            } else {

                throw new NullPointerException("Could not get root. This initial parent item is null.");
            }

        } while (parent != null);

        return root;
    }

    //
    // Initializable.
    //

    /**
     * Initializes this item.
     */
    public void initialize() throws Exception {

        setParent(createParent());
    }

    /**
     * Finalizes this item.
     *
     * The method's name is <code>finalizz</code> and not
     * <code>finalize</code> because java's top most class
     * <code>Object</code> already uses <code>finalize</code>.
     */
    public void finalizz() throws Exception {

        destroyParent(getParent());
        setParent(null);

        destroyJavaObject(getJavaObject());
        setJavaObject(null);

        destroyJavaTreeNode(getJavaTreeNode());
        setJavaTreeNode(null);

        destroyChildren(getChildren());
        setChildren(null);
    }
}

