/*
 * $RCSfile: Component.java,v $
 *
 * Copyright (c) 1999-2002. The Res Medicinae developers. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package org.resmedicinae.resmedlib;

import org.resmedicinae.resmedlib.chain.*;
import org.resmedicinae.resmedlib.term.*;

/**
 * This class represents a component.
 *
 * It is a super class and forms an own level in the framework's ontology:
 *  <ul>
 *      <li>Network</li>
 *      <li>Family</li>
 *      <li>System</li>
 *      <li>Block</li>
 *      <li>Region</li>
 *      <li>Component</li>
 *      <li>Part</li>
 *      <li>Chain</li>
 *      <li>Term</li>
 *      <li>Sign</li>
 *      <li>Number</li>
 *      <li>Digit</li>
 *  </ul>
 *
 * A component corresponds to a cell in biology.
 *
 * A component has the following lifecycle:
 *  <ul>
 *      <li>constructor(): called without any parameters on keyword <code>new</code></li>
 *      <li>globalize(Structure globals): hands over global items for reuse to save time and memory</li>
 *      <li>[several local set methods]: hands over local items (context parameters)</li>
 *      <li>initialize(): creates items (attributes) that are specific to this item; allocates necessary memory</li>
 *      <li>configure(): adapts the properties of an item to the preferences of a user; read from a source</li>
 *      <li>link(): associates children of an item among each other</li>
 *      <li>unlink(): cuts associations between children of an item</li>
 *      <li>deconfigure(): writes system settings to configuration and makes it persistent</li>
 *      <li>finalize(): destroys items (attributes) that are specific to this item; deallocates memory</li>
 *      <li>destructor(): called without any parameters; not available for Java where a garbage collector destroys objects incidentally!</li>
 *  </ul>
 *
 * The <code>globalize</code> method must be called before <code>initialize</code>
 * because some global parameters (such as the configuration) need to be forwarded
 * to children. 
 *
 * @version $Revision: 1.17 $ $Date: 2002/12/18 17:08:14 $ $Author: chrissy $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class Component extends Part {

    //
    // Static attribute ids.
    //

    /** The globals id. */
    public static final org.resmedicinae.resmedlib.term.String GLOBALS = new org.resmedicinae.resmedlib.term.String("globals");

    /** The finalize globals id. */
    public static final org.resmedicinae.resmedlib.term.String FINALIZE_GLOBALS = new org.resmedicinae.resmedlib.term.String("finalize_globals");

    /** The configuration id. */
    public static final org.resmedicinae.resmedlib.term.String CONFIGURATION = new org.resmedicinae.resmedlib.term.String("configuration");

    /** The finalize configuration id. */
    public static final org.resmedicinae.resmedlib.term.String FINALIZE_CONFIGURATION = new org.resmedicinae.resmedlib.term.String("finalize_configuration");

    /** The log record id. */
    public static final org.resmedicinae.resmedlib.term.String LOG_RECORD = new org.resmedicinae.resmedlib.term.String("log_record");

    /** The finalize log record id. */
    public static final org.resmedicinae.resmedlib.term.String FINALIZE_LOG_RECORD = new org.resmedicinae.resmedlib.term.String("finalize_log_record");

    /** The named subsystem id. */
    public static final org.resmedicinae.resmedlib.term.String NAMED_SUBSYSTEM = new org.resmedicinae.resmedlib.term.String("named_subsystem");

    /** The logger output id. */
    public static final org.resmedicinae.resmedlib.term.String LOGGER_OUTPUT = new org.resmedicinae.resmedlib.term.String("logger_output");

    /** The log level id. */
    public static final org.resmedicinae.resmedlib.term.String LOG_LEVEL = new org.resmedicinae.resmedlib.term.String("log_level");

    //???? Replace later by an attribute of an ontology level <= Component class!
    /** The encoder id. */
    public static final org.resmedicinae.resmedlib.term.String ENCODER = new org.resmedicinae.resmedlib.term.String("encoder");

    //
    // Static log levels.
    //
    
    /** The log level to turn off logging. */
    public static final org.resmedicinae.resmedlib.number.Integer OFF_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(0);

    /** The log level indicating a serious failure. */
    public static final org.resmedicinae.resmedlib.number.Integer ERROR_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(1);

    /** The log level indicating a potential problem. */
    public static final org.resmedicinae.resmedlib.number.Integer WARNING_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(2);

    /** The log level for informational messages. */
    public static final org.resmedicinae.resmedlib.number.Integer INFO_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(3);

    /** The log level providing tracing information. */
    public static final org.resmedicinae.resmedlib.number.Integer DEBUG_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(4);

    /** The log level indicating that all messages should be logged. */
    public static final org.resmedicinae.resmedlib.number.Integer ALL_LOG_LEVEL = new org.resmedicinae.resmedlib.number.Integer(5);

    //
    // Globals.
    //

    /**
     * Creates a globals structure.
     *
     * @return the globals structure
     * @exception NullPointerException if the globals structure is null
     */
    public Item createGlobals() throws Exception {

        Item g = new Item();

        if (g != null) {
            
            g.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create globals. The globals structure is null.");
        }
        
        return g;
    }

    /**
     * Destroys the globals structure.
     *
     * @param g the globals structure
     * @exception NullPointerException if the globals structure is null
     */
    public void destroyGlobals(Item g) throws Exception, NullPointerException {

        if (g != null) {
            
            g.finalizz();
            
        } else {
            
            throw new NullPointerException("Could not destroy globals. The globals structure is null.");
        }
    }

    //
    // Configuration.
    //
    
    /**
     * Creates a configuration.
     *
     * @return the configuration
     * @exception NullPointerException if the configuration is null
     */
    public Configuration createConfiguration() throws Exception, NullPointerException {

        Configuration c = new Configuration();

        if (c != null) {

//??            Preferences p = Preferences.systemNodeForPackage(getClass());
//??            Preferences p = Preferences.userNodeForPackage(getClass());
//??            location = get("system_configuration_location") ...from command line!

//??            c.setPreferences("preferences", p);
//??            c.setLocation("location", l);

            c.initialize();

        } else {

            throw new NullPointerException("Could not create configuration. The configuration is null.");
        }
        
        return c;
    }

    /**
     * Destroys the configuration.
     *
     * @param c the configuration
     * @exception NullPointerException if the configuration is null
     */
    public void destroyConfiguration(Configuration c) throws Exception, NullPointerException {

        if (c != null) {

            c.finalizz();
            
        } else {

            throw new NullPointerException("Could not destroy configuration. The configuration is null.");
        }
    }

    //
    // Log record.
    //

    /**
     * Creates a log record.
     *
     * @return the log record
     * @exception NullPointerException if the log record is null
     */
    public LogRecord createLogRecord() throws Exception, NullPointerException {

        LogRecord r = new LogRecord();

        if (r != null) {        

            r.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create log record. The log record is null.");
        }

        return r;
    }

    /**
     * Destroys the log record.
     *
     * @param r the log record
     * @exception NullPointerException if the log record is null
     */
    public void destroyLogRecord(LogRecord r) throws Exception, NullPointerException {
        
        if (r != null) {
    
            r.finalizz();
            
        } else {

            throw new NullPointerException("Could not destroy log record. The log record is null.");
        }
    }

    //
    // Finalize globals flag.
    //

    /**
     * Creates the finalize globals flag.
     *
     * @return the finalize globals flag
     * @exception NullPointerException if the finalize globals flag is null
     */
    public org.resmedicinae.resmedlib.digit.Boolean createFinalizeGlobals() throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.digit.Boolean f = new org.resmedicinae.resmedlib.digit.Boolean();

        if (f != null) {
            
            f.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create finalize globals flag. The finalize globals flag is null.");
        }
        
        return f;
    }

    /**
     * Destroys the finalize globals flag.
     *
     * @param f the finalize globals flag
     * @exception NullPointerException if the finalize globals flag is null
     */
    public void destroyFinalizeGlobals(org.resmedicinae.resmedlib.digit.Boolean f) throws Exception, NullPointerException {

        if (f != null) {
            
            f.finalizz();
            
        } else {
            
            throw new NullPointerException("Could not destroy finalize globals flag. The finalize globals flag is null.");
        }
    }

    //
    // Finalize configuration flag.
    //

    /**
     * Creates the finalize configuration flag.
     *
     * @return the finalize configuration flag
     * @exception NullPointerException if the finalize configuration flag is null
     */
    public org.resmedicinae.resmedlib.digit.Boolean createFinalizeConfiguration() throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.digit.Boolean f = new org.resmedicinae.resmedlib.digit.Boolean();

        if (f != null) {
            
            f.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create finalize configuration flag. The finalize configuration flag is null.");
        }
        
        return f;
    }

    /**
     * Destroys the finalize configuration flag.
     *
     * @param f the finalize configuration flag
     * @exception NullPointerException if the finalize configuration flag is null
     */
    public void destroyFinalizeConfiguration(org.resmedicinae.resmedlib.digit.Boolean f) throws Exception, NullPointerException {

        if (f != null) {
            
            f.finalizz();
            
        } else {
            
            throw new NullPointerException("Could not destroy finalize configuration flag. The finalize configuration flag is null.");
        }
    }

    //
    // Finalize log record flag.
    //

    /**
     * Creates the finalize log record flag.
     *
     * @return the finalize log record flag
     * @exception NullPointerException if the finalize log record flag is null
     */
    public org.resmedicinae.resmedlib.digit.Boolean createFinalizeLogRecord() throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.digit.Boolean f = new org.resmedicinae.resmedlib.digit.Boolean();

        if (f != null) {
            
            f.initialize();
            
        } else {
            
            throw new NullPointerException("Could not create finalize log record flag. The finalize log record flag is null.");
        }
        
        return f;
    }

    /**
     * Destroys the finalize log record flag.
     *
     * @param f the finalize log record flag
     * @exception NullPointerException if the finalize log record flag is null
     */
    public void destroyFinalizeLogRecord(org.resmedicinae.resmedlib.digit.Boolean f) throws Exception, NullPointerException {

        if (f != null) {
            
            f.finalizz();
            
        } else {
            
            throw new NullPointerException("Could not destroy finalize log record flag. The finalize log record flag is null.");
        }
    }

    //
    // Named subsystem.
    //

    /**
     * Creates a named subsystem.
     *
     * @return the named subsystem
     * @exception NullPointerException if the named subsystem is null
     */
    public org.resmedicinae.resmedlib.term.String createNamedSubsystem() throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.term.String s = new org.resmedicinae.resmedlib.term.String();

        if (s != null) {
            
            s.initialize();
            s.setJavaObject(getClass().getName());

        } else {
            
            throw new NullPointerException("Could not create named subsystem. The named subsystem is null.");
        }
        
        return s;
    }

    /*
     * Destroys the named subsystem.
     *
     * @param s the named subsystem
     * @exception NullPointerException if the named subsystem is null
     */
    public void destroyNamedSubsystem(org.resmedicinae.resmedlib.term.String s) throws Exception, NullPointerException {

        if (s != null) {
            
            s.finalizz();

        } else {
            
            throw new NullPointerException("Could not destroy named subsystem. The named subsystem is null.");
        }
    }

    //
    // Logger Output.
    //

    /**
     * Creates a logger output.
     *
     * @return the logger output
     * @exception NullPointerException if the logger output is null
     */
    public org.resmedicinae.resmedlib.term.String createLoggerOutput() throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.term.String o = new org.resmedicinae.resmedlib.term.String();

        if (o != null) {
            
            o.initialize();
            o.setJavaObject("console");

        } else {
            
            throw new NullPointerException("Could not create logger output. The logger output is null.");
        }
        
        return o;
    }

    /*
     * Destroys the logger output.
     *
     * @param o the logger output
     * @exception NullPointerException if the logger output is null
     */
    public void destroyLoggerOutput(org.resmedicinae.resmedlib.term.String o) throws Exception, NullPointerException {

        if (o != null) {
            
            o.finalizz();

        } else {
            
            throw new NullPointerException("Could not destroy logger output. The logger output is null.");
        }
    }

    //
    // Log level.
    //

    /**
     * Creates a log level.
     *
     * @return the log level
     * @exception NullPointerException if the log level is null
     */
    public org.resmedicinae.resmedlib.number.Integer createLogLevel() throws Exception, NullPointerException {

        return Component.ALL_LOG_LEVEL;
    }

    /*
     * Destroys the log level.
     *
     * @param l the log level
     * @exception NullPointerException if the log level is null
     */
    public void destroyLogLevel(org.resmedicinae.resmedlib.number.Integer l) throws Exception, NullPointerException {

        if (l != null) {
            
            l.finalizz();

        } else {
            
            throw new NullPointerException("Could not destroy log level. The log level is null.");
        }
    }

    //
    // Globalizable.
    //

    /**
     * Globalizes this component.
     *
     * @param g the globals item
     * @exception NullPointerException if the globals item is null
     */
    public void globalize(Item g) throws Exception, NullPointerException {

        if (g != null) {

            set(Component.GLOBALS, g);
            set(Component.CONFIGURATION, g.get(Component.CONFIGURATION));
            set(Component.LOG_RECORD, g.get(Component.LOG_RECORD));

        } else {

            log(Component.WARNING_LOG_LEVEL, "Could not globalize component. The globals item is null.");
        }
    }

    /**
     * Deglobalizes this component.
     *
     * @param g the globals item
     * @exception NullPointerException if the globals item is null
     */
    public void deglobalize(Item g) throws Exception, NullPointerException {

        if (g != null) {

            remove(Component.LOG_RECORD);
            remove(Component.CONFIGURATION);
            remove(Component.GLOBALS);

        } else {
            
            log(Component.WARNING_LOG_LEVEL, "Could not deglobalize component. The globals item is null.");
        }
    }

    //
    // Initializable.
    //

    /**
     * Initializes this component.
     *
     * @exception NullPointerException if the globals item is null
     * @exception NullPointerException if the configuration is null
     */
    public void initialize() throws Exception, NullPointerException {

        super.initialize();

        set(Component.FINALIZE_GLOBALS, createFinalizeGlobals());
        set(Component.FINALIZE_CONFIGURATION, createFinalizeConfiguration());
        set(Component.FINALIZE_LOG_RECORD, createFinalizeLogRecord());

        // If no globals item was set in the globalize method,
        // then create a globals item here.
        if (get(Component.GLOBALS) == null) {

            set(Component.GLOBALS, createGlobals());
            set(Component.FINALIZE_GLOBALS, new org.resmedicinae.resmedlib.digit.Boolean(true));
        }

        Item g = (Item) get(Component.GLOBALS);

        if (g != null) {

            // If no global configuration was set in the globalize method,
            // then create a configuration here.            
            if (get(Component.CONFIGURATION) == null) {

                set(Component.CONFIGURATION, createConfiguration());
                set(Component.FINALIZE_CONFIGURATION, new org.resmedicinae.resmedlib.digit.Boolean(true));
                g.set(Component.CONFIGURATION, (Configuration) get(Component.CONFIGURATION));
            }

            // If no global log record was set in the globalize method,
            // then create a log record here.            
            if (get(Component.LOG_RECORD) == null) {

                set(Component.LOG_RECORD, createLogRecord());
                set(Component.FINALIZE_LOG_RECORD, new org.resmedicinae.resmedlib.digit.Boolean(true));
                g.set(Component.LOG_RECORD, (LogRecord) get(Component.LOG_RECORD));
            }

        } else {
            
            throw new NullPointerException("Could not initialize component. The globals item is null.");
        }
        
        Configuration c = (Configuration) get(Component.CONFIGURATION);

        if (c != null) {

            set(Component.NAMED_SUBSYSTEM, c.get(Component.NAMED_SUBSYSTEM, createNamedSubsystem()));
            set(Component.LOGGER_OUTPUT, c.get(Component.LOGGER_OUTPUT, createLoggerOutput()));
            set(Component.LOG_LEVEL, c.get(Component.LOG_LEVEL, createLogLevel()));

        } else {
            
            throw new NullPointerException("Could not initialize component. The configuration is null.");
        }
    }

    /**
     * Finalizes this component.
     *
     * @exception NullPointerException if the configuration is null
     * @exception NullPointerException if the globals item is null
     * @exception NullPointerException if the finalize log record item is null
     * @exception NullPointerException if the finalize configuration item is null
     * @exception NullPointerException if the finalize globals item is null
     */
    public void finalizz() throws Exception, NullPointerException {

        try {

            Configuration c = (Configuration) get(Component.CONFIGURATION);

            if (c != null) {

                c.set(Component.LOG_LEVEL, (org.resmedicinae.resmedlib.number.Integer) get(Component.LOG_LEVEL));
                destroyLogLevel((org.resmedicinae.resmedlib.number.Integer) get(Component.LOG_LEVEL));
                remove(Component.LOG_LEVEL);

                c.set(Component.LOGGER_OUTPUT, (org.resmedicinae.resmedlib.term.String) get(Component.LOGGER_OUTPUT));
                destroyLoggerOutput((org.resmedicinae.resmedlib.term.String) get(Component.LOGGER_OUTPUT));
                remove(Component.LOGGER_OUTPUT);

                c.set(Component.NAMED_SUBSYSTEM, (org.resmedicinae.resmedlib.term.String) get(Component.NAMED_SUBSYSTEM));
                destroyNamedSubsystem((org.resmedicinae.resmedlib.term.String) get(Component.NAMED_SUBSYSTEM));
                remove(Component.NAMED_SUBSYSTEM);

                Item g = (Item) get(Component.GLOBALS);
                
                if (g != null) {
        
                    // Only destroy log record, if it was also created
                    // in this component, which can be seen on the flag.            
                    if (get(Component.FINALIZE_LOG_RECORD) != null) {

                        if (((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_LOG_RECORD)).getJavaPrimitive()) {
    
                            g.remove(Component.LOG_RECORD);
                            destroyLogRecord((LogRecord) get(Component.LOG_RECORD));
                            remove(Component.LOG_RECORD);
                        }
    
                    } else {
                        
                        throw new NullPointerException("Could not finalize component. The finalize log record item is null.");
                    }

                    // Only destroy configuration, if it was also created
                    // in this component, which can be seen on the flag.            
                    if (get(Component.FINALIZE_CONFIGURATION) != null) {

                        if (((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_CONFIGURATION)).getJavaPrimitive()) {
    
                            g.remove(Component.CONFIGURATION);
                            destroyConfiguration((Configuration) get(Component.CONFIGURATION));
                            remove(Component.CONFIGURATION);
                        }

                    } else {
                        
                        throw new NullPointerException("Could not finalize component. The finalize configuration item is null.");
                    }

                } else {
                    
                    throw new NullPointerException("Could not finalize component. The globals item is null.");
                }

                // Only destroy globals item, if it was also created
                // in this component, which can be seen on the flag.            
                if (get(Component.FINALIZE_GLOBALS) != null) {
    
                    if (((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_GLOBALS)).getJavaPrimitive()) {
        
                        destroyGlobals((Item) get(Component.GLOBALS));
                        remove(Component.GLOBALS);
                    }

                } else {
                    
                    throw new NullPointerException("Could not finalize component. The finalize globals item is null.");
                }

                destroyFinalizeLogRecord((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_LOG_RECORD));
                remove(Component.FINALIZE_LOG_RECORD);
                destroyFinalizeConfiguration((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_CONFIGURATION));
                remove(Component.FINALIZE_CONFIGURATION);
                destroyFinalizeGlobals((org.resmedicinae.resmedlib.digit.Boolean) get(Component.FINALIZE_GLOBALS));
                remove(Component.FINALIZE_GLOBALS);
            
            } else {
                
                throw new NullPointerException("Could not finalize component. The configuration is null.");
            }

        } finally {

            super.finalizz();
        }
    }

    //
    // Loggable.
    //

    /**
     * Logs a message with no arguments.
     *
     * @param v the log level value
     * @param msg the message
     */
    public void log(org.resmedicinae.resmedlib.number.Integer v, java.lang.String msg) throws Exception {

        log(v, msg, null);
    }

    /**
     * Logs a message with associated throwable information.
     *
     * @param v the log level
     * @param m the message
     * @param t the throwable
     * @exception NullPointerException if the log level is null
     * @exception NullPointerException if the log record is null
     */
    public void log(org.resmedicinae.resmedlib.number.Integer l, java.lang.String m, java.lang.Throwable t) throws Exception, NullPointerException {

        if (l != null) {

            if (l.isSmallerThanOrEqualTo((org.resmedicinae.resmedlib.number.Integer) get(Component.LOG_LEVEL))) {

                LogRecord r = (LogRecord) get(Component.LOG_RECORD);
    
                if (r != null) {

                    r.set(LogRecord.MESSAGE, new org.resmedicinae.resmedlib.term.String(m));
                    r.setThrowable(t);

                } else {
                    
                    throw new NullPointerException("Could not log message. The log record is null.");
                }
    
                log(r);
            }

        } else {
            
            throw new NullPointerException("Could not log message. The log level is null.");
        }
    }

    /**
     * Logs a log record.
     *
     * @param r the log record
     * @exception NullPointerException if the log record is null
     */
    public void log(LogRecord r) throws Exception, NullPointerException {

        org.resmedicinae.resmedlib.region.controller.Encoder e = (org.resmedicinae.resmedlib.region.controller.Encoder) get(Component.ENCODER);

        if (e != null) {

            e.drive(r);
            
        } else {

            /*??
             * Temporary replacement for logging.
             * The motor (output mechanism) has to be assigned here later.
             * For now, the system console is used for message output.
             */
            if (r.getThrowable() != null) {

                java.lang.System.out.println(this + " log\n" + "INFO" + ": " + ((org.resmedicinae.resmedlib.term.String) r.get(LogRecord.MESSAGE)).getJavaObject() + "\n" + r.getThrowable());
                r.getThrowable().printStackTrace();

            } else {

                java.lang.System.out.println(this + " log\n" + "INFO" + ": " + ((org.resmedicinae.resmedlib.term.String) r.get(LogRecord.MESSAGE)).getJavaObject());
            }

//??            throw new NullPointerException("Could not log record. The motor is null.");
        }
    }

    /**
     * Logs a message with associated throwable information.
     *
     * Displays a graphical message dialog, in addition to the pure logging
     * being done in the parent class's log method.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
/*??
    public void log(Level lev, String msg, Throwable t) throws Exception {

        super.log(lev, msg, t);

        DisplayManager dm = getDisplayManager();

        if (dm != null) {

//??            dm.showMessage(lev, msg, t);

            //?? Example for localization!
            //?? showError(e.getLocalizedSourceControlName(), e.getLocalizedMessage());

        } else {

            // Don't throw exception here cause not all controllers/applications
            // use a graphical display, i.e. not all have a display manager.
        }
    }

    /**
     * Shows a message dialog.
     *
     * @param lev the level
     * @param msg the message
     * @param t the throwable
     */
/*??
    public void showMessage(Level lev, String msg, Throwable t) throws Exception {
    }
*/
}

