/*
 * $RCSfile: ExtensibleMarkupLanguageAssembler.java,v $
 *
 * Copyright (c) 1999-2002. Christian Heller. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package reform.resmedicinae.reform.region.assembler;

import org.apache.xerces.dom.*;
import org.apache.xerces.parsers.*;
import org.apache.xml.serialize.*;
import org.resmedicinae.application.healthcare.record.*;
import reform.resmedicinae.reform.gui.*;
import reform.resmedicinae.reform.printer.*;
import reform.resmedicinae.reform.region.model.*;
import reform.resmedicinae.reform.region.*;
import reform.resmedicinae.reform.xml.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.heading.problem.*;
import org.resmedicinae.domain.healthcare.unit.*;
import org.resmedicinae.resmedlib.region.assembler.*;
import org.resmedicinae.resmedlib.term.String;
import org.w3c.dom.*;

import javax.swing.*;
import java.io.*;
import java.util.*;

/**
 * This class represents an extended markup language model assembler.
 * It can be used to assemble/disassemble an extended markup language model from/to a domain model.
 * @version $Revision: 1.2 $ $Date: 2003/03/24 20:12:18 $ $Author: zone3 $
 * @author Christian Heller <christian.heller@tuxtax.de>
 */
public class ExtensibleMarkupLanguageAssembler extends PersistenceAssembler {
    public static final String ASSEMBLER_KEY = "CONCRETE_XML_ASSEMBLER_KEY";
    private Document document = null;
    private HealthRecordViewModel model;

    /** The global path for storing and searching the xml files. */
    private String globalPath = "";

    /** The name of the xml index file. */
    private String indexFile = "";

    /** instance of JFrame */
    private JFrame generalHintDialogFrame = null;

    /** instance of GeneralHintDialog */
    private GeneralHintDialog generalHintDialog = null;

    /**  */

    /** Get / Set / Create / Destroy ***************************************** */

    /**  */
    public HealthRecordViewModel getModel() {
        return model;
    }

    public void setModel(HealthRecordViewModel model) {
        this.model = model;
    }

    /** creates the member variable indexFile. */
    public void createIndexFile() {
        this.indexFile = "C:/Diplom/cvsroot/resmedicinae/share/application/healthcare/reform/xml/index.xml";
    }

    /**
     * Sets the member variable indexFile.
     * @param indexFile the new value for the member variable indexFile
     */
    public void setIndexFile(String indexFile) {
        this.indexFile = indexFile;
    }

    /**
     * Returns the member variable indexFile.
     * @return indexFile
     */
    public String getIndexFile() {
        return this.indexFile;
    }

    /** creates the member variable globalPath. */
    public void createGlobalPath() {
        this.globalPath = "C:/Diplom/cvsroot/resmedicinae/share/application/healthcare/reform/xml/";
    }

    /**
     * Sets the member variable globalPath.
     * @param globalPath the new value for the member variable globalPath
     */
    public void setGlobalPath(String globalPath) {
        this.globalPath = globalPath;
    }

    /**
     * Returns the member variable globalPath.
     * @return globalPath
     */
    public String getGlobalPath() {
        return this.globalPath;
    }

    public void setDocument(Document document) {
        this.document = document;
    }

    /** creates the member variable generalHintDialogFrame. */
    public void createGeneralHintDialogFrame() {
        this.generalHintDialogFrame = new JFrame();
    }

    /**
     * Sets the member variable generalHintDialogFrame.
     * @param generalHintDialogFrame the new value for the member variable generalHintDialogFrame
     */
    public void setGeneralHintDialogFrame(JFrame generalHintDialogFrame) {
        this.generalHintDialogFrame = generalHintDialogFrame;
    }

    /**
     * Returns the member variable generalHintDialogFrame.
     * @return generalHintDialogFrame
     */
    public JFrame getGeneralHintDialogFrame() {
        return this.generalHintDialogFrame;
    }

    /** creates the member variable generalHintDialog. */
    public void createGeneralHintDialog() {
    }

    /**
     * Sets the member variable generalHintDialog.
     * @param generalHintDialog the new value for the member variable generalHintDialog
     */
    public void setGeneralHintDialog(GeneralHintDialog generalHintDialog) {
        this.generalHintDialog = generalHintDialog;
    }

    /**
     * Returns the member variable generalHintDialog.
     * @return generalHintDialog
     */
    public GeneralHintDialog getGeneralHintDialog() {
        return this.generalHintDialog;
    }

    /**  */

    /** Methods For All  ***************************************************** */

    /**  */
    public void initialize() throws Exception {
        try {
            super.initialize();
            this.createIndexFile();
            this.createGlobalPath();
        } catch (Exception e) {
            throw new Exception(e);
        }
    }

    /**  */

    /** Methods For ExtensibleMarkupLanguageModel  *************************************** */

    /**  */
    public Object getData() {
        return (Object) this.document;
    }

    public void setData(Object document) {
        setDocument((Document) document);
    }

    public void saveHealthRecordIndex(HealthRecordIndex healthRecordIndex) {
        saveXMLFile(setAllHealthRecordIndexesToDocument(healthRecordIndex),
                getIndexFile().substring(getIndexFile().lastIndexOf("/") + 1));
    }

    public HealthRecord loadHealthRecord(org.resmedicinae.resmedlib.term.String healthRecordId) throws Exception {
        return getHealthRecordDataFromDocument(parseXMLFile(getGlobalPath() + healthRecordId.getValue() + ".xml"));
    }

    public void saveHealthRecord(HealthRecord healthRecord) {
        setHealthRecordDataToDocument(healthRecord);
    }

    public void saveAllHealthRecords(LoadedHealthRecords loadedHealthRecords) {
        for (Enumeration enum = loadedHealthRecords.keys(); enum.hasMoreElements();) {
            saveHealthRecord((HealthRecord) loadedHealthRecords.get(enum.nextElement()));
        }
    }

    /** removes a HealthRecord from all objects and also the appropriate file from the disk */
    public void removeHealthRecord(org.resmedicinae.resmedlib.term.String healthRecordId) {
        String fileName = getGlobalPath() + healthRecordId.getValue() + ".xml";
        try {
            File f = new File(fileName);
            if (f.exists())
                f.delete();
            else
                System.out.println("HealthRecord file " + fileName + " doesn't exist!");
        } catch (Exception e) {
            System.out.println("HealthRecord couldn't be removed!\n");
            e.printStackTrace();
        }
    }

    /** Imports a single xml file that includes data of a healthRecord. The user has to choose the file from a JFileChooser. */
    public void importHealthRecord() throws Exception {
        JFileChooser chooser = new JFileChooser();
        ExampleFileFilter filterImg = new ExampleFileFilter();
        ExampleFileFilter filterXml = new ExampleFileFilter();
        filterXml.addExtension("xml");
        filterXml.setDescription("XML");
        chooser.setFileFilter(filterXml);
        int returnVal = chooser.showDialog(new JFrame(), "Import HealthRecord");
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            String curFile = chooser.getSelectedFile().getAbsolutePath();
            curFile = curFile.replace('\\', '/');
            HealthRecord importedHealthRecord = getHealthRecordDataFromDocument(parseXMLFile(curFile));
            importedHealthRecord.setIdentifier(ObjectID.getNewOId());
            Hashtable healthRecordData = new Hashtable();
            healthRecordData.put(HealthRecordIndex.ID, importedHealthRecord.getIdentifier());
            healthRecordData.put(HealthRecordIndex.NAME, importedHealthRecord.getName());
            healthRecordData.put(HealthRecordIndex.FIRSTNAME, importedHealthRecord.getFirstName());
            getModel().getHealthRecordIndex().addHealthRecord(healthRecordData);
            setHealthRecordDataToDocument(importedHealthRecord);
            //     setUpdateHealthRecordList();
            System.out.println("You chose to import this file: " + curFile);
        }
    }

    /**
     * Imports all xml files what includes data of healthRecords from a chosen directory.
     * The user has to choose the directory from a JFileChooser.
     */
    public void importHealthRecordDirectory() throws Exception {
        JFileChooser chooser = new JFileChooser();
        ExampleFileFilter filterImg = new ExampleFileFilter();
        ExampleFileFilter filterXml = new ExampleFileFilter();
        filterXml.addExtension("xml");
        filterXml.setDescription("XML");
        chooser.setFileFilter(filterXml);
        chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
        int returnVal = chooser.showDialog(new JFrame(), "Import Directory");
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            String curDirectory = chooser.getSelectedFile().getAbsolutePath();
            curDirectory = curDirectory.replace('\\', '/');
            System.out.println("You chose to import this directory: " + curDirectory);
            HealthRecord importedHealthRecord;
            String curFile;
            File[] fileList = chooser.getSelectedFile().listFiles();
            for (int i = 0; i < fileList.length; i++) {
                curFile = fileList[i].getAbsolutePath();
                if (curFile.toUpperCase().endsWith(".XML")) {
                    importedHealthRecord = getHealthRecordDataFromDocument(parseXMLFile(curFile));
                    try {
                        importedHealthRecord.setIdentifier(ObjectID.getNewOId());
                    } catch (Exception e) {
                        throw new Exception(e);
                    }
                    Hashtable healthRecordData = new Hashtable();
                    healthRecordData.put(HealthRecordIndex.ID, importedHealthRecord.getIdentifier());
                    healthRecordData.put(HealthRecordIndex.NAME, importedHealthRecord.getName());
                    healthRecordData.put(HealthRecordIndex.FIRSTNAME, importedHealthRecord.getFirstName());
                    getModel().getHealthRecordIndex().addHealthRecord(healthRecordData);
                    setHealthRecordDataToDocument(importedHealthRecord);
                    System.out.println("The following file was succesfully imported: " + curFile);
                }
            }
            // setUpdateHealthRecordList();
        }
    }

    /**
     * Loads the healthRecord index file from disk and stores its data in a HealthRecordIndex object.
     * The user has to choose the file from a JFileChooser. The chosen file is stored in the member variable indexFile and
     * the path of this index file is stored in the member variable globalPath.
     */
    public void loadHealthRecordIndexByUser() {
        try {
            JFileChooser chooser = new JFileChooser();
            ExampleFileFilter filterImg = new ExampleFileFilter();
            filterImg.addExtension("jpg");
            filterImg.addExtension("gif");
            filterImg.setDescription("JPG & GIF Images");
            chooser.setFileFilter(filterImg);
            ExampleFileFilter filterTxt = new ExampleFileFilter();
            filterTxt.addExtension("txt");
            filterTxt.setDescription("Text");
            chooser.setFileFilter(filterTxt);
            ExampleFileFilter filterXml = new ExampleFileFilter();
            filterXml.addExtension("xml");
            filterXml.setDescription("XML");
            chooser.setFileFilter(filterXml);
            int returnVal = chooser.showOpenDialog(new JFrame());
            if (returnVal == JFileChooser.APPROVE_OPTION) {
                String curFile = chooser.getSelectedFile().getAbsolutePath();
                System.out.println("B: curfile: " + curFile);
                curFile = curFile.replace('\\', '/');
                setIndexFile(curFile);
                setGlobalPath(curFile.substring(0, curFile.lastIndexOf("/") + 1));
                // setUpdateAfterChangingHealthRecordData();
                System.out.println("You choose to open this file: " + curFile);
            } else
                System.out.println("returnVal: " + returnVal);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void loadHealthRecordIndex() {
        getModel().setHealthRecordIndex(getAllHealthRecordIndexesFromDocument(parseXMLFile(getIndexFile())));
    }

    /**
     * Saves the index file as an specified file selected by the user from a JFileChooser.
     * Saves all loaded HealthRecord data, too, but in the new specified path.
     * The new name of the index file is stored in the member variable indexFile and
     * the path of this index file is stored in the member variable globalPath.
     */
    public void saveDataAs(HealthRecordIndex healthRecordIndex, LoadedHealthRecords loadedHealthRecords) {
        JFileChooser chooser = new JFileChooser();
        ExampleFileFilter filterImg = new ExampleFileFilter();
        filterImg.addExtension("jpg");
        filterImg.addExtension("gif");
        filterImg.setDescription("JPG & GIF Images");
        chooser.setFileFilter(filterImg);
        ExampleFileFilter filterTxt = new ExampleFileFilter();
        filterTxt.addExtension("txt");
        filterTxt.setDescription("Text");
        chooser.setFileFilter(filterTxt);
        ExampleFileFilter filterXml = new ExampleFileFilter();
        filterXml.addExtension("xml");
        filterXml.setDescription("XML");
        chooser.setFileFilter(filterXml);
        int returnVal = chooser.showSaveDialog(new JFrame());
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            String curFile = chooser.getSelectedFile().getAbsolutePath();
            curFile = curFile.replace('\\', '/');
            setIndexFile(curFile);
            setGlobalPath(curFile.substring(0, curFile.lastIndexOf("/") + 1));
            System.out.println("You choose to save this file: " + curFile);
            System.out.println("save index...");
            saveHealthRecordIndex(healthRecordIndex);
            saveAllHealthRecords(loadedHealthRecords);
            System.out.println("healthRecord index saved");
        }
    }

    /**  */

    /** HealthRecordManager  ************************************************** */

    /**  */

    /**
     * Saves a xml file represented by a xml document. The name of the xml file is the name of the document.
     * @param doc the document to save
     * @param fName the name of the file to save the data,
     * the path depends on the member variable globalPath of the actual controller
     * @see ReForm
     */
    public void saveXMLFile(Document doc, String fName) {
        try {
            if (fName.indexOf(".xml") > -1)
                fName = fName.substring(0, fName.indexOf(".xml"));
            String fileName = getGlobalPath() + fName + ".xml";
            System.out.println("Speichern der Datei " + fileName);
            FileOutputStream out = new FileOutputStream(fileName);
            XMLSerializer serializer = new XMLSerializer(out, null);
            try {
                serializer.serialize(doc);
                out.flush();
                out.close();
                System.out.println("...erfolgreich.");
            } catch (Exception e) {
                System.out.println("Document nicht vorhanden:");
                e.printStackTrace();
            }
        } catch (IOException e) {
            System.out.println("Datei kann nicht geschrieben werden:");
            e.printStackTrace();
        }
    }

    /**
     * Parses a xml-file given by the name fileName and stores the data in a Document-object
     * @param fileName the fileName of the file to parse
     * @return the Document that include the parsed data
     */
    public Document parseXMLFile(String fileName) {
        Document doc = new DocumentImpl();
        System.out.println("Versuche, Datei " + fileName + " zu parsen...");
        try {
            DOMParser p = new DOMParser();
            p.setFeature("http://xml.org/sax/features/validation", true);
            p.parse(fileName);
            doc = p.getDocument();
            doc.normalize();
            System.out.println("Parsen der Datei " + fileName + " war erfolgreich.");
        } catch (Exception e) {
            System.out.println(fileName + ": Parsen schlug fehl:" + e);
        }
        return doc;
    }

    /**  */

    /**
     * der neue stuff ***********************************************************************************************************
     */

    /**  */

    /**
     *Gets text data from a particular tag.
     * @param doc the document
     * @param tagName the tag name
     * @return the text from the tag
     */
    public org.resmedicinae.resmedlib.term.String getTagDataFromDocument(Document doc, String tagName) {
        NodeList elementList = doc.getElementsByTagName(tagName);
        if (elementList.getLength() != 0) {
            if (hasNodeValue(elementList.item(0))) {
                return new org.resmedicinae.resmedlib.term.String(getNodeValue(elementList.item(0)));
            } else
                return new org.resmedicinae.resmedlib.term.String(RecordStrings.NO_ENTRY);
        } else
            return new org.resmedicinae.resmedlib.term.String(RecordStrings.NO_ENTRY);
    }

    /**
     *Gets a ehr object from a ehr document.
     * @param doc the ehr document from which the data should be extracted
     */
    public HealthRecord getHealthRecordDataFromDocument(Document doc) throws Exception {
        //       String healthRecordId = new String();
        //        String healthRecordName = new String();
        //        String streetName = new String();
        //        String zipCode = new String();
        //        String street = new String();
        //        String town = new String();
        //        String birthday = new String();
        //        String gender = new String();
        String anamnese = new String("Anamnese");
        String subject = new String();
        //        String insurance = new String();
        String objectText = new String();
        String overlayName = new String();
        Vector xCoordinates = new Vector();
        Vector yCoordinates = new Vector();
        String assessment = new String();
        String plan = new String();
        String problemName = new String();
        org.resmedicinae.resmedlib.term.String problemId = new org.resmedicinae.resmedlib.term.String();
        String partialContactId = new String();
        String partialContactDate = new String();
        String partialContactNumber = new String();
        int partialContactNumberAsInt = 0;
        //        String image = new String();
        //String tempCaveId = new String("0");
        //String tempProblemId = new String("0");
        //String tempEpisodeId = new String("0");
        //String tempPartialContactId = new String("0");
        //        String tempSubjectId = new String("0");
        //        String tempObjectId = new String("0");
        //        String tempAssessmentId = new String("0");
        //String tempPlanId = new String("0");
        //get the name of healthRecord and so on....
        HealthRecord healthRecord = new HealthRecord();
        healthRecord.setIdentifier(getTagDataFromDocument(doc, "HealthRecordId"));
        healthRecord.setName(getTagDataFromDocument(doc, "Name"));
        System.out.println("+++++++++++++++++++++++Getting first name: " + (getTagDataFromDocument(doc, "FirstName")).getValue());
        healthRecord.setFirstName(getTagDataFromDocument(doc, "FirstName"));
        healthRecord.setBirthday(getTagDataFromDocument(doc, "Birthday"));
        healthRecord.setGender(getTagDataFromDocument(doc, "Gender"));
        healthRecord.setStreet(getTagDataFromDocument(doc, "Street"));
        healthRecord.setZipCode(getTagDataFromDocument(doc, "ZipCode"));
        healthRecord.setTown(getTagDataFromDocument(doc, "Town"));
        healthRecord.setCountry(getTagDataFromDocument(doc, "Country"));
        healthRecord.setPhoneNumber(getTagDataFromDocument(doc, "PhoneNumber"));
        healthRecord.setInsurance(getTagDataFromDocument(doc, "Insurance"));
        healthRecord.setInsuranceNumber(getTagDataFromDocument(doc, "InsuranceNumber"));
        healthRecord.setBloodType(getTagDataFromDocument(doc, "BloodType"));
        healthRecord.setHeight(getTagDataFromDocument(doc, "Height"));
        healthRecord.setWeight(getTagDataFromDocument(doc, "Weight"));
        NodeList caveElementList = doc.getElementsByTagName("Cave");
        for (int cavesIndex = 0; cavesIndex < caveElementList.getLength(); cavesIndex++) {
            NodeList cavePropertyNodeList = caveElementList.item(cavesIndex).getChildNodes();
            Cave cave = new Cave();
            cave.initialize();
            for (int cavePropertyIndex = 0; cavePropertyIndex < cavePropertyNodeList.getLength(); cavePropertyIndex++) {
                if (cavePropertyNodeList.item(cavePropertyIndex).getNodeName().equals("CaveID")) {
                    cave.setIdentifier(new org.resmedicinae.resmedlib.term.String(getNodeValue(cavePropertyNodeList.item(cavePropertyIndex))));
                } else if (cavePropertyNodeList.item(cavePropertyIndex).getNodeName().equals("CaveName")) {
                    cave.setName(new org.resmedicinae.resmedlib.term.String(getNodeValue(cavePropertyNodeList.item(cavePropertyIndex))));
                } else if (cavePropertyNodeList.item(cavePropertyIndex).getNodeName().equals("CaveDescription")) {
                    cave.setDescription(new org.resmedicinae.resmedlib.term.String(getNodeValue(cavePropertyNodeList.item(cavePropertyIndex))));
                }
            }
            healthRecord.set(cave.getIdentifier(), cave);
        }
        //extraction the problems, getting tags with name "problem"
        NodeList elementList = doc.getElementsByTagName("Problem");

        /*if there are problems, go on*/

        if (elementList.getLength() != 0) {
            System.out.println("Jens: Health Record has problems!!!");
            for (int i = 0; i < elementList.getLength(); i++) {
                Problem problem = new Problem();
                //initialization from item
                problem.initialize();
                if (elementList.item(i).hasChildNodes()) {
                    //getting all sub elements of one problem, the problem properties
                    NodeList problemPropertyNodeList = elementList.item(i).getChildNodes();
                    for (int f = 0; f < problemPropertyNodeList.getLength(); f++) {
                        //has a property sub elements??? sub elements are ProblemName or ProblemId for example
                        System.out.println("f: " + f + "  problemPropertyNodeList.item(f).getNodeName():" + problemPropertyNodeList.item(f).getNodeName());
                        // if (problemPropertyNodeList.item(f).hasChildNodes()) {
                        // problem name is a property
                        if (problemPropertyNodeList.item(f).getNodeName().equals("ProblemName")) {
                            if (hasNodeValue(problemPropertyNodeList.item(f))) {
                                System.out.println("Catching Problem name");
                                problemName = getNodeValue(problemPropertyNodeList.item(f));
                                System.out.println("Jens: Catching Problem Name!!!");
                            }
                            problem.setName(new org.resmedicinae.resmedlib.term.String(problemName));
                        } else if (problemPropertyNodeList.item(f).getNodeName().equals("ProblemId")) {
                            // problem id is a property
                            if (hasNodeValue(problemPropertyNodeList.item(f))) {
                                System.out.println("Catching Problem id");
                                problemId.setValue(getNodeValue(problemPropertyNodeList.item(f)));
                            }
                            problem.setIdentifier(problemId);
                        } else if (problemPropertyNodeList.item(f).getNodeName().equals("Anamnese")) {
                            System.out.println("<<<<<<<<<<<<<<<<<<<<anamnese: " + anamnese);
                            if (hasNodeValue(problemPropertyNodeList.item(f))) {
                                anamnese = getNodeValue(problemPropertyNodeList.item(f));
                            }
                            System.out.println("<<<<<<<<<<<<<<<<<<<<anamnese: " + anamnese);
                            problem.setAnamnese(new org.resmedicinae.resmedlib.term.String(anamnese));
                        } else if (problemPropertyNodeList.item(f).getNodeName().equals("Episodes")) {
                            System.out.println("Catching Episodes");
                            if (problemPropertyNodeList.item(f).hasChildNodes()) {
                                NodeList episodesNodeList = problemPropertyNodeList.item(f).getChildNodes();
                                for (int ee = 0; ee < episodesNodeList.getLength(); ee++) {
                                    if ((episodesNodeList.item(ee).getNodeName().equals("Episode")) && (episodesNodeList.item(ee).hasChildNodes())) {

                                        System.out.println("a: " + episodesNodeList.item(ee).getNodeName());
                                        System.out.println("b: " + episodesNodeList.item(ee).hasChildNodes());

                                        NodeList episodePropertyNodeList = episodesNodeList.item(ee).getChildNodes();

                                        Episode episode = new Episode();
                                        for (int tt = 0; tt < episodePropertyNodeList.getLength(); tt++) {
                                            System.out.println("G");
                                            System.out.println("Catching Episode");

                                            if (episodePropertyNodeList.item(tt).getNodeName().equals("EpisodeId")) {
                                                if (hasNodeValue(episodePropertyNodeList.item(tt))) {
                                                    String episodeId = getNodeValue(episodePropertyNodeList.item(tt));
                                                    episode.setIdentifier(new org.resmedicinae.resmedlib.term.String(episodeId));
                                                }
                                            } else if (episodePropertyNodeList.item(tt).getNodeName().equals("PartialContacts")) {
                                                NodeList partialContactsNodeList = episodePropertyNodeList.item(tt).getChildNodes();
                                                for (int tu = 0; tu < partialContactsNodeList.getLength(); tu++) {
                                                    if (partialContactsNodeList.item(tu).getNodeName().equals("PartialContact")) {
                                                        System.out.println("Catching Partial Contact");
                                                        NodeList partialContactNodeList =
                                                                partialContactsNodeList.item(tu).getChildNodes();
                                                        PartialContact partialContact = new PartialContact();
                                                        //initialization from item
                                                        partialContact.initialize();
                                                        for (int t = 0; t < partialContactNodeList.getLength(); t++) {
                                                            if (partialContactNodeList.item(t).getNodeName().equals("PartialContactId")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    partialContactId = getNodeValue(partialContactNodeList.item(t));
                                                                    //Construction of an partialContact. add(...) from Item is used as well as
                                                                    //set...-methods from the business objects (which have to be removed...!!!)
                                                                    partialContact.setIdentifier(new org.resmedicinae.resmedlib.term.String(partialContactId));
                                                                    partialContact.setPartialContactNumber(partialContactNumberAsInt);
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("PartialContactNumber")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    partialContactNumber = getNodeValue(partialContactNodeList.item(t));
                                                                    partialContactNumberAsInt =
                                                                            (Integer.valueOf(partialContactNumber)).intValue();
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("PartialContactDate")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    partialContactDate = getNodeValue(partialContactNodeList.item(t));
                                                                    partialContact.setPartialContactDate(new org.resmedicinae.resmedlib.term.String(partialContactDate));
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("Subjective")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    subject = getNodeValue(partialContactNodeList.item(t));

                                                                    /*Adding the subject*/

                                                                    System.out.println("Catching subjective");
                                                                    Subjective soapSubject = new Subjective(subject);
                                                                    partialContact.set(
                                                                            new org.resmedicinae.resmedlib.term.String("subjective"), soapSubject);
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("Objectives")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    NodeList objectNodeList = partialContactNodeList.item(t).getChildNodes();
                                                                    for (int oo = 0; oo < objectNodeList.getLength(); oo++) {
                                                                        if (objectNodeList.item(oo).getNodeName().equals("Objective")) {
                                                                            if (objectNodeList.item(oo).hasChildNodes()) {
                                                                                /*Adding the object*/

                                                                                Objective soapObject = new Objective();
                                                                                NodeList objectPropertiesList = objectNodeList.item(oo).getChildNodes();
                                                                                for (int te = 0; te < objectPropertiesList.getLength(); te++) {
                                                                                    if (objectPropertiesList.item(te).getNodeName().equals("ObjectiveID")) {
                                                                                        if (hasNodeValue(objectPropertiesList.item(te))) {
                                                                                            soapObject.setIdentifier(new org.resmedicinae.resmedlib.term.String(getNodeValue(objectPropertiesList.item(te))));
                                                                                        }
                                                                                    } else if (objectPropertiesList.item(te).getNodeName().equals("Text")) {
                                                                                        if (hasNodeValue(objectPropertiesList.item(te))) {
                                                                                            objectText = getNodeValue(objectPropertiesList.item(te));
                                                                                            soapObject.setText(new org.resmedicinae.resmedlib.term.String(objectText));
                                                                                        }
                                                                                    } else if (objectPropertiesList.item(te).getNodeName().equals("OverlayName")) {
                                                                                        if (hasNodeValue(objectPropertiesList.item(te))) {
                                                                                            overlayName = getNodeValue(objectPropertiesList.item(te));
                                                                                            soapObject.setOverlayName(overlayName);
                                                                                        }
                                                                                    } else if (objectPropertiesList.item(te).getNodeName().equals("Markers")) {
                                                                                        String y = new String();
                                                                                        String x = new String();
                                                                                        //new Vector() not clear()!!!!!!
                                                                                        xCoordinates = new Vector();
                                                                                        yCoordinates = new Vector();
                                                                                        NodeList markerList = objectPropertiesList.item(te).getChildNodes();
                                                                                        for (int r = 0; r < markerList.getLength(); r++) {
                                                                                            if (markerList.item(r).getNodeName().equals("Marker")) {
                                                                                                NodeList coordList = markerList.item(r).getChildNodes();
                                                                                                for (int a = 0; a < coordList.getLength(); a++) {
                                                                                                    if (coordList.item(a).getNodeName().equals("X")) {
                                                                                                        if (hasNodeValue(coordList.item(a))) {
                                                                                                            x = getNodeValue(coordList.item(a));
                                                                                                            xCoordinates.add(new Integer(x));
                                                                                                        }
                                                                                                    } else if (coordList.item(a).getNodeName().equals("Y")) {
                                                                                                        if (hasNodeValue(coordList.item(a))) {
                                                                                                            y = getNodeValue(coordList.item(a));
                                                                                                            yCoordinates.add(new Integer(y));
                                                                                                        }
                                                                                                    }
                                                                                                }
                                                                                                soapObject.setMarkerXCoordinates(xCoordinates);
                                                                                                soapObject.setMarkerYCoordinates(yCoordinates);
                                                                                            }
                                                                                        }
                                                                                    }
                                                                                }

                                                                                /*Add another soapObject to vector*/

                                                                                System.out.println("Catching objective");
                                                                                partialContact.set(soapObject.getIdentifier(), soapObject);
                                                                            }
                                                                        }
                                                                    }
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("Assessment")) {
                                                                if (hasNodeValue(partialContactNodeList.item(t))) {
                                                                    assessment = getNodeValue(partialContactNodeList.item(t));

                                                                    /*Adding the assessment*/

                                                                    System.out.println("Catching assessment");
                                                                    Assessment soapAssessment = new Assessment(assessment);
                                                                    partialContact.set(
                                                                            new org.resmedicinae.resmedlib.term.String("assessment"), soapAssessment);
                                                                }
                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("Plan")) {
                                                                Plan soapPlan = new Plan(plan);
                                                                NodeList planPropertyNodeList =
                                                                        partialContactNodeList.item(t).getChildNodes();
                                                                for (int ppIndex = 0; ppIndex < planPropertyNodeList.getLength();
                                                                     ppIndex++) {
                                                                    if (planPropertyNodeList.item(ppIndex).getNodeName().equals("PlanText")) {
                                                                        plan = getNodeValue(planPropertyNodeList.item(ppIndex));
                                                                        soapPlan.setText(new org.resmedicinae.resmedlib.term.String(plan));
                                                                    } else if (planPropertyNodeList.item(ppIndex).getNodeName().equals("Medications")) {
                                                                        NodeList medicationsList =
                                                                                planPropertyNodeList.item(ppIndex).getChildNodes();
                                                                        System.out.println("++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++Medication gefunden!!!!!!!");
                                                                        System.out.println("Medications List ist " +
                                                                                medicationsList.getLength() + " lang.");
                                                                        for (int medIndex = 0; medIndex < medicationsList.getLength();
                                                                             medIndex++) {
                                                                            if (medicationsList.item(medIndex).getNodeName().equals("Medication")) {
                                                                                Medication medication = new Medication();
                                                                                String medicationName = new String();
                                                                                String medicationDose = new String();
                                                                                NodeList medicationNodeList =
                                                                                        medicationsList.item(medIndex).getChildNodes();
                                                                                for (int medicationIndex = 0; medicationIndex < medicationNodeList.getLength();
                                                                                     medicationIndex++) {
                                                                                    if (medicationNodeList.item(medicationIndex).getNodeName().equals("MedicationName")) {
                                                                                        medicationName =
                                                                                                getNodeValue(medicationNodeList.item(medicationIndex));
                                                                                        System.out.println("..............Medikament-Name: " +
                                                                                                medicationName);
                                                                                    } else if (medicationNodeList.item(medicationIndex).getNodeName().equals("MedicationDose")) {
                                                                                        medicationDose =
                                                                                                getNodeValue(medicationNodeList.item(medicationIndex));
                                                                                        System.out.println("..............Medikament-Dose: " +
                                                                                                medicationDose);
                                                                                    }
                                                                                }
                                                                                medication.setName(new org.resmedicinae.resmedlib.term.String(medicationName));
                                                                                Dose dose = new Dose();
                                                                                dose.setDose(new org.resmedicinae.resmedlib.term.String(medicationDose));
                                                                                medication.set(
                                                                                        new org.resmedicinae.resmedlib.term.String("1"), dose);
                                                                                dose.setIdentifier(new org.resmedicinae.resmedlib.term.String("1"));
                                                                                org.resmedicinae.resmedlib.term.String id =
                                                                                        new org.resmedicinae.resmedlib.term.String((
                                                                                        new Integer(medIndex)).toString());
                                                                                soapPlan.set(id, medication);
                                                                                medication.setIdentifier(id);
                                                                            }
                                                                        }
                                                                    }
                                                                }

                                                                /*Adding the plan*/

                                                                System.out.println("Catching plan");
                                                                partialContact.set(
                                                                        new org.resmedicinae.resmedlib.term.String("plan"), soapPlan);

                                                            } else if (partialContactNodeList.item(t).getNodeName().equals("Markers")) {
                                                            }
                                                            //partialContact.setPartialContactDate(partialContactDate);
                                                        }

                                                        /*Adding the partialContact to the episode*/

                                                        //   partialContact.setIdentifier(ObjectID.getNewOId());
                                                        episode.set(partialContact.getIdentifier(), partialContact);
                                                    }
                                                }

                                            }
                                        }

                                        problem.set(episode.getIdentifier(), episode);
                                        // tempPartialContactId = "0";

                                        //  episode.setIdentifier(new org.resmedicinae.resmedlib.term.String(tempEpisodeId));
                                        //  tempEpisodeId = incrementString(tempEpisodeId);
                                    }

                                }
                            }
                        }

                    }
                    healthRecord.set(problem.getIdentifier(), problem);
                }
            }
        }
        return healthRecord;
    }

    /**
     *Increases the number a string contains
     * @param string the string
     * @return the incremented string
     */
    public String incrementString(String string) {
        return (new Integer(Integer.parseInt(string) + 1)).toString();
    }

    /**
     *Tests whether a node has a value or not
     * @param node the node
     * @return true if the node has a value, false if not
     */
    public boolean hasNodeValue(Node node) {
        if (node.getFirstChild() != null) {
            return true;
        } else {
            return false;
        }
    }

    /**
     *Gets the value from a particular node
     * @param node the node
     * @return the node value if the node has one, otherwise a string indicating the node has no value
     */
    public String getNodeValue(Node node) {
        if (hasNodeValue(node)) {
            return node.getFirstChild().getNodeValue();
        } else {
            return RecordStrings.NO_ENTRY;
        }
    }

    /** Creates a new healthRecord document. */
    public Document setHealthRecordDataToDocument(HealthRecord healthRecord) {
        Document doc = new DocumentImpl();
        Element root = doc.createElement("EHR");
        doc.appendChild(root);
        Element idTag = doc.createElement("HealthRecordId");
        Element nameTag = doc.createElement("Name");
        Element preNameTag = doc.createElement("FirstName");
        Element genderTag = doc.createElement("Gender");
        Element birthdayTag = doc.createElement("Birthday");
        Element streetTag = doc.createElement("Street");
        Element zipCodeTag = doc.createElement("ZipCode");
        Element townTag = doc.createElement("Town");
        Element countryTag = doc.createElement("Country");
        Element phoneNumberTag = doc.createElement("PhoneNumber");
        Element insuranceTag = doc.createElement("Insurance");
        Element insuranceNumberTag = doc.createElement("InsuranceNumber");
        Element bloodTypeTag = doc.createElement("BloodType");
        Element heightTag = doc.createElement("Height");
        Element weightTag = doc.createElement("Weight");
        Element problemsTag = doc.createElement("Problems");
        if (healthRecord.getIdentifier() != null)
            idTag.appendChild(doc.createTextNode(healthRecord.getIdentifier().getValue()));
        if (healthRecord.getName() != null) nameTag.appendChild(doc.createTextNode(healthRecord.getName().getValue()));
        if (healthRecord.getFirstName() != null)
            preNameTag.appendChild(doc.createTextNode(healthRecord.getFirstName().getValue()));
        if (healthRecord.getGender() != null) genderTag.appendChild(doc.createTextNode(healthRecord.getGender().getValue()));
        if (healthRecord.getBirthday() != null)
            birthdayTag.appendChild(doc.createTextNode(healthRecord.getBirthday().getValue()));
        if (healthRecord.getStreet() != null) streetTag.appendChild(doc.createTextNode(healthRecord.getStreet().getValue()));
        if (healthRecord.getZipCode() != null)
            zipCodeTag.appendChild(doc.createTextNode(healthRecord.getZipCode().getValue()));
        if (healthRecord.getTown() != null) townTag.appendChild(doc.createTextNode(healthRecord.getTown().getValue()));
        if (healthRecord.getCountry() != null) countryTag.appendChild(doc.createTextNode(healthRecord.getCountry().getValue()));
        if (healthRecord.getPhoneNumber() != null) phoneNumberTag.appendChild(doc.createTextNode(healthRecord.getPhoneNumber().getValue()));
        if (healthRecord.getInsurance() != null)
            insuranceTag.appendChild(doc.createTextNode(healthRecord.getInsurance().getValue()));
        if (healthRecord.getInsuranceNumber() != null)
            insuranceNumberTag.appendChild(doc.createTextNode(healthRecord.getInsuranceNumber().getValue()));
        if (healthRecord.getBloodType() != null)
            bloodTypeTag.appendChild(doc.createTextNode(healthRecord.getBloodType().getValue()));
        if (healthRecord.getHeight() != null)
            heightTag.appendChild(doc.createTextNode(healthRecord.getHeight().getValue()));
        if (healthRecord.getWeight() != null)
            weightTag.appendChild(doc.createTextNode(healthRecord.getWeight().getValue()));
        if (healthRecord.hasCaves()) {
            Object[] caves = healthRecord.getCaves();
            Element cavesTag = doc.createElement("Caves");
            for (int caveIndex = 0; caveIndex < caves.length; caveIndex++) {
                Cave cave = (Cave) caves[caveIndex];
                Element caveTag = doc.createElement("Cave");
                Element caveIDTag = doc.createElement("CaveID");
                Element caveNameTag = doc.createElement("CaveName");
                Element caveDescriptionTag = doc.createElement("CaveDescription");
//<<<<<<< ExtensibleMarkupLanguageAssembler.java
                Element caveIdTag = doc.createElement("CaveId");
                caveNameTag.appendChild(doc.createTextNode(cave.getName().getValue()));
                caveDescriptionTag.appendChild(doc.createTextNode(cave.getDescription().getValue()));
                caveIdTag.appendChild(doc.createTextNode(cave.getIdentifier().getValue()));
//=======
                caveIDTag.appendChild(doc.createTextNode(cave.getIdentifier().getValue()));
                caveNameTag.appendChild(doc.createTextNode(cave.getName().getValue()));
                caveDescriptionTag.appendChild(doc.createTextNode(cave.getDescription().getValue()));
                caveTag.appendChild(caveIDTag);
//>>>>>>> 1.2
                caveTag.appendChild(caveNameTag);
                caveTag.appendChild(caveDescriptionTag);
                caveTag.appendChild(caveIdTag);
                cavesTag.appendChild(caveTag);

            }
            root.appendChild(cavesTag);
        }
        if (healthRecord.hasProblems()) {
            Object[] problems = healthRecord.getProblems();
            System.out.println("================================");
            System.out.println("Problem-Anzahl: " + problems.length);
            System.out.println("================================");
            for (int i = 0; i < problems.length; i++) {
                Problem problem = (Problem) problems[i];
                String problemName = problem.getName().getValue();
                System.out.println("Problem Nr. : " + problem.getIdentifier().getValue());
                String problemDescription = problem.getAnamnese().getValue();
                Element problemTag = doc.createElement("Problem");
                Element problemIdTag = doc.createElement("ProblemId");
                problemIdTag.appendChild(doc.createTextNode(problem.getIdentifier().getValue()));
                Element problemNameTag = doc.createElement("ProblemName");
                problemNameTag.appendChild(doc.createTextNode(problemName));
                Element anamneseTag = doc.createElement("Anamnese");
                anamneseTag.appendChild(doc.createTextNode(problemDescription));
                problemTag.appendChild(problemIdTag);
                problemTag.appendChild(problemNameTag);
                problemTag.appendChild(anamneseTag);
                if (problem.hasEpisodes()) {
                    Element episodesTag = doc.createElement("Episodes");
                    Object[] episodes = problem.getEpisodes();
                    System.out.println("Episoden-Anzahl: " + episodes.length);
                    for (int ii = 0; ii < episodes.length; ii++) {
                        Episode episode = (Episode) episodes[ii];
                        System.out.println("Episoden-Nr. :" + episode.getIdentifier().getValue());
                        Element episodeTag = doc.createElement("Episode");
                        Element episodeIdTag = doc.createElement("EpisodeId");
                        episodeIdTag.appendChild(doc.createTextNode(episode.getIdentifier().getValue()));
                        episodeTag.appendChild(episodeIdTag);
                        if (episode.hasPartialContacts()) {
                            Element partialContactsTag = doc.createElement("PartialContacts");
                            Object[] partialContacts = episode.getPartialContacts();
                            System.out.println("Partial Contact-Anzahl: " + partialContacts.length);
                            for (int iii = 0; iii < partialContacts.length; iii++) {
                                PartialContact partialContact = (PartialContact) partialContacts[iii];
                                System.out.println("Partial Contact Nr. :" + partialContact.getIdentifier().getValue());
                                Element partialContactTag = doc.createElement("PartialContact");
                                Element partialContactIdTag = doc.createElement("PartialContactId");
                                partialContactIdTag.appendChild(doc.createTextNode(partialContact.getIdentifier().getValue()));
                                Element partialContactNumberTag = doc.createElement("PartialContactNumber");
                                partialContactNumberTag.appendChild(doc.createTextNode((
                                        new Integer(partialContact.getPartialContactNumber())).toString()));
                                Element partialContactDateTag = doc.createElement("PartialContactDate");
                                partialContactDateTag.appendChild(doc.createTextNode(partialContact.getPartialContactDate().getValue()));
                                System.out.println("++++++++++++++++++++Partial Contact date: " +
                                        partialContact.getPartialContactDate());
                                Element subjectTag = doc.createElement("Subjective");
                                if (partialContact.hasSubjectives()) {
                                    Subjective subjective = (Subjective) partialContact.getSubjectives()[0];
                                    subjectTag.appendChild(doc.createTextNode(subjective.getText().getValue()));
                                }
                                Element objectsTag = doc.createElement("Objectives");
                                if (partialContact.hasObjectives()) {
                                    for (int iiii = 0; iiii < partialContact.getObjectiveCount(); iiii++) {
                                        Element objectTag = doc.createElement("Objective");
                                        Element objectIDTag = doc.createElement("ObjectiveID");
                                        Element textTag = doc.createElement("Text");
                                        Objective soapObject = ((Objective) partialContact.getObjectives()[iii]);
                                        String soapObjectText = soapObject.getText().getValue();
                                        objectIDTag.appendChild(doc.createTextNode(soapObject.getIdentifier().getValue()));
                                        textTag.appendChild(doc.createTextNode(soapObjectText));
                                        Element overlayNameTag = doc.createElement("OverlayName");
                                        String soapObjectOverlayName = soapObject.getOverlayName();
                                        overlayNameTag.appendChild(doc.createTextNode(soapObjectOverlayName));
                                        Element markersTag = doc.createElement("Markers");
                                        for (int index = 0; index < soapObject.getMarkerCount(); index++) {
                                            Integer x = soapObject.getXMarker(index);
                                            Integer y = soapObject.getYMarker(index);
                                            System.out.println("X : " + x.intValue());
                                            System.out.println("Y : " + y.intValue());
                                            System.out.println("------------------");
                                            Element markerTag = doc.createElement("Marker");
                                            Element xMarkerTag = doc.createElement("X");
                                            xMarkerTag.appendChild(doc.createTextNode(x.toString()));
                                            Element yMarkerTag = doc.createElement("Y");
                                            yMarkerTag.appendChild(doc.createTextNode(y.toString()));
                                            markerTag.appendChild(xMarkerTag);
                                            markerTag.appendChild(yMarkerTag);
                                            markersTag.appendChild(markerTag);
                                        }
                                        objectTag.appendChild(objectIDTag);
                                        objectTag.appendChild(textTag);
                                        objectTag.appendChild(overlayNameTag);
                                        objectTag.appendChild(markersTag);
                                        objectsTag.appendChild(objectTag);
                                    }
                                }
                                Element assessmentTag = doc.createElement("Assessment");
                                if (partialContact.hasAssessments()) {
                                    Assessment assessment = (Assessment) partialContact.getAssessments()[0];
                                    assessmentTag.appendChild(doc.createTextNode(assessment.getText().getValue()));
                                }
                                Element planTag = doc.createElement("Plan");
                                if (partialContact.hasPlans()) {
                                    Plan plan = (Plan) partialContact.getPlans()[0];
                                    Element planTextTag = doc.createElement("PlanText");
                                    planTextTag.appendChild(doc.createTextNode(plan.getText().getValue()));
                                    planTag.appendChild(planTextTag);
                                    if (plan.hasMedication()) {
                                        Element medicationsTag = doc.createElement("Medications");
                                        Object[] medications = plan.getMedication();
                                        for (int medicationIndex = 0; medicationIndex < medications.length;
                                             medicationIndex++) {
                                            Medication medication = (Medication) medications[medicationIndex];
                                            Element medicationTag = doc.createElement("Medication");
                                            Element medicationNameTag = doc.createElement("MedicationName");
                                            Element medicationDoseTag = doc.createElement("MedicationDose");
                                            medicationNameTag.appendChild(doc.createTextNode(medication.getName().getValue()));
                                            medicationDoseTag.appendChild(doc.createTextNode(((Dose)
                                                    medication.getDose()[0]).getDose().getValue()));
                                            medicationTag.appendChild(medicationNameTag);
                                            medicationTag.appendChild(medicationDoseTag);
                                            medicationsTag.appendChild(medicationTag);
                                        }
                                        planTag.appendChild(medicationsTag);
                                    }
                                }
                                partialContactTag.appendChild(partialContactDateTag);
                                partialContactTag.appendChild(partialContactIdTag);
                                partialContactTag.appendChild(partialContactNumberTag);
                                //partialContactTag.appendChild(partialContactDateTag);
                                partialContactTag.appendChild(subjectTag);
                                partialContactTag.appendChild(objectsTag);
                                partialContactTag.appendChild(assessmentTag);
                                partialContactTag.appendChild(planTag);
                                partialContactsTag.appendChild(partialContactTag);
                                episodeTag.appendChild(partialContactsTag);
                            }
                            episodesTag.appendChild(episodeTag);
                        }
                        problemTag.appendChild(episodesTag);
                    }
                }
                problemsTag.appendChild(problemTag);
            }
        }
        root.appendChild(idTag);
        root.appendChild(nameTag);
        root.appendChild(preNameTag);
        root.appendChild(genderTag);
        root.appendChild(birthdayTag);
        root.appendChild(streetTag);
        root.appendChild(zipCodeTag);
        root.appendChild(townTag);
        root.appendChild(countryTag);
        root.appendChild(phoneNumberTag);
        root.appendChild(insuranceTag);
        root.appendChild(insuranceNumberTag);
        root.appendChild(bloodTypeTag);
        root.appendChild(heightTag);
        root.appendChild(weightTag);
        root.appendChild(problemsTag);
        saveXMLFile(doc, healthRecord.getIdentifier().getValue());
        return doc;
    }

    /**
     * Creates a DOM-Tree of the specified data and then calls saveXMLFile()
     * @param comp the component to print
     * @param jobName the name of the print job
     */
    public Document setAllHealthRecordIndexesToDocument(HealthRecordIndex healthRecordIndex) {
        Document doc = new DocumentImpl();
        Element elHealthRecords = doc.createElement("HealthRecords");
        Element elHealthRecord;
        Element elHealthRecordId;
        Element elName;
        Element elFirstName;
        Hashtable healthRecordData;
        for (Enumeration enum = healthRecordIndex.getAllHealthRecords().elements(); enum.hasMoreElements();) {
            elHealthRecord = doc.createElement("HealthRecord");
            elHealthRecordId = doc.createElement("HealthRecordId");
            elName = doc.createElement("Name");
            elFirstName = doc.createElement("FirstName");
            healthRecordData = (Hashtable) enum.nextElement();
            elHealthRecordId.appendChild(doc.createTextNode(((org.resmedicinae.resmedlib.term.String)
                    healthRecordData.get(HealthRecordIndex.ID)).getValue()));
            elName.appendChild(doc.createTextNode(((org.resmedicinae.resmedlib.term.String)
                    healthRecordData.get(HealthRecordIndex.NAME)).getValue()));
            elFirstName.appendChild(doc.createTextNode(((org.resmedicinae.resmedlib.term.String)
                    healthRecordData.get(HealthRecordIndex.FIRSTNAME)).getValue()));
            elHealthRecord.appendChild(elHealthRecordId);
            elHealthRecord.appendChild(elName);
            elHealthRecord.appendChild(elFirstName);
            elHealthRecords.appendChild(elHealthRecord);
        }
        doc.appendChild(elHealthRecords);
        return doc;
    }

    /**
     * Gets the data from the given document and stores them in the HealthRecordIndex object
     * @param doc the Document object that includes the data
     * @return the healthRecord index that includes the data from the document
     */
    public HealthRecordIndex getAllHealthRecordIndexesFromDocument(Document doc) {
        //        Integer maximumId = ((ConcreteDomainModel) getController().getModel()).getHealthRecordIndex().getMaximumId();
        //        Integer id;
        Hashtable healthRecordData;
        HealthRecordIndex healthRecordIndex = new HealthRecordIndex();
        org.resmedicinae.resmedlib.term.String healthRecordId = new org.resmedicinae.resmedlib.term.String();
        org.resmedicinae.resmedlib.term.String healthRecordName = new org.resmedicinae.resmedlib.term.String();
        org.resmedicinae.resmedlib.term.String healthRecordFirstName = new org.resmedicinae.resmedlib.term.String();
        HealthRecord healthRecord = new HealthRecord();
        NodeList healthRecordList = doc.getElementsByTagName("HealthRecord");
        if (healthRecordList.getLength() != 0) {
            //	NodeList elementList = healthRecordList.item(0).getChildNodes();
            for (int i = 0; i < healthRecordList.getLength(); i++) {
                //Are there healthRecords?
                if (healthRecordList.item(i).hasChildNodes()) {
                    NodeList dataNodeList = healthRecordList.item(i).getChildNodes();
                    for (int o = 0; o < dataNodeList.getLength(); o++) {
                        if (dataNodeList.item(o).hasChildNodes()) {
                            if (dataNodeList.item(o).getNodeName().equals("HealthRecordId"))
                                healthRecordId = new org.resmedicinae.resmedlib.term.String(dataNodeList.item(o).getFirstChild().getNodeValue());
                            else if (dataNodeList.item(o).getNodeName().equals("Name"))
                                healthRecordName = new org.resmedicinae.resmedlib.term.String(dataNodeList.item(o).getFirstChild().getNodeValue());
                            else if (dataNodeList.item(o).getNodeName().equals("FirstName"))
                                healthRecordFirstName = new org.resmedicinae.resmedlib.term.String(dataNodeList.item(o).getFirstChild().getNodeValue());
                        }
                    }
                }
                healthRecordData = new Hashtable();
                healthRecordData.put(HealthRecordIndex.ID, healthRecordId);
                healthRecordData.put(HealthRecordIndex.NAME, healthRecordName);
                healthRecordData.put(HealthRecordIndex.FIRSTNAME, healthRecordFirstName);
                healthRecordIndex.addHealthRecord(healthRecordData);
                //                id = new Integer(healthRecordId);
                //                if (id.intValue() > maximumId.intValue())
                //                    maximumId = id;
            }
        }
        //        healthRecordIndex.setMaximumId(maximumId);
        return healthRecordIndex;
    }
}
