/*
 * $RCSfile: ObjectModel.java,v $
 *
 * Copyright (c) 1999-2003. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.treetable;

import resmedicinae.record.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.heading.problem.*;
import org.resmedicinae.domain.healthcare.unit.*;

import javax.swing.event.*;
import java.util.*;

/**
 * This class represents the data structure of the JTreeTable. As every model
 * of a JTree this class represents a tree-like structure. Here nodes are business objects.
 * The HealthRecord ist the root object. HealthRecord's children are problems. Problem's children are episodes and
 * so on... . In that way the tree structure is build. So the tree monitores all relevant patient
 * data in tree like structure.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class ObjectModel extends AbstractTreeTableModel {
    /** Column names. */
    public static final String COLUMN_NAME_ARRAY = "column_name_array";
    /** . Column types. */
    public static final String COLUMN_TYPE_ARRAY = "column_type_array";
    /** A constant named zero */
    public static final Integer ZERO = new Integer(0);
    /** The root of the model. This could be a patient. */
    public static final String ROOT = "root";

    /**
     * A reference to the record model. This is needed because the record model has to be updated after
     * changes are made in the tree model.
     */
    public static final String RECORD_MODEL = "record_model";

    /** The default constructor. */
    public ObjectModel() {
        super();
    }

    /**
     *The constructor.
     * @param recordModel the record model
     */
    public ObjectModel(RecordModel recordModel) {
        super();
        setRecordModel(recordModel);
    }

    /**The constructor with root and record model
     * @param root the root of the treeTable
     * @param recordModel the model of the MVC
     */

    public ObjectModel(Object root, RecordModel recordModel) {
        super(root);
        setRoot(root);
        setRecordModel(recordModel);
        //???don't know if this is needed:
        fireTreeStructureChanged(root);
    }

    /**
     * Returns the number of children of the node . TreeModel interface.
     * @param node the node
     * @return the number of child nodes
     */
    public int getChildCount(Object node) {
        return getChildren(node).length;
    }

    /**
     *Gets all children of the node
     * @param node the node
     * @return Vector of children
     */
    public Object[] getChildren(Object node) {
        Object[] returnArray = new Object[0];
        Vector returnVector = new Vector();
        //if the node is a problem then the children are episodes
        if (node.getClass() == Problem.class) {
            returnArray = ((Problem) node).getEpisodes();
            //if the node is a episode then the children are SOAP elements
        } else if (node.getClass() == Episode.class) {
            returnArray = ((Episode) node).getPartialContacts();
        } else if (node.getClass() == PartialContact.class) {
            Object[] tempArray = ((PartialContact) node).getChildren().values().toArray();
            Vector returnVector2 = new Vector();
            for (int i = 1; i < tempArray.length + 1; i++) {
                returnVector2.add(tempArray[tempArray.length - i]);
            }
            returnArray = returnVector2.toArray();
        } else if (node.getClass() == Plan.class) {
            returnArray = ((Plan) node).getMedication();
        } else if (node.getClass() == Medication.class) {
            returnArray = ((Medication) node).getChildren().values().toArray();
        } else if (node == String.class) {
            //if the node is a patient then the children are problems
        } else if (node.getClass() == HealthRecord.class) {
            returnArray = ((HealthRecord) node).getProblems();
        } else if (node.getClass() == TreeTableRoot.class) {
            Vector container = (Vector) node;
            for (int i = 0; i < container.size(); i++) {
                returnVector.add(container.elementAt(i));
            }
            return returnVector.toArray();
        }
        //other patient values can be parsed too....for example name, gender.....
        return returnArray;
    }

    /**
     * Returns if the node is a leaf. Overridden from AbstractTreeTableModel
     * @param node the node
     * @return true if node is leaf, false if not
     */
    public boolean isLeaf(Object node) {
        if (getChildren(node).length == 0) {
            return true;
        } else
            return false;
    }

    /** Returns the number of columns. TreeTableModel interface.
     * @return the number of columns */
    public int getColumnCount() {
        return cNames.length;
    }

    /**
     * Returns the column name. TreeTableModel interface.
     * @param column the column
     * @return the column name
     */
    public String getColumnName(int column) {
        return cNames[column];
    }

    /**
     * Returns the column class. Overriden from AbstractTreeTableModel.
     * @param column the column number
     * @return the column class
     */
    public Class getColumnClass(int column) {
        return cTypes[column];
    }

    /**
     * Sets value aValue for a particaluar node in a column. TreeTableModel interface.
     * If the cell entries changes because of user input this method will be called. The corresponding
     * object will be edited (patient, problem episode, SOAP elements).
     * @param aValue value for the node
     * @param node the node
     * @param column the column of the node
     */
    public void setValueAt(Object aValue, Object node, int column) {
        if (node.getClass() == Problem.class) {
            System.out.println("Column: " + column);
            if (column == 1) {
                ((Problem) node).setName(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
            } else if (column == 2) {
                System.out.println("Setting Anamnese Value");
                ((Problem) node).setAnamnese(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
            }
        } else if (node.getClass() == Episode.class) {
            node = "Episode";
            //if the node is a Subjective the text of the Subjective will be set to the value
        } else if (node.getClass() == PartialContact.class) {
            node = "Partial Contact";
            //if the node is a Subjective the text of the Subjective will be set to the value
        } else if (node.getClass() == Subjective.class) {
            ((Subjective) node).setText(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
            //if the node is a Objective the text of the Objective will be set to the value
        } else if (node.getClass() == Objective.class) {
            ((Objective) node).setText(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
            //if the node is a Assessment the text of the Assessment will be set to the value
        } else if (node.getClass() == Assessment.class) {
            ((Assessment) node).setText(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
            //if the node is a Plan the text of the Plan will be set to the value
        } else if (node.getClass() == Plan.class) {
            ((Plan) node).setText(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
        } else if (node.getClass() == Medication.class) {
            ((Medication) node).setName(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
        } else if (node.getClass() == Dose.class) {
            ((Dose) node).setDose(new org.resmedicinae.resmedlib.term.String(aValue.toString()));
        }
        //after every user input the patient will be saved
    }

    /**
     * Gets the value of the node at the particular column. This method is called after loading the model.
     * The node can be a patient, problem, episode or SOAP element. These objects represent the object model
     * of the tree and have to be read by this method.
     * @param node the node
     * @param column the column
     * @return the value of the node in the column
     */
    public Object getValueAt(Object node, int column) {
        String entry = new String();
        //the remark will be dynamically changed according to the node
        String remark1 = new String("");
        String remark2 = new String("");
        //if the node is a patient, read name, firstname and birthday
        if (node.getClass() == HealthRecord.class) {
            /*String firstName = ((HealthRecord) node).getFirstName().getValue();
            String name = ((HealthRecord) node).getName().getValue();
            String birthday = ((HealthRecord) node).getBirthday().getValue();
            remark1 = firstName + ", " + name + ", " + birthday;
            //if the node is a problem, read the problem name   */
        } else if (node.getClass() == Problem.class) {
            Problem problem = (Problem) node;
            remark1 = problem.getName().getValue();
            remark2 = problem.getAnamnese().getValue();
        } else if (node.getClass() == Episode.class) {
            Episode episode = (Episode) node;
            if (episode.hasPartialContacts()) {
                remark2 = episode.getFirstPartialContact().getPartialContactDate().getValue();
                remark1 = episode.getLastPartialContact().getPartialContactDate().getValue();
            }
            //((Episode)node).getPartialContactDate();
            //if the node is a Subjective, read the subject text
        } else if (node.getClass() == PartialContact.class) {
            PartialContact contact = (PartialContact) node;
            remark1 = contact.getPartialContactDate().getValue();
            //if the node is a Objective, read the object text
        } else if (node.getClass() == Subjective.class) {
            Subjective s = (Subjective) node;
            remark1 = (String) ((Subjective) node).getText().getValue();
            //if the node is a Objective, read the object text
        } else if (node.getClass() == Objective.class) {
            Objective objective = (Objective) node;
            remark1 = (String) ((Objective) node).getText().getValue();
            //if the node is a Assessment, read the assessment text
        } else if (node.getClass() == Assessment.class) {
            Assessment assessment = (Assessment) node;
            remark1 = (String) ((Assessment) node).getText().getValue();
            //if the node is a Subjective, read the plan text
        } else if (node.getClass() == Plan.class) {
            Plan plan = (Plan) node;
            remark1 = (String) ((Plan) node).getText().getValue();
        } else if (node.getClass() == Medication.class) {
            Medication medication = (Medication) node;
            remark1 = medication.getName().getValue();
        } else if (node.getClass() == Dose.class) {
            Dose dose = (Dose) node;
            remark1 = dose.getDose().getValue();
        }
        try {
            if (column == 0) {
                return entry;
            } else if (column == 1) {
                return remark1;
            } else if (column == 2) {
                return remark2;
            }
        } catch (SecurityException se) {
        }
        return null;
    }

    /**
     * Returns whether the cell is editable. Overridden from AbstractTreeTableModel
     * @param node the node
     * @param column the column
     */
    public boolean isCellEditable(Object node, int column) {
        return true;
    }

    /**
     * Returns a particular child i for a node. Here the getChildren(node) method is called. Interface TreeModel.
     * @param node the node
     * @param i number i of children of node
     * @return the child of the node at index i
     */
    public Object getChild(Object node, int i) {
        return getChildren(node)[i];
    }

    /**
     *Initializes the root of the tree. Deprecated???
     * @param root the root object
     */
    public void initializeRoot(Object root) {
        Object path[] = getChildren(root);
        int childCount = ((HealthRecord) root).getProblems().length;
        int childIndices[] = {0, 1, 2};
        Object[] children = getChildren(root);
        fireTreeStructureChanged(root);
    }

    /**
     *Fires when the tree structure changes. Is deprecated since the model is new instatiated in the case of drastic changes.
     * @param patient the new root
     */
    protected void fireTreeStructureChanged(Object patient) {
        Object[] listeners = getListenerList().getListenerList();
        Object[] root = {patient};
        TreeModelEvent e = new TreeModelEvent(this, root);
        for (int i = 0; i < listeners.length; i++) {
            ((TreeModelListener) listeners[i]).treeStructureChanged(e);
        }
    }
}
