/*
 * $RCSfile: AbstractTreeTableModel.java,v $
 *
 * Copyright (c) 1999-2003. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.treetable;

import org.resmedicinae.resmedlib.region.model.*;

import javax.swing.event.*;
import javax.swing.tree.*;

//import org.resmedicinae.resmedlib.model.*;

/**
 * An abstract implementation of the TreeTableModel interface, handling the list of listeners.
 * @version %I% %G%
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public abstract class AbstractTreeTableModel extends ViewModel implements TreeTableModel {
    
    public static final ROOT = "root";
    
    public static final String EVENT_LISTENER_LIST = "event_listener_list";

    /** The constructor.
     * @param root the root of the treeTable
     *  */
    public AbstractTreeTableModel(Object root) {
        this.root = root;
        this.listenerList = new EventListenerList();
    }

    /** The default constructor. */
    public AbstractTreeTableModel() {
    }

    /** Tests whether the node is a leaf node. Interface TreeModel.
     * @param node the node
     * @return true if the node is leaf, false if not
     * */
    public boolean isLeaf(Object node) {
        //when the node has no children it is a leaf
        return getChildCount(node) == 0;
    }

    /** Sets a new value for the path. Not implemented yet. Interface TreeModel.
     * @param path the treePath
     * @param newValue the new value
     * */
    public void valueForPathChanged(TreePath path, Object newValue) {
    }

    /**Gets the index of a particular child
     * @param parent the parentNode
     * @param child the childNode
     * @return the index of the childNode
     */
    public int getIndexOfChild(Object parent, Object child) {
        for (int i = 0; i < getChildCount(parent); i++) {
            if (getChild(parent, i).equals(child)) {
                return i;
            }
        }
        return -1;
    }

    /** Adds a tree model listener to the listenerList. Interface TreeModel.
     *@param l the treeModelListener
     * */
    public void addTreeModelListener(TreeModelListener l) {
        listenerList.add(TreeModelListener.class, l);
    }

    /** Removes a tree model listener from the listenerList. Interface TreeModel.
     *@param l the treeModelListener
     * */
    public void removeTreeModelListener(TreeModelListener l) {
        listenerList.remove(TreeModelListener.class, l);
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesChanged(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesChanged(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesInserted(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesInserted(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesRemoved(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesRemoved(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeStructureChanged(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeStructureChanged(e);
            }
        }
    }

    /** Gets the class of column. Interface TreeModel.
     * @param column the column
     * @return the class of column
     * */
    public Class getColumnClass(int column) {
        return Object.class;
    }

    /** Tests whether a cell is editable. Interface TreeModel.
     * @param node the node to be tested
     * @param column the column of the node
     * @return true if the columnClass == TreeTableModelClass, false if not
     * */
    public boolean isCellEditable(Object node, int column) {
        //only the tree table column is editable
        return getColumnClass(column) == TreeTableModel.class;
    }
}
