/*

 * $RCSfile: PatientInfoPanel.java,v $
 *
 * Copyright (c) 1999-2003. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.topology;

import resmedicinae.record.*;
import org.resmedicinae.domain.healthcare.*;
import org.resmedicinae.domain.healthcare.description.*;
import org.resmedicinae.domain.healthcare.unit.*;

import javax.swing.*;
import java.awt.*;

/** View component. This panel shows basic patient properties.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 * */
public class PatientInfoPanel extends RecordPanel {
    /** TopologyLabel showing the name of patient */
    public static final String PATIENT_NAME_LABEL = "patient_name_label";
    /** TopologyLabel showing the first name of patient */
    public static final String PATIENT_FIRST_NAME_LABEL = "patient_first_name_label";
    /** TopologyLabel showing the gender of patient */
    public static final String PATIENT_GENDER_LABEL = "patient_gender_label";
    /** TopologyLabel showing the birthday of patient */
    public static final String PATIENT_BIRTHDAY_LABEL = "patient_birthday_label";
    /** TopologyLabel showing the town of patient */
    public static final String PATIENT_TOWN_LABEL = "patient_town_label";
    /** TopologyLabel showing the adress of patient */
    public static final String PATIENT_ADRESS_LABEL = "patient_adress_label";
    /** TopologyLabel showing the adress of patient */
    public static final String PATIENT_TELEFON_NUMBER_LABEL = "patient_telefon_number_label";
    /** TopologyLabel showing the insurance of patient */
    public static final String PATIENT_INSURANCE_LABEL = "patient_insurance_label";
    /** TopologyLabel showing the insurance number of patient */
    public static final String PATIENT_INSURANCE_NUMBER_LABEL = "patient_insurance_number_label";
    /** TopologyLabel showing the insurance type of patient */
    public static final String PATIENT_INSURANCE_TYPE_LABEL = "patient_insurance_type_label";
    /** JList showing the problems of patient */
    public static final String PATIENT_PROBLEM_LIST = "patient_problem_list";
    /** TopologyScrollPane for problemList */
    public static final String PATIENT_PROBLEM_LIST_SCROLLPANE = "patient_problem_list_scrollpane";    
    /** Listmodel for problemList */
    public static final String PATIENT_PROBLEM_LISTMODEL = "problem_list_model";
    /** Listmodel for caveList */
    public static final String CAVE_LIST_MODEL = "cave_list_model";
    /** Listmodel for medicamentsList */
    public static final String MEDICAMENTS_LIST_MODEL = "medicaments_list_model";
    /** JList showing the important information about the patient */
    public static final String CAVE_LIST = "cave_list";
    /** JList showing all medicaments */
    public static final String MEDICAMENTS_LIST = "medicaments_list";
    /**ScrollPane for the caveList*/
    public static final String CAVE_LIST_SCROLLPANE = "cave_list_scrollpane";
    /**ScrollPane for the medicamentsList*/
    public static final String MEDICAMENTS_LIST_SCROLLPANE = "medicaments_list_scrollpane";
    /** The layout manager */
    public static final String LAYOUT_MANAGER = "layout_manager";
    /** Action command */
    public static final String CHOOSE_PROBLEM_ID = new String("choose_problem");
    /** Label containing basic information concerning a patient */
    public static final String BASIC_INFORMATION_LABEL = "basic_information_label";
    /** Label containing general information concerning a patient like blood type */
    public static final String GENERAL_INFORMATION_LABEL = "general_information_label";

    /**Default constructor*/
    public PatientInfoPanel() {
        super();
    }

    /** Initialization of components. */
    public void initialize() {
        gridbag = new GridBagLayout();
        setLayout(gridbag);
        GridBagConstraints c = new GridBagConstraints();
        c.fill = GridBagConstraints.VERTICAL;
        setBorder(BorderFactory.createTitledBorder("Patient Data"));
        setToolTipText("General patient information");
        setVisible(true);

        setGeneralInformationLabel(createGeneralInformationLabel());
        setBasicInformationLabel(createBasicInformationLabel());
        setPatientAdressLabel(createPatientAdressLabel());
        setPatientInsuranceLabel(createPatientInsuranceLabel());
        setProblemListModel(createProblemListModel());
        setProblemList(createProblemList());
        setProblemListScrollPane(createProblemListScrollPane());
        setCaveListModel(createCaveListModel());
        setCaveList(createCaveList());
        setCaveListScrollPane(createCaveListScrollPane());
        setMedicamentsListModel(createMedicamentsListModel());
        setMedicamentsList(createMedicamentsList());
        setMedicamentsListScrollPane(createMedicamentsListScollPane());

//        anamneseTextArea = new TopologyTextArea(new Dimension(200, 100));
        //TopologyScrollPane anamneseScrollPane = new TopologyScrollPane(anamneseTextArea, "Anamnese");

        c.anchor = GridBagConstraints.WEST;
        c.gridx = 0;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(5, 5, 5, 5);
        gridbag.setConstraints(basicInformationLabel, c);
        add(basicInformationLabel);

        c.gridx = 0;
        c.gridy = 1;
        gridbag.setConstraints(patientAdressLabel, c);
        add(patientAdressLabel);
        c.gridx = 0;
        c.gridy = 2;
        gridbag.setConstraints(generalInformationLabel, c);
        add(generalInformationLabel);

        c.anchor = GridBagConstraints.WEST;
        c.gridx = 0;
        c.gridy = 3;
        gridbag.setConstraints(patientInsuranceLabel, c);
        add(patientInsuranceLabel);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 1;
        c.gridy = 0;
        gridbag.setConstraints(caveListScrollPane, c);
        add(caveListScrollPane);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 2;
        c.gridy = 0;
        gridbag.setConstraints(problemListScrollPane, c);
        add(problemListScrollPane);

        c.anchor = GridBagConstraints.WEST;
        c.gridwidth = 1;
        c.gridheight = 4;
        c.gridx = 3;
        c.gridy = 0;
        gridbag.setConstraints(medicamentsListScrollPane, c);
        add(medicamentsListScrollPane);


    }

    /**Gets the preferred size of this panel
     *@return the preferred size
     *
     */
    public Dimension getPreferredSize() {
        return new Dimension(1000, 180);
    }

    /** Changes the current patient. Fills all needed labels and the problemList
     * @param patient the healthRecord to be displayed
     * */
    public void displayPatient(HealthRecord patient) {
        System.out.println("patient:"+patient);
        System.out.println(patient.getFirstName().getValue());
        System.out.println(patient.getName().getValue());
        System.out.println(patient.getBirthday().getValue());
        getBasicInformationLabel().setText(patient.getFirstName().getValue() + ", " + patient.getName().getValue() +
                ", " + patient.getBirthday().getValue());
        getPatientAdressLabel().setText(patient.getStreet().getValue() + ", " + patient.getZipCode().getValue() + ", " + patient.getTown().getValue());
        getPatientInsuranceLabel().setText(patient.getInsurance().getValue());
        getGeneralInformationLabel().setText(patient.getHeight().getValue()+", "+patient.getWeight().getValue()+", "+patient.getBloodType().getValue());
        getCaveListModel().clear();
        Object[] caves = patient.getCaves();
        for (int cavesIndex = 0; cavesIndex < caves.length; cavesIndex++) {
            getCaveListModel().addElement(((Cave) caves[cavesIndex]).getName().getValue());
        }
        problemListModel.removeAllElements();
        Object[] problems = patient.getProblems();
        for (int problemsIndex = 0; problemsIndex < problems.length; problemsIndex++) {
            getProblemListModel().addElement(((Problem) problems[problemsIndex]).getName().getValue());
        }
    }


    /** Adds a problem to the problem list.
     * @param problemName the name of the problem
     * */
    public void addProblemToProblemList(String problemName) {
        this.problemListModel.addElement(problemName);
    }

    /** Enables/Disables problemList and anamnese text area.
     * @param enabled indicates whether probelm list and anamnese textArea should be enabled or not
     * */
    public void setEnabled(boolean enabled) {
        problemList.setEnabled(enabled);
        anamneseTextArea.setEnabled(enabled);
    }

    /** Clears the anamnese text area. */
    public void clearAllTextAreas() {
        anamneseTextArea.setText("");
    }

    /**
     *Creates the patientAdressLabel
     * @return patientAdressLabel
     */
    public TopologyLabel createPatientAdressLabel(){
        TopologyLabel patientAdressLabel = new TopologyLabel(true);
        patientAdressLabel.setPreferredSize(new Dimension(280, 25));
        return patientAdressLabel;
    }

    public TopologyLabel createPatientInsuranceLabel(){
        TopologyLabel patientInsuranceLabel = new TopologyLabel(true);
        patientInsuranceLabel.setPreferredSize(new Dimension(280, 25));
        return patientInsuranceLabel;
    }

    /**
     *Creates the problemList
     * @return problemList
     */
    public JList createProblemList(){
        JList problemList = new JList(problemListModel);
        problemList.setPreferredSize(new Dimension(180, 90));
        problemList.setBackground(RecordModel.BackgroundColor);
        problemList.addListSelectionListener((Record)getController());
        return problemList;
    }

    /**
     *Creates the problemListScrollPane
     * @return problemListScrollPane
     */
    public TopologyScrollPane createProblemListScrollPane(){
        TopologyScrollPane problemListScrollPane = new TopologyScrollPane(getProblemList(), "Problems");
        problemListScrollPane.setPreferredSize(new Dimension(200, 80));
        problemListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        problemListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return problemListScrollPane;
    }

    /**
     *Creates the problemListModel
     * @return problemListModel
     */
    public DefaultListModel createProblemListModel(){
        return new DefaultListModel();
    }

    /**
     *Creates the caveListModel
     * @return caveListModel
     */
    public DefaultListModel createCaveListModel(){
        DefaultListModel caveListModel = new DefaultListModel();
        return caveListModel;
    }

    /**
     *Creates the medicamentsListModel
     * @return medicamnetsListModel
     */
    public DefaultListModel createMedicamentsListModel(){
         return new DefaultListModel();
    }

    /**
     *Creates the caveList
     * @return caveList
     */
    public JList createCaveList(){
        JList caveList = new JList(getCaveListModel());
        caveList.setPreferredSize(new Dimension(180, 90));
        caveList.setBackground(RecordModel.BackgroundColor);
        caveList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        caveList.addListSelectionListener((Record) getController());
        return caveList;
    }

    /**
     *Creates the medicamentsList
     * @return medicamentsList
     */
    public JList createMedicamentsList(){
        JList medicamentsList = new JList(medicamentsListModel);
        medicamentsList.setPreferredSize(new Dimension(180, 90));
        medicamentsList.setBackground(RecordModel.BackgroundColor);
        return medicamentsList;
    }


    /**
     *Creates the caveListScrollPane
     * @return caveListScrollPane
     */
    public TopologyScrollPane createCaveListScrollPane(){
        TopologyScrollPane caveListScrollPane = new TopologyScrollPane(getCaveList(), "CAVE!");
        caveListScrollPane.setPreferredSize(new Dimension(200, 80));
        caveListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        caveListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return caveListScrollPane;
    }

    /**
     *Creates the medicamentsListScrollPane
     * @return medicamentsListScrollPane
     */
    public TopologyScrollPane createMedicamentsListScollPane(){
        TopologyScrollPane medicamentsListScrollPane = new TopologyScrollPane(getMedicamentsList(), "Medication");
        medicamentsListScrollPane.setPreferredSize(new Dimension(200, 80));
        medicamentsListScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        medicamentsListScrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return medicamentsListScrollPane;
    }

    /**Creates the basicInformationLabel
     * @return the basicInformationLabel
     *
     */
    public JLabel createBasicInformationLabel(){
        JLabel basicInformationLabel = new TopologyLabel(true);
        basicInformationLabel.setPreferredSize(new Dimension(280, 25));
        return basicInformationLabel;
    }

    /**Creates the generalInformationLabel
     * @return the generalInformationLabel
     *
     */
    public JLabel createGeneralInformationLabel(){
        JLabel generalInformationLabel = new TopologyLabel(true);
        generalInformationLabel.setPreferredSize(new Dimension(280, 25));
        return generalInformationLabel;
    }
}
