/*
 * $RCSfile: OverlayTreeModel.java,v $
 *
 * Copyright (c) 1999-2003. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.topology;

import resmedicinae.record.RecordModel;

import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import javax.swing.event.TreeModelListener;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Hashtable;

/**A treeModel is the model "behind" a tree. It contains the data of the tree. In Java any objects can be treeNodes.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 * */
public class OverlayTreeModel implements TreeModel{

    /**The root of the tree. In this model the root is an overlay.*/
    public static final String ROOT_OF_TREE = "root";
    /**The model of the MVC. Gives access to the overlayHashtable.*/
    public static final String RECORD_MODEL = "record_model";
    /**List of eventListeners*/
    public static final String EVENT_LISTENER_LIST = "event_listener_list";

    /**Default constructor*/
    public OverlayTreeModel(){

    }

    /**Constructor
     * @param model the model of the MVC
     */
    public OverlayTreeModel(RecordModel model){
        setModel(model);
        setListenerList(new EventListenerList());
        Overlay overlay = (Overlay)model.getOverlayHashtable().get("skeleton_ventral");
        setRoot(overlay);
    }


    /**Constructor
     * @param root the root of the tree
     */
    public OverlayTreeModel(Overlay root){
        setRoot(root);
    }


    /**Gets the child with a particular index for a node
     * @param parent the parent node
     * @param index the index of the child
     * @return the child object
     */
    public Object getChild(Object parent, int index){
        return(getChildren(parent).elementAt(index));
    }

    /**Gets all children of a node
     * @param parent the parent node
     * @return vector containing the child nodes
     */
    public Vector getChildren(Object parent){
        //Diseases have no children
        if(parent.getClass()==Disease.class){
            return null;
        }

        Overlay overlay = (Overlay)parent;
        if((Boolean)overlay.get(Overlay.ALTERNATIVE)){
            //if the overlay is alternative the only children are its diseases
            return overlay.getDiseases();
        }
        //node is an ordinary overlay, no alternative overlay:
        Enumeration overlayRegions = overlay.getOverlayRegionHashtable().elements();
        Vector alternativeOverlays = overlay.getAlternativeOverlays();
        Vector diseases  =  ((Overlay)parent).getDiseases();
        Vector children = new Vector();

        //in that case children are diseases...
        for(int i=0; i<diseases.size();i++){
            children.add(diseases.elementAt(i));
        }
        //...alternative overlays...
        if(alternativeOverlays!=null){
            for(int i=0; i<alternativeOverlays.size();i++){
                //getting alternative overlays by its name
                if(model==null)System.out.println("MODEL IS NULL");
                children.add(model.getOverlayHashtable().get(alternativeOverlays.elementAt(i)));
            }
        }
        //...and the target overlays of the regions
        while(overlayRegions.hasMoreElements()){
            OverlayRegion overlayRegion = (OverlayRegion)overlayRegions.nextElement();
            String childOverlayName = overlayRegion.get(Overlay.TARGET);
            if(getModel().getOverlayHashtable().containsKey(childOverlayName)){
                Overlay childOverlay = (Overlay)getModel().getOverlayHashtable().get(childOverlayName);
                if(!children.contains(childOverlay)){
                    children.add(childOverlay);
                }
            }
        }

        return children;
    }

    /**Gets the number of children of a node
     * @param parent the node
     * @return the number of children
     */
    public int getChildCount(Object parent) {
        if(getChildren(parent)==null){
            return 0;
        }else
        return getChildren(parent).size();
    }

    /**Gets the index of a particular child
     * @param parent the parentNode
     * @param child the childNode
     * @return the index of the childNode
     */
    public int getIndexOfChild(Object parent, Object child) {
        for (int i = 0; i < getChildCount(parent); i++) {
            if (getChild(parent, i).equals(child)) {
                return i;
            }
        }
        return -1;
    }

    /**Tests whether a node is a leaf
     * @return true if the node is a leaf, false if not
     */
    public boolean isLeaf(Object node){
        if(getChildCount(node)==0){
            return true;
        }else{
            return false;
        }
    }

    /**Removes a treeModelListener from the listenerList
     * @param l the treeModelListener
     */
    public void removeTreeModelListener(TreeModelListener l) {
        listenerList.remove(TreeModelListener.class, l);
    }

    /**Invoked when the a path in the tree changed. Not implemented.
     * @param path the treePath
     * @param newValue the new value in the path
     */
    public void valueForPathChanged(TreePath path, Object newValue) {
    }

    /**Adds a treeModelListener to the listenerList
     * @param l the treeModelListener
     */
    public void addTreeModelListener(TreeModelListener l) {
        listenerList.add(TreeModelListener.class, l);
    }

    /**
     *Gets the event listener list
     * @return the event listener list
     */
    public EventListenerList getListenerList() {
        return listenerList;
    }

    /**
     *Sets the event listener list
     * @param the event listener list
     */
    public void setListenerList(EventListenerList listenerList) {
        this.listenerList = listenerList;
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesChanged(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesChanged(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesInserted(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesInserted(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeNodesRemoved(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeNodesRemoved(e);
            }
        }
    }

    /**
     * Notify all listeners that have registered interest for notification on this event type.  The event instance
     * is lazily created using the parameters passed into the fire method.
     * @see EventListenerList
     */
    protected void fireTreeStructureChanged(Object source, Object[] path, int[] childIndices, Object[] children) {
        Object[] listeners = listenerList.getListenerList();
        TreeModelEvent e = null;
        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == TreeModelListener.class) {
                if (e == null)
                    e = new TreeModelEvent(source, path, childIndices, children);
                ((TreeModelListener) listeners[i + 1]).treeStructureChanged(e);
            }
        }
    }
}
