/*
 * $RCSfile: ImagePanel.java,v $
 *
 * Copyright (c) 1999-2003. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.topology;

import resmedicinae.record.*;
import org.resmedicinae.domain.healthcare.heading.problem.*;

import javax.swing.*;
import java.awt.*;
import java.util.*;

/**
 * This panel contains the current image, respectively shows it. The image has sensible regions where the user can
 * click on and move over. On click  the corresponding document will change.
 * On mouse move the specific overlayRegion will be highlighted.
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class ImagePanel extends RecordPanel {
    /** The image shown in the panel. */
    public static final String IMAGE = new String("image");
    /** The toolkit needed. Toolkits are used for loading images, such as JPG or GIF. */
    public static final String TOOL_KIT = new String("tool_kit");
    /** The preferred size of the panel */
    public static final String PREFERRED_SIZE = new String("preferred_size");
    /** The maximum size of the panel */
    public static final String MAX_SIZE = new String("max_size");
    /** Vector of x-Coordinates of a overlayRegion dynamically filled when clicking on the image. */
    public static final String X_COORDINATES = new String("x_coordinates");
    /** Vector of y-Coordinates of a overlayRegion dynamically filled when clicking on the image. */
    public static final String Y_COORDINATES = new String("y_coordinates");
    /** A vector containing all overlayRegions (refernces). Used for fast access on the regions.*/
    public static final String CURRENT_OVERLAY_REGIONS_VECTOR = new String("current_overlay_regions");
    /** Vector of x-Coordinates of the marked region */
    public static final String MARKER_X_COORDINATES_VECTOR = new String("marker_x_coordinates_vector");
    /** Vector of y-Coordinates of the marked region */
    public static final String MARKER_Y_COORDINATES_VECTOR = new String("marker_y_coordinates_vector");
    /** Reference to the current objective */
    public static final String CURRENT_OBJECTIVE = new String("current_objective");
    /**ToolTipManager for toolTips of regions. Not in use yet. Tooltips should appear in further versions as hints
     * over overlay regions.
     * */
    public static final String TOOL_TIP_MANAGER = new String("tool_tip_manager");

    /**Default constructor*/
    public ImagePanel() {
        set(ImagePanel.TOOL_KIT, Toolkit.getDefaultToolkit());
        setPreferredSize(new Dimension(390, 385));
        setMaximumSize(new Dimension(390, 285));
        set(ImagePanel.X_COORDINATES, new Vector());
        set(ImagePanel.Y_COORDINATES, new Vector());
        set(ImagePanel.MARKER_X_COORDINATES_VECTOR, new Vector());
        set(ImagePanel.MARKER_Y_COORDINATES_VECTOR, new Vector());
    }

    /**
     * Returns the image.
     * @return the image
     */
    public Image getImage() {
        return this.image;
    }

    /**
     * Sets the image.
     * @param image the image
     */
    public void setImage(Image image) {
        this.image = image;
    }

    /** Initializes this object. */
    public void initialize() {
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        addMouseListener((Record) getController());
        addMouseMotionListener((Record) getController());
        addKeyListener((Record) getController());
        addFocusListener((Record) getController());
        setToolTipManager(createToolTipManager());
    }

    /** Finalizes this object. */
    public void finalizz() {
        //??        super.finalizz();
    }

    /**
     * Paints the topology image.
     * @param g the graphics context
     * @exception NullPointerException if the graphics is null
     * @exception NullPointerException if the size is null
     */
    public void paintComponent(Graphics g) throws NullPointerException {
        super.paintComponent(g);
        g.setFont(new Font("LoadingFont",Font.BOLD,30));
        g.drawString("Loading Picture...", 50,190);
        RecordModel model = (RecordModel) getModel();
        if (g != null) {
            Dimension s = getSize();
            if ((s != null) && (getImage() != null)) {
                g.drawImage(getImage(), 0, 0, s.width, s.height, this);
                /*Draw also markers of the current object if there are any*/
                if (get(ImagePanel.CURRENT_OBJECTIVE) != null) {
                    if (((Objective)get(ImagePanel.CURRENT_OBJECTIVE)).hasMarkers()) {
                        drawMarkersForObject(getCurrentObjective());
                    }
                }
            }
        } else {
            throw new NullPointerException("Could not paint component. The graphics is null.");
        }
    }

    /**
     *Gets preferred size of JPanel
     *  @return the preferred size of the panel
     */
    public Dimension getPreferredSize() {
        return preferredSize;
    }

    /**
     *Gets maximum size of JPanel
     * @return the maximum size of the panel
     */
    public Dimension getMaximumSize() {
        return maxSize;
    }

    /**
     *Paints an overlayRegion with graphics context g. The color is red with a low alpha value, or solid or red.
     * This is some kind of highlighting the area used by registering mouse moves.
     * @param g the graphics context
     * @param overlayRegion the overlayRegion to be painted
     */
    public void paintOverlayRegion(Graphics g, OverlayRegion overlayRegion) {
        if (RecordModel.FILL_POLYGON_ALPHA) {
            g.setColor(new Color(.9f, 0f, 0f, 0.3f));
            g.fillPolygon(overlayRegion);
        } else if (RecordModel.FILL_POLYGON) {
            g.setColor(new Color(.9f, 0f, 0f));
            g.fillPolygon(overlayRegion);
        } else if (RecordModel.DRAW_POLYGON) {
            g.setColor(new Color(.9f, 0f, 0f));
            g.drawPolygon(overlayRegion);
        }
    }

    /** Paints the current overlayRegion represented by the vector xCoordinates and yCoordinates on the current overlay.*/
    public void paintCurrentRegionOnOverlay() {
        int[] x = new int[xCoordinates.size()];
        int[] y = new int[yCoordinates.size()];
        //vector to array
        for (int i = 0; i < xCoordinates.size(); i++) {
            x[i] = ((Integer) getXCoordinates().elementAt(i)).intValue();
            y[i] = ((Integer) getYCoordinates().elementAt(i)).intValue();
        }
        getGraphics().drawPolygon(x, y, xCoordinates.size());
    }

    /**
     *Clears the vectors xCoordinates and yCoordinates and repaints the panel. This only
     * removes a overlayRegion after it is drawn visually from the panel. This is no remove of an
     * overlayRegion from an document!!! Do not mix up that!!
     */
    public void deleteCurrentRegionFromOverlay() {
        xCoordinates.clear();
        yCoordinates.clear();
        update(getGraphics());
    }

    /**
     *Deletes the last drawn point from the panel. This means removing the last elements from vector xCoordinates and yCoordinates.
     */
    public void deleteLastPaintedPoint() {
        xCoordinates.remove(xCoordinates.lastElement());
        yCoordinates.remove(yCoordinates.lastElement());
        update(getGraphics());
        drawRegionPointsOnOverlay();
    }

    /** This method paints an overlayRegions. No solid drawing, only line borders. This is used for showing all
     * overlay regions.
     * @param overlayRegion the overlayRegion to be drawn
     *  */
    public void drawOverlayRegion(OverlayRegion overlayRegion) {
        getGraphics().drawPolygon(overlayRegion);
    }

    /** This method paints all points represented by values of the vectors xCoordinates and yCoordinates.
     * Not complete overlayRegions, only its points!!!
     * */
    public void drawRegionPointsOnOverlay() {
        for (int i = 0; i < xCoordinates.size(); i++) {
            int x = ((Integer) xCoordinates.elementAt(i)).intValue();
            int y = ((Integer) yCoordinates.elementAt(i)).intValue();
            getGraphics().drawRect(x, y, 3, 3);
        }
        if (xCoordinates.size() > 1) {
            for (int i = 0; i < xCoordinates.size() - 1; i++) {
                int x = ((Integer) xCoordinates.elementAt(i)).intValue();
                int y = ((Integer) yCoordinates.elementAt(i)).intValue();
                int next_x = ((Integer) xCoordinates.elementAt(i + 1)).intValue();
                int next_y = ((Integer) yCoordinates.elementAt(i + 1)).intValue();
                System.out.println("Redrawing line");
                getGraphics().drawLine(x, y, next_x, next_y);
            }
        }
    }

    /** Removes all markers from the overlay. Clearing vectors xCoordinates and yCoordinates.*/
    public void removeAllMarkersFromOverlay() {
        markerXCoordinates.clear();
        markerYCoordinates.clear();
    }

    /** Paints a marker at (x,y).
     * @param x the x position of the marker
     * @param y the y position of the marker
     * */
    public void drawMarker(int x, int y) {
        getGraphics().setColor(Color.RED);
        getGraphics().fillOval(x, y, 5, 5);
        markerXCoordinates.add(new Integer(x));
        markerYCoordinates.add(new Integer(y));
    }

    /**
     *Paints all markers for an object.
     * @param object the object for that the markers have to be painted
     */
    public void drawMarkersForObject(Objective object) {
        //removeAllMarkersFromOverlay();
        for (int i = 0; i < object.getMarkerCount(); i++) {
            getGraphics().fillOval(object.getXMarker(i).intValue(), object.getYMarker(i).intValue(), 5, 5);
        }
    }

    /** Updates the image. Forces the panel to redraw.*/
    public void updateImage() {
        update(getGraphics());
    }

}

