/*
 * $RCSfile: EditOverlayRegionDialog.java,v $
 *
 * Copyright (c) 1999-2002. Jens Bohl. All rights reserved.
 *
 * This software is published under the GPL GNU General Public License.
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.resmedicinae.org
 * - Information in Medicine -
 */

package resmedicinae.record.topology;

import resmedicinae.record.*;
import org.resmedicinae.resmedlib.component.control.*;
import org.resmedicinae.resmedlib.component.view.swing.*;

import javax.swing.*;
import java.awt.*;

import cybop.basic.String;

/**
 * View Component. This dialog is for editing existing overlay regions. The name and target
 * can be changed. The overlay region can be removed as well. This dialog appears after clicking on an overlay region
 * in "edit overlay region mode".
 * @author Jens Bohl <info@jens-bohl-professional.de>
 */
public class EditOverlayRegionDialog extends SwingView {
    
    /** Layout Manager of this dialog */
    public static final String LAYOUT_MANAGER = new String("layout_manager");
    
    /** Name label of polygon */
    public static final String OVERLAY_REGION_LABEL = new String("polygon_label");
    
    /** Name textField of the overlay region */
    public static final String OVERLAY_REGION_NAME_TEXT_FIELD = new String("overlay_region_name_textfield");
    
    /** Target label of the polygon */
    public static final String TARGET_OVERLAY_LABEL = new String("target_overlay_label");
    
    /** Ok Button */
    public static final String OK_BUTTON = new String("ok_button");
    
    /** Delete Button */
    public static final String DELETE_OVERLAY_REGION_BUTTON = new String("delete_button");
    
    /** ComboBox containing possible targets. */
    public static final String POSSIBLE_TARGETS_COMBOBOX = new String("possible_targets_combobox");
    
    /** String containing the old name of the polygon */
    public static final String OLD_NAME_OF_POLYGON = new String("old_name_of_polygon");
    
    /**Right NeighbourLabel*/
    public static final String RIGHT_NEIGHBOUR_LABEL = new String("right_neighbour_label");
    
    /** Right neighbour combobox */
    public static final String RIGHT_NEIGHBOUR_COMBOBOX = new String("right_neighbour_comboBox");
    
    /** Right neighbour checkbox */
    public static final String RIGHT_NEIGHBOUR_CHECKBOX = new String("right_neighbour_checkbox");
    
    /**Right NeighbourLabel*/
    public static final String LEFT_NEIGHBOUR_LABEL = new String("left_neighbour_label");
    
    /** Right neighbour combobox */
    public static final String LEFT_NEIGHBOUR_COMBOBOX = new String("left_neighbour_comboBox");
    
    /** Right neighbour checkbox */
    public static final String LEFT_NEIGHBOUR_CHECKBOX = new String("left_neighbour_checkbox");
    
    /** Top neighbour label */
    public static final String TOP_NEIGHBOUR_LABEL = new String("top_neighbour_label");
    
    /** Right neighbour combobox */
    public static final String TOP_NEIGHBOUR_COMBOBOX = new String("top_neighbour_comboBox");
    
    /** Right neighbour checkbox */
    public static final String TOP_NEIGHBOUR_CHECKBOX = new String("top_neighbour_checkbox");
    
    /** Bottom neighbour label */
    public static final String BOTTOM_NEIGHBOUR_LABEL = new String("bottom_neighbour_label");
    
    /** Right neighbour combobox */
    public static final String BOTTOM_NEIGHBOUR_COMBOBOX = new String("bottom_neighbour_comboBox");
    
    /** Right neighbour checkbox */
    public static final String BOTTOM_NEIGHBOUR_CHECKBOX = new String("bottom_neighbour_checkbox");
    
    /** The neighbour panel. This panel conatins all the other neighbour panels.*/
    public static final String NEIGHBOUR_PANEL = new String("neighbour_panel");
    
    /** The top neighbour panel */
    public static final String TOP_NEIGHBOUR_PANEL = new String("top_neighbour_panel");
    
    /** The bottom neighbour panel */
    public static final String BOTTOM_NEIGHBOUR_PANEL = new String("bottom_neighbour_panel");
    
    /** The left neighbour panel */
    public static final String LEFT_NEIGHBOUR_PANEL = new String("left_neighbour_panel");
    
    /** The right neighbour panel */
    public static final String RIGHT_NEIGHBOUR_PANEL = new String("right_neighbour_panel");
    
    /** Target image combobox */
    public static final String TARGET_IMAGE_COMBOBOX = new String("target_image_combo_box");

    /** Gridbag Constraints*/
    public static final String GRID_BAG_CONSTRAINTS = new String("grid_bag_constraints");

    /** The model of MVC */
    public static final String RECORD_MODEL = new String("record_model");
    
    /** Default Constructor */
    public EditOverlayRegionDialog() {
    }

    /** Sets the overlay region name in the textField as well as the oldname oldName. The old name of the overlay
     * region can then be modified.
     * @param overlayRegioName the old name of the overlayRegion
     * */
    public void setOverlayRegionName(String overlayRegioName) {
        ((JTextField)get(EditOverlayRegionDialog.OVERLAY_REGION_NAME_TEXT_FIELD)).setText(overlayRegioName);
        set(EditOverlayRegionDialog.OLD_NAME_OF_POLYGON,overlayRegioName);
    }

    /** Sets the overlayRegion target in the comboBox
     * @param overlayRegionTarget the target of the overlayRegion
     * */
    public void setOverlayRegionTarget(String overlayRegionTarget) {
        ((JComboBox)get(EditOverlayRegionDialog.TARGET_IMAGE_COMBOBOX)).setSelectedItem(overlayRegionTarget);
    }
    

    /** Initialization of components */
    public void initialize() {
        setTitle(new String("Edit overlay region..."));
        
        set(EditOverlayRegionDialog.LAYOUT_MANAGER, new GridBagLayout());
        GridBagLayout gridbag = (GridBagLayout)get(EditOverlayRegionDialog.LAYOUT_MANAGER); 
        setLayout(gridbag);

        setDisplayMode(DisplayFactory.MODELESS_DIALOG_DISPLAY);
        setDialogBounds(createDialogBounds());

        GridBagConstraints c = (GridBagConstraints)get(EditOverlayRegionDialog.GRID_BAG_CONSTRAINTS);
        
        set(EditOverlayRegionDialog.OVERLAY_REGION_LABEL, createNameLabel());
        set(EditOverlayRegionDialog.OVERLAY_REGION_NAME_TEXT_FIELD, createNameTextField());
        set(EditOverlayRegionDialog.TARGET_OVERLAY_LABEL, createTargetImageLabel());
        set(EditOverlayRegionDialog.TARGET_IMAGE_COMBOBOX, createTargetImageComboBox());
        set(EditOverlayRegionDialog.NEIGHBOUR_PANEL, createNeighbourPanel());
        set(EditOverlayRegionDialog.OK_BUTTON, createOkButton());
        set(EditOverlayRegionDialog.DELETE_OVERLAY_REGION_BUTTON, createDeleteOverlayRegionButton());                

        c.fill = GridBagConstraints.HORIZONTAL;
        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridx = 0;
        c.gridy = 0;
        gridbag.setConstraints((JLabel)get(EditOverlayRegionDialog.OVERLAY_REGION_LABEL),c);
        add((JLabel)get(TARGET_OVERLAY_LABEL));

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 0;
        gridbag.setConstraints((JTextField)get(EditOverlayRegionDialog.OVERLAY_REGION_NAME_TEXT_FIELD), c);
        add((JTextField)get(EditOverlayRegionDialog.OVERLAY_REGION_NAME_TEXT_FIELD));

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 0;
        c.gridy = 1;
        gridbag.setConstraints((JLabel)get(EditOverlayRegionDialog.TARGET_OVERLAY_LABEL), c);
        add((JLabel)get(EditOverlayRegionDialog.TARGET_OVERLAY_LABEL));

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 2, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 1;
        gridbag.setConstraints((JComboBox)get(EditOverlayRegionDialog.TARGET_IMAGE_COMBOBOX), c);
        add((JComboBox)get(EditOverlayRegionDialog.TARGET_IMAGE_COMBOBOX));

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridwidth = 2;
        c.gridx = 0;
        c.gridy = 2;
        gridbag.setConstraints((JPanel)get(EditOverlayRegionDialog.NEIGHBOUR_PANEL), c);
        add((JPanel)get(EditOverlayRegionDialog.NEIGHBOUR_PANEL));

        c.anchor = GridBagConstraints.EAST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridwidth = 1;
        c.gridx = 1;
        c.gridy = 3;
        gridbag.setConstraints((JButton)get(EditOverlayRegionDialog.DELETE_OVERLAY_REGION_BUTTON), c);
        add((JButton)get(EditOverlayRegionDialog.DELETE_OVERLAY_REGION_BUTTON));

        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets(10, 10, 10, 10);
        c.gridx = 0;
        c.gridy = 3;
        gridbag.setConstraints((JButton)get(EditOverlayRegionDialog.OK_BUTTON), c);
        add((JButton)get(EditOverlayRegionDialog.OK_BUTTON));

    }

     /**
     * Creates a rectangle storing the dialog bounds.
     * @return the rectangle storing the dialog bounds
     */
    public Rectangle createDialogBounds() {
        // Determine width.
        int screenWidth = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getWidth());
        int w = 600;
        if (w > screenWidth) {
            w = screenWidth;
        }
        // Use centered x position by default.
        int x = (screenWidth - w) / 2;
        // Determine height.
        int screenHeight = (int) Math.floor(Toolkit.getDefaultToolkit().getScreenSize().getHeight());
        int h = 500;
        if (h > screenHeight) {
            h = screenHeight;
        }
        // Use centered y position by default.
        int y = (screenHeight - h) / 2;
        return new Rectangle(x, y, w, h);
    }

    /**
     * Creates a display mode.
     * @return the display mode
     */
    public int createDisplayMode() {
        return DisplayFactory.MODELESS_DIALOG_DISPLAY;
    }

    /**
     * Creates a control issued when the window containing this view is closed.
     * @return the control issued when the window containing this view is closed
     * @exception NullPointerException if the control is null
     */
    public Control createCloseControl() throws NullPointerException {
        Control c = new Control();
        if (c != null) {
            c.setId(Record.DESTROY_LOAD_PATIENT_DIALOG_CONTROL_ID);
            c.setSender((Controller) getController());
        } else {
            throw new NullPointerException("Could not create close control. The control is null.");
        }
        return c;
    }

    /** Finalizes this component. */
    public void finalizz() throws Exception {
        try {
            // Set parameters.
            setBorder(null);
        } finally {
            super.finalizz();
        }
    }

    /**Creates the overlay region name label
     * @return the name label
     */
    public TopologyLabel createNameLabel(){
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Name");
        l.setMinimumSize(new Dimension(100, 50));
        l.setPreferredSize(new Dimension(100, 50));
        return l;
    }

    /**Creates the overlay region name taext field
     * @return the name text field
     */
    public JTextField createNameTextField(){
        JTextField nameTextField = new JTextField(15);
        nameTextField.setEditable(true);
        nameTextField.setEnabled(true);
        nameTextField.setMinimumSize(new Dimension(250, 20));
        nameTextField.setPreferredSize(new Dimension(250, 20));
        return nameTextField;
    }

    /**Creates the overlay region target label
     * @return the target label
     */
    public TopologyLabel createTargetImageLabel(){
        TopologyLabel targetImageLabel = new TopologyLabel(false);
        targetImageLabel.setText("Target overlay");
        targetImageLabel.setMinimumSize(new Dimension(100, 50));
        targetImageLabel.setPreferredSize(new Dimension(80, 50));
        return targetImageLabel;
    }

    /**Creates the ok button
     * @return the ok button
     */
    public JButton createOkButton(){
        JButton okButton = new JButton("Ok");
        okButton.setActionCommand(Record.EDIT_OVERLAY_REGION_DIALOG_OK_ID);
        okButton.addActionListener((Record) getController());
        okButton.setToolTipText("Confirm");
        return okButton;
    }

    /**Creates the delete overlay region button
     * @return the delet overlay region button
     */
    public JButton createDeleteOverlayRegionButton(){
        JButton deleteOverlayRegionButton = new JButton("Remove overlay region");
        deleteOverlayRegionButton.setActionCommand(Record.EDIT_OVERLAY_REGION_DIALOG_DELETE_POLYGON_ID);
        deleteOverlayRegionButton.addActionListener((Record) getController());
        return deleteOverlayRegionButton;
    }

    /**Creates the overlay region target image combobox
     * @return the overlay region target image combobox
     */
    public JComboBox createTargetImageComboBox(){
        JComboBox targetImageComboBox = new JComboBox();
        if (((RecordModel) getModel()).getFileNames() != null) {
            targetImageComboBox = new JComboBox(((RecordModel) getModel()).getFileNames());
        }
        targetImageComboBox.setToolTipText("Choose image");
        return targetImageComboBox;
    }


    /**Creates the old name for the overlay region
     * @return the name label
     */
    public String createOldName(){
        return new String();
    }


    /**Creates the neighbour panel.
     * @return the neighbour panel
     */
    public ResPanel createNeighbourPanel() {
        ResPanel p = new ResPanel();
        if (p != null) {
            p.setBorder(BorderFactory.createTitledBorder("Neighbours"));
            GridBagLayout neighbourPanelLayout = new GridBagLayout();
            GridBagConstraints c = new GridBagConstraints();
            p.setLayout(neighbourPanelLayout);
            set(EditOverlayRegionDialog.TOP_NEIGHBOUR_PANEL, createTopNeighbourPanel());
            set(EditOverlayRegionDialog.BOTTOM_NEIGHBOUR_PANEL, createBottomNeighbourPanel());
            set(EditOverlayRegionDialog.LEFT_NEIGHBOUR_PANEL, createLeftNeighbourPanel());
            set(EditOverlayRegionDialog.RIGHT_NEIGHBOUR_PANEL, createRightNeighbourPanel());
            c.anchor = GridBagConstraints.WEST;
            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 0;
            neighbourPanelLayout.setConstraints((JPanel)get(EditOverlayRegionDialog.TOP_NEIGHBOUR_PANEL), c);
            p.add((JPanel)get(EditOverlayRegionDialog.TOP_NEIGHBOUR_PANEL));
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 1;
            neighbourPanelLayout.setConstraints((JPanel)get(EditOverlayRegionDialog.BOTTOM_NEIGHBOUR_PANEL), c);
            p.add((JPanel)get(EditOverlayRegionDialog.BOTTOM_NEIGHBOUR_PANEL));
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 2;
            neighbourPanelLayout.setConstraints((JPanel)get(EditOverlayRegionDialog.LEFT_NEIGHBOUR_PANEL), c);
            p.add((JPanel)get(EditOverlayRegionDialog.LEFT_NEIGHBOUR_PANEL));
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(2, 2, 2, 2);
            c.gridwidth = 1;
            c.gridx = 0;
            c.gridy = 3;
            neighbourPanelLayout.setConstraints((JPanel)get(EditOverlayRegionDialog.RIGHT_NEIGHBOUR_PANEL), c);
            p.add((JPanel)get(EditOverlayRegionDialog.RIGHT_NEIGHBOUR_PANEL));
        } else {
            throw new NullPointerException("Could not create NeighbourPanel. The panel is null");
        }
        return p;
    }


    /**Creates the top neighbour panel
     * @return the top neighbour panel
     */
    public ResPanel createTopNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel tnp = new ResPanel();
        tnp.setPreferredSize(new Dimension(550,50));
        tnp.setLayout(f);
        if (tnp != null) {
            tnp.setBorder(BorderFactory.createTitledBorder("Top"));
            setTopNeighbourLabel(createTopNeighbourLabel());
            setTopNeighbourComboBox(createTopNeighbourComboBox());
            setTopNeighbourCheckBox(createTopNeighbourCheckBox());
            tnp.add(getTopNeighbourCheckBox(), BorderLayout.WEST);
            tnp.add(getTopNeighbourLabel(), BorderLayout.CENTER);
            tnp.add(getTopNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create topNeighbourPanel. The panel is null");
        }
        return tnp;
    }

    /**Creates the bottom neighbour panel
     * @return the bottom neighbour panel
     */
    public ResPanel createBottomNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel bnp = new ResPanel();
        bnp.setLayout(f);
        if (bnp != null) {
            bnp.setBorder(BorderFactory.createTitledBorder("Bottom"));
            setBottomNeighbourLabel(createBottomNeighbourLabel());
            setBottomNeighbourComboBox(createBottomNeighbourComboBox());
            setBottomNeighbourCheckBox(createBottomNeighbourCheckBox());
            bnp.add(getBottomNeighbourCheckBox(), BorderLayout.WEST);
            bnp.add(getBottomNeighbourLabel(), BorderLayout.CENTER);
            bnp.add(getBottomNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create bottomNeighbourPanel. The panel is null");
        }
        return bnp;
    }

    /**Creates the left neighbour panel
     * @return the left neighbour panel
     */
    public ResPanel createLeftNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel lnp = new ResPanel();
        lnp.setLayout(f);
        if (lnp != null) {
            lnp.setBorder(BorderFactory.createTitledBorder("Left"));
            setLeftNeighbourLabel(createLeftNeighbourLabel());
            setLeftNeighbourComboBox(createLeftNeighbourComboBox());
            setLeftNeighbourCheckBox(createLeftNeighbourCheckBox());

            lnp.add(getLeftNeighbourCheckBox(), BorderLayout.WEST);
            lnp.add(getLeftNeighbourLabel(), BorderLayout.CENTER);
            lnp.add(getLeftNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return lnp;
    }

    /**Creates the right neighbour panel
     * @return the right neighbour panel
     */
    public ResPanel createRightNeighbourPanel() {
        BorderLayout f = new BorderLayout();
        ResPanel rnp = new ResPanel();
        rnp.setLayout(f);
        if (rnp != null) {
            rnp.setBorder(BorderFactory.createTitledBorder("Right"));
            setRightNeighbourLabel(createRightNeighbourLabel());
            setRightNeighbourComboBox(createRightNeighbourComboBox());
            setRightNeighbourCheckBox(createRightNeighbourCheckBox());
            rnp.add(getRightNeighbourCheckBox(), BorderLayout.WEST);
            rnp.add(getRightNeighbourLabel(), BorderLayout.CENTER);
            rnp.add(getRightNeighbourComboBox(), BorderLayout.EAST);
        } else {
            throw new NullPointerException("Could not create leftNeighbourPanel. The panel is null");
        }
        return rnp;
    }

    /**ReInitializes the component
     * This is needed to reset the dialog properties in respect of the different overlay regions that can be choosed.
     * The different comboBoxes will be reset.
     * */
    public void reInitialize() {
        RecordModel model = (RecordModel) getModel();
        int currentRegionCount = model.getCurrentOverlayRegionNames().length;
        topNeighbourComboBox.removeAllItems();
        bottomNeighbourComboBox.removeAllItems();
        leftNeighbourComboBox.removeAllItems();
        rightNeighbourComboBox.removeAllItems();
        for (int i = 0; i < currentRegionCount; i++) {
            topNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            bottomNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            leftNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
            rightNeighbourComboBox.addItem(model.getCurrentOverlayRegionNames()[i]);
        }
    }

    /**Creates the right neighbour Label
     * @return the right neighbour Label
     */
    public TopologyLabel createRightNeighbourLabel(){
        TopologyLabel rightNeighbourLabel = new TopologyLabel(false);
        rightNeighbourLabel.setText("Right neighbour");
        return rightNeighbourLabel;
    }


    /**Creates the right neighbour comboBox
     * @return the right neighbour comboBox
     */
    public JComboBox createRightNeighbourComboBox(){
        JComboBox rightNeighbourComboBox = new JComboBox();
        rightNeighbourComboBox.setEnabled(false);
        return rightNeighbourComboBox;
    }


    /**Creates the right neighbour checkBox
     * @return the right neighbour checkBox
     */
    public JCheckBox createRightNeighbourCheckBox(){
        JCheckBox rightNeighbourCheckBox = new JCheckBox();
        rightNeighbourCheckBox.addItemListener((Record) getController());
        return rightNeighbourCheckBox;
    }


    /**Creates the left neighbour label
     * @return the left neighbour label
     */
    public TopologyLabel createLeftNeighbourLabel(){
        TopologyLabel leftNeighbourLabel = new TopologyLabel(false);
        leftNeighbourLabel.setText("Left neighbour");
        return leftNeighbourLabel;
    }

    /**Creates the left neighbour comboBox
     * @return the left neighbour comboBox
     */
    public JComboBox createLeftNeighbourComboBox(){
        JComboBox leftNeighbourComboBox = new JComboBox();
        leftNeighbourComboBox.setEnabled(false);
        return leftNeighbourComboBox;
    }


    /**Creates the left neighbour checkBox
     * @return the left neighbour checkBox
     */
    public JCheckBox createLeftNeighbourCheckBox(){
        JCheckBox leftNeighbourCheckBox = new JCheckBox();
        leftNeighbourCheckBox.addItemListener((Record) getController());
        return leftNeighbourCheckBox;
    }


    /**Creates the top neighbour label
     * @return the top neighbour label
     */
    public TopologyLabel createTopNeighbourLabel(){
        TopologyLabel l = new TopologyLabel(false);
        l.setText("Top neighbour");
        return l;
    }


    /**Creates the top neighbour comboBox
     * @return the top neighbour comboBox
     */
    public JComboBox createTopNeighbourComboBox(){
        JComboBox c = new JComboBox();
        c.setEnabled(false);
        return c;
    }


    /**Creates the top neighbour checkBox
     * @return the top neighbour checkBox
     */
    public JCheckBox createTopNeighbourCheckBox(){
        JCheckBox c = new JCheckBox();
        c.addItemListener((Record) getController());
        return c;
    }


    /**Creates the bottom neighbour label
     * @return the bottom neighbour label
     */
    public TopologyLabel createBottomNeighbourLabel(){
        TopologyLabel bottomNeighbourLabel = new TopologyLabel(false);
        bottomNeighbourLabel.setText("Bottom neighbour");
        return bottomNeighbourLabel;
    }


    /**Creates the bottom neighbour comboBox
     * @return the bottom neighbour comboBox
     */
    public JComboBox createBottomNeighbourComboBox(){
        JComboBox c = bottomNeighbourComboBox = new JComboBox();
        c.setEnabled(false);
        return c;
    }

    /**Creates the bottom neighbour checkBox
     * @return the bottom neighbour checkBox
     */
    public JCheckBox createBottomNeighbourCheckBox(){
        JCheckBox bottomNeighbourCheckBox = new JCheckBox();
        bottomNeighbourCheckBox.addItemListener((Record) getController());
        return bottomNeighbourCheckBox;
    }

}
